<?php
/**
 * This is a utils file and helps in doing some common functionalities around the modules,
 *
 *
 * @author miniOrange
 * @version 1.0
 */
namespace Drupal\miniorange_oauth_client\MoHelper;

use Drupal\Component\Utility\Html;
use Drupal\Core\Url;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoDTO\MoModuleSettings;
use Drupal\miniorange_oauth_client\MoDTO\MoModuleSettings as MoModuleSettingsEntity;
use Drupal\miniorange_oauth_client\MoLibrary\MoLogger;
use GuzzleHttp\Exception\GuzzleException;
use RecursiveArrayIterator;
use RecursiveIteratorIterator;

final class MoUtilities
{


  /**
   * This function is to encrypt the data using a key by an encryption algorithm
   *
   *
   * @param string $data - The data to be encrypted
   * @param string $key - The encryption key
   * @param string $method - The encryption method
   *
   * @return string
   */
  public static function encryptData(string $data, string $key, string $method = 'AES-128-CBC'): string
  {
    $key    = openssl_digest($key, 'sha256');
    $ivSize = openssl_cipher_iv_length($method);
    $iv     = openssl_random_pseudo_bytes($ivSize);
    $strCrypt = openssl_encrypt ($data, $method, $key,OPENSSL_RAW_DATA||OPENSSL_ZERO_PADDING, $iv);
    return base64_encode($iv.$strCrypt);
  }

  /**
   * This function is to decrypt the data based on the key and the algorithm
   *
   * @param string|null $data - encrypted data for decrypting
   * @param string|null $key - The key to decrypt the data
   * @param string $method - The decryption method
   *
   * @return string
   */
  public static function decryptData(?string $data, ?string $key, string $method = 'AES-128-CBC'): string
  {
    if(empty($data)) {
      return $data;
    }
    $strIn = base64_decode($data);
    $key    = openssl_digest($key, 'sha256');
    $ivSize = openssl_cipher_iv_length($method);
    $iv     = substr($strIn,0,$ivSize);
    $data   = substr($strIn,$ivSize);
    try {
      return openssl_decrypt ($data, $method, $key, OPENSSL_RAW_DATA||OPENSSL_ZERO_PADDING, $iv);
    } catch (\Exception $exception) {
      MoLogger::error($exception->getMessage());
      throw new \Exception($exception->getMessage());
    }
  }

  public static function getCallBackUrl(?string $app_name): string
  {

    if (!empty($app_name)) {
      $callback_url = Url::fromRoute(
        'mo_oauth.authorization_response',
        ['mo_client_config' => $app_name],
        ['absolute' => TRUE]
      )->toString();
      if (MoModuleSettings::readMe()->getEnforceHttpsCallback()) {
        $callback_url = preg_replace('/http:\/\//', 'https://', $callback_url);
      }
      return $callback_url;
    }
    return '';
  }

  public static function getModifiedAppName(?string $app_name): string
  {
    return empty($app_name) ? '' : str_replace(' ', '-', strtolower($app_name));
  }

  /**
   * sanitizes an associative array.
   *
   * @param array $input
   *   The input array to sanitize.
   *
   * @return array
   *   The sanitized array.
   */
  public static function sanitizeRecursive(array $input): array {
    $sanitized = [];
    foreach ($input as $key => $value) {
      $sanitizedKey = Html::escape($key);
      if (is_array($value)) {
        $sanitizedValue = self::sanitizeRecursive($value);
      }
      else {
        $sanitizedValue = Html::escape($value);
      }
      $sanitized[$sanitizedKey] = $sanitizedValue;
    }
    return $sanitized;
  }

  /**
   * This function helps in normalizing the actual array in the effective way for using it
   *
   * This function was tested on the basis of the below input
   *
   * ---------dummy array
   *
   * $array = array(
   *  'a' => "avc",
   *  'av' => "dddd",
   *  'Semester1' => [
   *    ['Math' => 90, 'Science' => 85, 'English' => 88],
   *    ['Math' => 91, 'Science' => 86, 'English' => 88],
   *    ['Math' => 92, 'Science' => 87, 'English' => 88],
   *    ['Math' => 93, 'Science' => 88],
   *  ],
   *  'dsf' => ['aaa' => ['created' => '2024-01-01'], 'sss', 'awqdwef', 'aaa'],
   *  'meta' => [
   *    'info' => [
   *      'created' => '2024-01-01',
   *      'status' => 'active'
   *    ]
   *  ]
   * );
   *
   * ---------received output
   *
   * $output = [
   *    'a' => 'avc',
   *    'av' => 'dddd',
   *    'Semester1>Math' => [90, 91, 92, 93],
   *    'Semester1>Science' => [85, 86, 87, 88],
   *    'Semester1>English' => [88],
   *    'dsf>0>created' => '2024-01-01',
   *    'dsf>1' => 'sss',
   *    'dsf>2' => 'awqdwef',
   *    'dsf>3' => 'aaa',
   *    'meta>info>created' => '2024-01-01',
   *    'meta>info>status' => 'active'
   *  ];
   *
   * -----------
   *
   * @param array $input the input array to process
   * @param string $prefix the prefix string to append it in the key
   * @param array $output the output array to be processed as a response and returned
   * @return array
   */
  public static function normalizeRecursive(?array $input, string $prefix = '', array &$output = []): array {
    if (!$input) {
      return [];
    }
    foreach ($input as $key => $value) {
      $new_key = $prefix === '' ? $key : "$prefix>$key";

      if (is_array($value)) {
        $is_indexed = array_keys($value) === range(0, count($value) - 1);
        $first = reset($value);

        // Case 1: Array of associative arrays (e.g., Semester1)
        if ($is_indexed && is_array($first) && array_keys($first) !== range(0, count($first) - 1)) {
          foreach ($value as $row) {
            foreach ($row as $sub_key => $sub_val) {
              $compound_key = "$new_key>$sub_key";
              $output[$compound_key][] = $sub_val;
            }
          }
        }
        // Case 2: Indexed array (e.g., dsf)
        elseif ($is_indexed) {
          $output[$new_key] = array_values(array_unique($value));
        }
        // Case 3: Associative array — recurse deeper
        else {
          self::normalizeRecursive($value, $new_key, $output);
        }
      }
      // Case 4: Scalar — always store as scalar (no array wrap)
      else {
        $output[$new_key] = $value;
      }
    }

    // Deduplicate only arrays
    foreach ($output as $k => $v) {
      if (is_array($v)) {
        $output[$k] = array_values(array_unique($v));
      }
    }

    return $output;
  }

  /**
   * Base64 Url encode
   *
   * @param string $data to encode
   */
  public static function base64url_encode(string $data): string
  {
    return rtrim( strtr( base64_encode( $data ), '+/', '-_' ), '=' );
  }

  /**
   * Base64 Url decode
   *
   * @param string $data to decode
   */
  public static function base64urlDecode(string $data): false|string
  {
    return base64_decode( str_pad( strtr( $data, '-_', '+/' ), strlen( $data ) % 4, '=', STR_PAD_RIGHT ) );
  }

  /**
   * this function is to decode the state which was encoded before initiating the authorization call
   *
   * @param string|null $state
   * @return array|mixed
   */
  public static function decodeState(?string $state = null): mixed
  {
    $state   = !empty($state) ? $state : '';
    return !empty($state) ? json_decode((self::base64urlDecode($state)),true) : [];
  }

  /**
   * Build anonymous redirect response if site encounters an issue
   *
   * @param string $redirect_url redirect url to take you to if any error found
   * @param \Throwable $exception exception object
   * @param bool $test_sso param to confirm is this testing.
   *
   * @return void
   */
  public static function buildErrorResponse(string &$redirect_url, \Throwable $exception, bool $test_sso, bool $user_creation_feature_restriction_err = false): void
  {
    MoLogger::error($exception->getMessage());
    if ($test_sso) {
      MoUtilities::showErrorOnTestConfigWindow($exception->getMessage());
    } else {
      $message = t(MoLogger::COMMON_ERROR_MESSAGE);
      if ($user_creation_feature_restriction_err) {
        $message = ['#markup' => $message. '<br>'.t('Administrator: Please check  Reports -> Recent log messages.')];
      }
      \Drupal::messenger()->addError($message);
      if (empty($redirect_url)) {
        $module_settings = MoModuleSettingsEntity::readMe();
        $redirect_url = $module_settings->getReplaceLoginPage() ?
          Url::fromRoute('<front>')->toString() :
          Url::fromRoute('user.login')->toString();
      }
    }
  }

  /**
   * Show Error On test config window
   *
   * @return void
   *
   * todo need to work on it giving hard HTML is a bad idea
   */
  public static function showErrorOnTestConfigWindow($error)
  {
    echo '<div style="font-family:Calibri;padding:0 3%;">
            <div
            style="color: #a94442;background-color: #f2dede;padding: 15px;margin-bottom: 20px;text-align:center;border:1px solid #E6B3B2;font-size:18pt;"
            >'. t("Test Configuration Failed").'
            </div>
            <div style="color: #a94442;font-size:14pt; margin-bottom:20px;">
              <div>
                  <strong>Error:</strong>
              </div>
                  <p>'.$error.'</p>
            </div>
          </div>';
    exit;
  }

  Public static function buildPostRequest(array $headers, array $params, string $url) {
    try{
      $body = [
        'headers' => $headers,
        'allow_redirects' => TRUE,
        'http_errors' => FALSE,
        'decode_content' => TRUE,
        'verify' => FALSE,
        'form_params' => $params,
      ];
      //todo commented because there is no need to share it with body just can we go with form_params
//    if ($send_as_form_param) {
//      parse_str($fieldString, $form_body);
//    }
//    $body += $send_as_form_param ? ['form_params' => $form_body] : ['body' => $fieldString];
      $response = \Drupal::httpClient()
        ->post($url, $body);
      return json_decode($response->getBody()->getContents(), true);
    } catch (\Exception $exception) {
      throw new \Exception($exception->getMessage());
    }
  }

  public static function buildGetRequest(?array $headers, string $url)
  {
    try{
      $body = [
        'headers' => $headers,
        'verify' => FALSE,
      ];
      $response = \Drupal::httpClient()->get($url, $body);
      return json_decode($response->getBody()->getContents(), true);
    } catch (\Exception $exception) {
      throw new \Exception($exception->getMessage());
    }
  }

  public static function flattenArray($array, $prefix = ''): array
  {
    $result = [];
    foreach ($array as $key => $value) {
      $newKey = $prefix !== '' ? $prefix . '>' . $key : $key; //
      $result += is_array($value) ? self::flattenArray($value, $newKey) : [$newKey => $value];
    }
    return $result;
  }

  public static function normalizeToArray(null|string|array $array): array
  {
    !empty($array) || $array = [];
    !is_string($array) || $array = [$array];
    $iterator = new RecursiveIteratorIterator(new RecursiveArrayIterator($array));
    foreach ($iterator as $value) {
      $result[] = $value;
    }
    return $result ?? [];
  }

// todo old approach by recursion
//  public static function normalizeToArray(array &$array, &$resp = []): array
//  {
//    foreach ($array as $key => $value) {
//      is_array($value) ? self::normalizeToArray($value, $resp) : array_push($resp, $value);
//    }
//    return $resp;
//  }

  /**
   * @throws \Exception
   */
  public static function getUpgradeMaintenanceLink(): string
  {
    $config = \Drupal::service('miniorange_oauth_client.uno_module_license_entity')->readMe();
    $redirect = MoModuleConstant::LICENSE_SERVER_BASE_URL . '/moas/admin/customer/licenserenewals?renewalrequest=' . $config->getLicenseType();
    return MoModuleConstant::LICENSE_SERVER_BASE_URL . '/moas/login?username=' . $config->getCustomerEmail() . '&redirectUrl=' . $redirect;
  }

  public static function callService(string $url, string $field_string, array $header): ?string {
    // Optimize request options for better performance and reliability
    $request_options = [
        'body' => $field_string,
        'allow_redirects' => TRUE,
        'decode_content' => TRUE,
        'http_errors' => FALSE,
        'verify' => FALSE,
        'headers' => $header,
        'timeout' => 30, // 30 second timeout
        'connect_timeout' => 10, // 10 second connection timeout
        'read_timeout' => 20, // 20 second read timeout
    ];

    try {
        $response = \Drupal::httpClient()->request('POST', $url, $request_options);
        $status_code = $response->getStatusCode();

        // Check for successful response
        if ($status_code >= 200 && $status_code < 300) {
            $content = $response->getBody()->getContents();

            // Validate response is not empty and is valid
            if (!empty($content) && strlen($content) > 0) {
                return $content;
            }

            MoLogger::warning('Empty response received from support API');
            return NULL;
        }

        // Log non-successful status codes
        MoLogger::warning('Support API returned status code: @code', ['@code' => $status_code]);
        return NULL;

    } catch (GuzzleException $e) {
        MoLogger::error('Support API Guzzle error: @message', ['@message' => $e->getMessage()]);
        return NULL;
    } catch (\Exception $e) {
        MoLogger::error('Support API general error: @message', ['@message' => $e->getMessage()]);
        return NULL;
    }
  }

  /**
   * This function is used to get the timestamp value.
   *
   * @return string
   *   Returns timestamp.
   */
  public static function getOAuthTimestamp() {
    $url = MoModuleConstant::LICENSE_SERVER_BASE_URL.'/moas/rest/mobile/get-timestamp';
    $content = self::callService($url, '', []);
    if (empty($content)) {
      $currentTimeInMillis = round(microtime(TRUE) * 1000);
      $currentTimeInMillis = number_format($currentTimeInMillis, 0, '', '');
    }
    return empty($content) ? $currentTimeInMillis : $content;
  }

  /**
   * Checks Drupal version of site.
   *
   * @return string
   *   Returns Drupal version of site.
   */
  public static function moGetDrupalCoreVersion() {
    $drupal_version = explode(".", \DRUPAL::VERSION);
    return $drupal_version[0];
  }
}
