<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\Controller;

use Drupal\Component\Utility\Html;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Url;
use Drupal\miniorange_oauth_client\Entity\MoClientConfiguration;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoClientConfiguration\MoGrant\MoUnoReadGrants;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoHooks\MoUnoHooksLoader;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoOperations\MoUnoLoginOperations;
use Drupal\miniorange_oauth_client\MoHelper\MoUtilities;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoLibrary\MoLogger;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\Session;
use function Symfony\Component\String\s;

/**
 * Returns responses for MiniOrange OAuth Client routes.
 */
final class MoAuthorizationRequestController extends ControllerBase
{
  use MoGhostInvoker;

  protected ModuleHandlerInterface $module_handler;
  protected MoClientConfiguration $client_config;
  protected Session $session;
  protected bool $test_sso = false;

  /**
   * The controller constructor.
   */
  public function __construct(
    ModuleHandlerInterface $module_handler,
  )
  {
    $this->module_handler = $module_handler;
    $this->session = $this->initializeSession();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self
  {
    return new MoAuthorizationRequestController(
      $container->get('module_handler'),
    );
  }

  protected final static function getService(string $name): mixed
  {
    return \Drupal::service($name);
  }

  protected final static function initializeSession()
  {
    if (session_status() === PHP_SESSION_NONE) {
      session_start();
    }
    return self::getService('session');
  }

  public final function updateSessionData(string $app_id, string $state): void
  {
    $this->session?->set('mo_oauth2state', $state);
    $this->session?->set('mo_oauth_app_id', $app_id);
    // to verify when the request was initiated and when it was ended (Found a check for this in JWT utils file so implemented it.)
    $this->session?->set('mo_sso_request_time', time());
  }

  // todo if the login page is reached through another site for SSO need to verify that scenario
  //  as of now referer is modified if there is destination it will remove the base url part need to observe
  //  for the last made scenario which is 50 sites client USE CASE
  public function generateState(Request $request): string
  {
    $base_url = Url::fromUri('internal:/', ['absolute' => true])->toString();

    $state['head'] = MoUtilities::base64url_encode(random_bytes(12));
    $state['app_id'] = $this->client_config->id();
    if (!$this->test_sso) {
      //destination parameter contains instance name on local host so eliminating it as it will be invalid url after appending with base url
      if (
        (
          !empty($destination = $request->query->get('destination')) &&
          !empty($destination = ltrim(Html::escape($destination), '/'))
        ) ||
        (
          !empty($destination = $request->getRequestUri()) &&
          !empty($destination = ltrim($destination, '/')) &&
          $destination !== ltrim(
            Url::fromRoute('mo_oauth.authorization_request', ['mo_client_config' => $this->client_config->id()])->toString(),
            '/'
          )
        )
      ) {

        $destination = ltrim(Html::escape($destination), '/');
        if (str_contains($base_url, 'localhost') && !empty($redirect_url = $request->server->get('REDIRECT_URL'))) {
          $instance = explode('/', $redirect_url)[1];
          $destination = str_replace('/' . $instance, '', $destination);
        }
        $state['destination'] = $base_url . $destination;
      }

      //checking if  user redirected to login page from another page on site
      if (empty($state['destination']) && !empty($referer = $request->server->get('HTTP_REFERER'))) {
        $referer_url = parse_url($referer);
        $referer_query = $referer_url['query'] ?? '';
        parse_str($referer_query, $query_params);

        if (!empty($query_params['destination'])) {
          $state['destination'] = $base_url . ltrim($query_params['destination'], '/');
        } elseif (Url::fromUri($referer)->toString() !== Url::fromRoute('user.login', [], ['absolute' => true])->toString()) {
          $state['destination'] = $referer;
        }
      }

      // adding language to the state data for fetching it after log-in
      if ($this->module_handler->moduleExists('locale')) {
        $state['user_lang'] = \Drupal::languageManager()->getCurrentLanguage()->getId();
      }
    } else {
      $state['test_sso'] = true;
    }
    $this->call([MoUnoHooksLoader::class, 'authorizationStateBuild'], ['module_handler' => $this->module_handler, 'state' => &$state]);
    $state['tail'] = MoUtilities::base64url_encode(random_bytes(12));
    return MoUtilities::base64url_encode(json_encode($state));
  }

  /**
   * Builds the response.
   */
  public function __invoke(Request $request)
  {
    try {
      \Drupal::service('page_cache_kill_switch')->trigger();
      $this->call([MoUnoLicenseTierManager::class, 'validateModuleLicense']);
      $this->client_config = $request->attributes->get(MoClientConfiguration::getEntityId());
      $query_params = MoUtilities::sanitizeRecursive($request->query->all());
      $this->test_sso = (bool)($query_params['test_sso'] ?? 0);
      if(!$this->test_sso) {
        // this piece is to create user login reports
        $this->call(
          [MoUnoLoginOperations::class, 'buildOrUpdateLoginReports'],
          ['report_info' => ['status' => 'initiated'], 'client_config_id' => $this->client_config->id()]
        );
      }
      if (!$this->client_config->getEnableLoginWithOauth() && !$this->test_sso) {
        throw new \Exception('Please ensure the Test Connection is Successfull and Login with OAuth is enabled.');
      }

      $state = $this->generateState($request);
      $this->updateSessionData($this->client_config->id(), $state);
      $redirect_url = $this->call(
        [MoUnoReadGrants::class, 'buildAuthorizationUrl'], ['client_config' => $this->client_config, 'state' => $state]
      );
      if ($this->client_config->getGrantType() !== 'password') {
        $this->call(
          [MoUnoHooksLoader::class, 'preAuthReqProcess'],
          ['module_handler' => $this->module_handler, 'authorization_url' => &$redirect_url, 'query_params' => $query_params]
        );
        return new TrustedRedirectResponse($redirect_url);
      }
    } catch (\Throwable $exception) {
      $redirect_url = '';

      $err_resp = MoUtilities::buildErrorResponse($redirect_url, $exception, $this->test_sso);
      if ($this->test_sso) {
        return $err_resp;
      }
    }
    return new RedirectResponse($redirect_url);
  }

}
