<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\miniorange_oauth_client\Entity\MoClientConfiguration;
use Drupal\miniorange_oauth_client\Entity\MoClientSettings;
use Drupal\miniorange_oauth_client\Form\MoTestConfigurationResultForm;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoFeatures\MoEnterpriseFeatures\MoClientConfiguration\MoGrant\MoGrantTypes\MoEntImplicitGrant;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoClientConfiguration\MoAuthProtocols\MoUnoReadProtocols;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoHooks\MoUnoHooksLoader;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoOperations\MoUnoClientSettings;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoOperations\MoUnoAttributeMapping;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoOperations\MoUnoAttributeRestriction;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoOperations\MoUnoGroupMapping;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoOperations\MoUnoLoginOperations;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoOperations\MoUnoModuleSettings;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoOperations\MoUnoProfileManagement;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoOperations\MoUnoRoleMapping;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoOperations\MoUnoUserManagement;
use Drupal\miniorange_oauth_client\MoHelper\MoUtilities;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoLibrary\MoLogger;
use Drupal\user\Entity\User;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\Session;

/**
 * Returns responses for MiniOrange OAuth Client routes.
 *
 *
 * todo:
 *  Possible use cases need to consider
 *  1. What if the module user disabled the anonymous session creation?
 *      ans: check for it first and store the state in any local storage (ex: state table of drupal with any identity related to particular request)
 */
final class MoAuthorizationResponseController extends ControllerBase
{
  use MoGhostInvoker;

  protected Session $session;
  protected ?MoClientConfiguration $client_config;
  protected ModuleHandlerInterface $module_handler;
  protected array $state = [];
  protected bool $test_sso = false;
  protected bool $user_creation_feature_restriction_err = false;
  /**
   * @var mixed|null
   */
  protected ?string $code = '';
  protected bool $is_refresh_token_grant = false;

  public function __construct(ModuleHandlerInterface $module_handler)
  {
    $this->module_handler = $module_handler;
    $this->session = $this->initializeSession();
  }

  public static function create(ContainerInterface $container): MoAuthorizationResponseController
  {
    return new MoAuthorizationResponseController(
      $container->get('module_handler')
    );
  }

  protected final static function getService(string $name): mixed
  {
    return \Drupal::service($name);
  }

  protected final static function initializeSession()
  {
    if (session_status() === PHP_SESSION_NONE) {
      session_start();
    }
    return self::getService('session');
  }

  /**
   *
   * This function is to separate the state from the request params and validate it with the local stored state
   *
   * @param array|null $query_params
   *
   * todo state validation part need's a fix
   * @throws \Exception
   */
  private function validateState(Request $request, ?array $query_params = []): void
  {
    $req_state = $query_params['state'] ?? null;
    $this->state = MoUtilities::decodeState($req_state);
    /**
     * todo need to check the session storage part from authorization request creation
     *  ---was implemented by considering the data in the session was stored as array -> means before encoding the state
     */
    $session_state = $this->session->get('mo_oauth2state');
    $this->test_sso = !empty($this->state['test_sso']) && $this->state['test_sso'] === true;

    if ($req_state !== $session_state) {
      throw new \Exception("State mismatched. Please try the request again.");
    }
    $current_app_id = $this->state['app_id'] ?? $this->session->get('mo_oauth_app_id') ?? '';
    if ($current_app_id !== $this->client_config->id()) {
      throw new \Exception('The state app-name does not match with the config app-name while state validation.');
    }
    $request->attributes->set(
      'mo_redirect_url',
      ($request->attributes->get('mo_redirect_url') ?? []) +
      [MoModuleConstant::STATE_DESTINATION => $this->state['destination'] ?? '']
    );
  }

  /**
   * Builds the response.
   * @throws \Exception
   * @throws \Throwable
   *
   * todo scenarios
   *  1. what if a person manually changes grant type to implicit in DB, the method will be initiated - (not found)
   *     so next line some other issue will be shown- will not be handled extraordinarily
   *     ans : how about creating a third param in ghost invoker to perform fail safe(if not found just simply fail.)
   *           This shows exact errors for exact use case
   *  2. user object is saved often how about storing it once after appending with all the data in mapping?
   */
  public function __invoke(Request $request)
  {

    try {
      /** @var MoClientConfiguration $client_config */
      $this->client_config = $request->attributes->get(MoClientConfiguration::getEntityId());
      $query_params = MoUtilities::sanitizeRecursive($request->query->all());
      $grant_type = $this->client_config->getGrantType();
      $this->is_refresh_token_grant = $grant_type == 'refresh_token';
      if (!$this->is_refresh_token_grant) {
        if ($grant_type == 'implicit') {
          $this->call(
            [MoEntImplicitGrant::class, 'handleAuthorizationCallback'],
            ['query_params' => $query_params]
          );
        } else {
          $this->code = $query_params['code'] ?? null;
        }
        // validate the state parameter
        $this->validateState($request, $query_params);

        // the below part is to import a hook into the loader
        $this->call(
          [MoUnoHooksLoader::class, 'preAuthRespProcess'],
          ['module_handler' => $this->module_handler, 'query_params' => $query_params]
        );
      }
      $token_response = $this->fetchToken($query_params);

      if (!$this->is_refresh_token_grant) {
        $this->session->remove('mo_oauth2state');

        // the below part is to import a hook into the loader
        $this->call(
          [MoUnoHooksLoader::class, 'postTokenRequest'],
          ['module_handler' => $this->module_handler, 'token_data' => $token_response, 'state' => $this->state]
        );
      }

      $response = $this->performSSOInDrupalByToken($token_response);

      // todo this piece works in openId flow need to implement it in oauth floe as well
      //  removing it at the final piece
      $this->session->remove('mo_sso_request_time');

    } catch (\Throwable $exception) {
      if ($this->is_refresh_token_grant) {
        throw $exception;
      }
      $redirect_url = \Drupal::request()->attributes->get('mo_redirect_url', '');
      if (is_array($redirect_url)) {
        $redirect_url = !empty($urls = array_filter($redirect_url)) ? $redirect_url[max(array_keys($urls))] : '';
      }
      $err_resp = MoUtilities::buildErrorResponse($redirect_url, $exception, $this->test_sso, $this->user_creation_feature_restriction_err);
      if ($this->test_sso) {
        return $err_resp;
      }
      $response = new RedirectResponse($redirect_url);
    }
    return $response;
  }

  /**
   * This function is to perform the after token fetch operations
   *
   * This method is seperated from actual flow is because we had a client who wants to perform SSO in drupal with the token he has
   *
   * @param array $token_data
   * @param string $token_app_name
   * @return mixed
   *
   * @throws \Exception
   * @throws \Throwable
   */
  public function performSSOInDrupalByToken(array $token_data, string $token_app_name = ''): mixed
  {
    $request = \Drupal::requestStack()->getCurrentRequest();
    try {
      if (!$this->is_refresh_token_grant) {
        (empty($token_app_name) && !empty($this->client_config)) ||
        !empty($this->client_config = MoClientConfiguration::load($token_app_name)) ||
        throw new \Exception("The is no configurations stored against provided app name =>'$token_app_name'");
        $module_version = $this->call([MoUnoLicenseTierManager::class, 'getLicenseVersion']);
        //this piece is for  gathering userInfo
        $resource_owner = $this->fetchResourceOwner($token_data);

        $normalized_resource_owner = MoUtilities::normalizeRecursive($resource_owner);

        // todo the below part for performing operations over the fetched resource owner data
        if ($this->test_sso) {
          // save the fetched server details to the client application
          $this->client_config->setServerTokenAttributes(
            json_encode(
              !empty($token_data) ?
                array_combine(array_keys($token_data), array_keys($token_data)) : []
            )
          );
          $this->client_config->setServerTestAttributes($normalized_resource_owner);
          $this->client_config->save();

          // it displays the info and terminated the process
          $form = new MoTestConfigurationResultForm($normalized_resource_owner, $this->client_config);
          return $this->formBuilder()->getForm($form);
        }

        // this piece is for applying restriction on the fetched attributes
        $this->call(
          [MoUnoAttributeRestriction::class, 'validateAttributeRestriction'],
          ['client_id' => $this->client_config->id(), 'resource_owner' => $normalized_resource_owner]
        );


        // this part is to fetch the user email
        $email = $this->call(
          [MoUnoAttributeMapping::class, 'getUserEmail'],
          ['resource_owner' => $normalized_resource_owner, 'client_id' => $this->client_config->id()]
        );

        // this piece is to validate the domain restriction
        $this->call([MoUnoUserManagement::class, 'checkDomainRestriction'], ['email' => $email]);

        // this part checks for user exist in drupal and returns the user object
        /** @var User $user */
        $user = $this->call([MoUnoUserManagement::class, 'checkUserExists'], ['email' => $email]);

        if (empty($user) || empty($user->id())) {
          // if user is not present create user and return the user object
          $user = $this->call(
            [MoUnoUserManagement::class, 'createUser'],
            ['email' => $email, 'resource_owner' => $normalized_resource_owner, 'client_id' => $this->client_config->id()]
          );

          if (empty($user) || empty($user->id())) {
            if ($module_version == MoModuleConstant::MO_FREE_VERSION) {
              $this->user_creation_feature_restriction_err = true;

              throw new \Exception(
                "Feature unavailable: The user with email ‘" . $email . "’
                     could not be created because this functionality is not available in the community version of the module.
                     <br>
                     Only existing users can log in. The ability to automatically create new users is available in the licensed versions of the module.
                     <br>To upgrade, please " .
                Link::fromTextAndUrl('click here.', Url::fromUri(
                  MoModuleConstant::MO_Landing_PAGE_PRICING,
                  [
                    'attributes' => [
                      'target' => '_blank',
                      'rel' => 'noopener noreferrer',
                    ],
                  ]
                ))->toString()
              );
            } else {
              throw new \Exception("Issue in Creating User with email ->" . $email);
            }
          }

          $request->attributes->set('is_new_user', true);

          // this part is for mapping default role while creating new user
          $this->call(
            [MoUnoRoleMapping::class, 'mapDefaultUserRole'],
            ['user' => &$user, 'client_id' => $this->client_config->id()]
          );
        }

        // this piece is to update user status to active on sso
        $this->call([MoUnoUserManagement::class, 'updateUserStatusOnSSO'], ['user' => &$user]);

        if (!reset($user->toArray()['status'])['value']) {
          throw new \Exception('Unable to login user with email -' . $email . '. The user Account is Inactive');
        }
        // this is for custom role mapping
        $this->call(
          [MoUnoAttributeMapping::class, 'mapCustomAttributes'],
          ['user' => &$user, 'client_id' => $this->client_config->id(), 'resource_owner' => $normalized_resource_owner]
        );

        // this piece is for custom role mapping
        $this->call(
          [MoUnoRoleMapping::class, 'mapCustomUserRoles'],
          ['user' => &$user, 'client_id' => $this->client_config->id(), 'resource_owner' => $normalized_resource_owner]
        );

        // this piece is to map user profiles
        $this->call(
          [MoUnoProfileManagement::class, 'mapUserProfiles'],
          ['user' => $user, 'client_id' => $this->client_config->id(), 'resource_owner' => $normalized_resource_owner]
        );

        // this piece is to map user groups
        $this->call(
          [MoUnoGroupMapping::class, 'mapUserGroups'],
          ['user' => $user, 'client_id' => $this->client_config->id(), 'resource_owner' => $normalized_resource_owner]
        );

        // this piece is to perform user login operation
        $this->call([MoUnoLoginOperations::class, 'userLogin'], ["user" => $user, 'client_id' => $this->client_config->id()]);

        if (!$this->test_sso) {
          // this piece is to create user login reports
          $this->call(
            [MoUnoLoginOperations::class, 'buildOrUpdateLoginReports'],
            [
              'report_info' => [
                'username' => $user->getAccountName(),
                'status' => 'success',
                'email' => $user->getEmail(),
              ]
            ]
          );
        }

        $this->call([MoUnoUserManagement::class, 'forceUpdateUserProfileAttributesOnSSO'], ['user' => $user]);
      }

      // this piece is to tie token expiry to drupal user session expiry
      $this->call([MoUnoClientSettings::class, 'tieTokenExpiryToUserSession'], ['user_token' => $token_data, 'client_id' => $this->client_config->id()]);

      // this piece is to save token in cookies and session
      $this->call([MoUnoModuleSettings::class, 'storeTokenInSessionAndCookie'], ['user_token' => $token_data]);

    } catch (\Throwable $exception) {
      $this->call([MoUnoLoginOperations::class, 'buildOrUpdateLoginReports'], ['report_info' => ['status' => 'failed']]);
      if ($this->is_refresh_token_grant) {
        throw $exception;
      }
      $anonymous_redirect = MoClientSettings::readMe($this->client_config->id())?->getAnonymousUserRedirectUrl();
      \Drupal::request()->attributes->set(
        'mo_redirect_url',
        (\Drupal::request()->attributes->get('mo_redirect_url', [])) +
        [
          MoModuleConstant::ANONYMOUS_REDIRECT =>
            !empty($anonymous_redirect) ?
              Url::fromUri($anonymous_redirect, ['query' => ['mo_force_stop_redirect' => true]])->toString() :
              Url::fromRoute(MoModuleConstant::DEFAULT_REDIRECT_ROUTE_NAME, ['mo_force_stop_redirect' => true])->toString()
        ]
      );
      if (!empty($token_app_name)) {
        MoLogger::error($exception->getMessage());
      }
      $error_resp = 1;
    }
    if (!$this->is_refresh_token_grant) {
      $redirect_url = \Drupal::request()->attributes->get('mo_redirect_url', '');
      if (is_array($redirect_url)) {
        $redirect_url = !empty($urls = array_filter($redirect_url)) ? $redirect_url[max(array_keys($urls))] : '';
      }
      if (empty($redirect_url) && !empty($user)) {
        $redirect_url = Url::fromRoute('entity.user.canonical', ['user' => $user->id()])->toString();
      }

      if (isset($error_resp) && $error_resp) {
        \Drupal::request()->attributes->set('mo_redirect_url', $redirect_url);
        throw new \Exception($exception->getMessage());
      }

      return new RedirectResponse($redirect_url);
    }
    return new Response();
  }

  /**
   * @throws \Exception
   */
  private function fetchToken(array $query_params = []): array
  {
    // validate the license and set it in the request body so it can be ignored in the resource owner fetch call
    $this->call([MoUnoLicenseTierManager::class, 'validateModuleLicense']);
    \Drupal::request()->attributes->set('mo_license_validated_at_token_call', true);
    return $this->call(
      [MoUnoReadProtocols::class, 'gatherTokenFromIDP'],
      ['client_config' => $this->client_config, 'code' => $this->code, 'query_params' => $query_params]
    );
  }

  /**
   * @throws \Exception
   */
  private function fetchResourceOwner(array $token_data)
  {
    if (!\Drupal::request()->attributes->get('mo_license_validated_at_token_call', false)) {
      // Here we are again verifying the license because once a customer requested to perform SSO from our module
      //  with a token provided from his end. so we created a method 'performSSOInDrupalByToken' so by that method the
      //  token fetch method will be useless so we need to validate the license.
      $this->call([MoUnoLicenseTierManager::class, 'validateModuleLicense']);
    }
    return $this->call(
      [MoUnoReadProtocols::class, 'gatherUserInfoFromIDP'],
      ['client_config' => $this->client_config, 'token_data' => $token_data]
    );
  }

}
