<?php

namespace Drupal\miniorange_oauth_client\Controller;

use Drupal;
use Drupal\Component\Render\MarkupInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\Core\Render\Markup;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Url;
use Drupal\miniorange_oauth_client\Entity\MoClientConfiguration;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoLicense\MoDTO\MoUnoLicense;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoOperations\MoUnoLogoutOperation;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;

class MoModuleController extends ControllerBase
{
  use MoGhostInvoker;
  public function downloadFiles(): RedirectResponse|Response
  {
    $store = \Drupal::service('tempstore.private')->get(MoModuleConstant::MODULE_NAME);
    $request = Drupal::request();

    $config_id = $request->get('config_id');
    $data = $store->get($config_id);

    if (!$data) {
      $this->messenger()->addError($this->t('No data available to download.'));
      return $this->redirect('<front>');
    }

    $response = new Response($data);
    $disposition = $response->headers->makeDisposition(
      ResponseHeaderBag::DISPOSITION_ATTACHMENT,
      $request->get('file_name'),
    );
    $response->headers->set('Content-Type', 'application/json');
    $response->headers->set('Content-Disposition', $disposition);

    $store->delete($config_id);

    return $response;
  }

  public function openCustomerSupportForm(): AjaxResponse
  {
    $response = new AjaxResponse();
    $params['form_type'] = 'customer-support';
    $modal_form = $this->formBuilder()->getForm('\Drupal\miniorange_oauth_client\Form\MoCustomerSupport', $params);
    $response->addCommand(new OpenModalDialogCommand('Contact miniOrange Support', $modal_form, ['width' => '45%'] ) );
    return $response;
  }

  /**
   * @throws \Exception
   */
  public function notifyUsPopUp(): AjaxResponse
  {
    $response = new AjaxResponse();
    if ($this->call([MoUnoLicenseTierManager::class, 'getLicenseVersion']) !== MoModuleConstant::MO_FREE_VERSION) {
      return $response;
    }
    $request = \Drupal::request();
    $action = $request->get('action', 'open');
    $page_for = $request->get('page_for', 'notify-us');
    $page_title = $request->get('page_title', 'Let\'s Get Acquainted');
    if ($action == 'open') {
      if (!\Drupal::state()->get('mo-oauth-free-tier-notify-us-closed', false) || $page_for !== 'notify-us') {
        $params['form_type'] = $page_for;
        $modal_form = $this->formBuilder()->getForm('\Drupal\miniorange_oauth_client\Form\MoCustomerSupport', $params);
        $response->addCommand(
          new OpenModalDialogCommand(
            t($page_title),
            $modal_form,
            ['width' => '45%', 'dialogClass' => 'mo-notify-us-dialog']
          )
        );
      }
    } elseif ($action == 'close') {
      \Drupal::state()->set('mo-oauth-free-tier-notify-us-closed', true);
    }
    return $response;
  }


  /**
   * todo: Need to define an endpoint for this method and define the functionality for this.
   *     case study: Azure provide Front-channel logout option for performing SLO on user logout from azure,
   *                 so the request was initiated from azure with param sid => (Session ID).
   *                 As per my understanding we should get the session-id in the id_token claims on SSO.
   *                 we can store it in session table and use the same to logout the user.
   *
   * @throws \Exception
   */
  public function frontChannelLogout(): TrustedRedirectResponse
  {
    $destination = \Drupal::request()->query->get('redirect');
    $this->call([MoUnoLogoutOperation::class, 'idpInitiatedLogout']);
    return new TrustedRedirectResponse($destination);
  }

  public function customizeRouteTitle(): MarkupInterface|string
  {
    $title = 'miniOrange OAuth Client ';

    // building upgrade URL
    if ($this->call([MoUnoLicenseTierManager::class, 'getLicenseVersion']) !== MoModuleConstant::MO_ENTERPRISE_VERSION) {
      $url = Url::fromUri(MoModuleConstant::MO_Landing_PAGE_PRICING);
      $link = Link::fromTextAndUrl('⬆ UPGRADE', $url)->toRenderable();
      $link['#attributes']['target'] = '_blank';
      $link['#attributes']['rel'] = 'noopener noreferrer';
      $link['#attributes']['class'] = ['button', 'button--primary', 'button--extrasmall'];
      $title .= \Drupal::service('renderer')->renderPlain($link);
    }

    // build request Demo button
    if ($this->call([MoUnoLicenseTierManager::class, 'getLicenseVersion']) == MoModuleConstant::MO_FREE_VERSION) {

      $url = Url::fromRoute(
        'mo_oauth.free_tier_notify_us_popup',
        ['action' => 'open', 'page_for' => 'trial-request', 'page_title' => 'Request 7 Days Free Trial']
      );
      $link = Link::fromTextAndUrl(t('Request Trial'), $url)->toRenderable();
      $link['#attributes']['class'] = ['button', 'button--primary', 'button--extrasmall', 'use-ajax',];
      $link['#attributes']['data-dialog-type'] = 'modal';
      $link['#attached']['library'][] = 'core/drupal.dialog.ajax';

      $title .= \Drupal::service('renderer')->renderPlain($link);
    }

    return Markup::create($title);
  }

  public function addClientConfig()
  {
    $module_version = $this->call([MoUnoLicenseTierManager::class, 'getLicenseVersion']);
    $license_config = $this->call([MoUnoLicense::class, 'readMe']);
    $provided_count = $license_config?->getNoOfSp() ?? 1;

    $my_list = count(MoClientConfiguration::loadMultiple());
    $response = new AjaxResponse();

    if (($provided_count - $my_list) <= 0) {
      $module_version !== MoModuleConstant::MO_FREE_VERSION || $module_version = 'community';
      if ($module_version == MoModuleConstant::MO_ENTERPRISE_VERSION) {
        $support_link = Link::fromTextAndUrl(
          MoModuleConstant::SUPPORT_EMAIL, Url::fromUri('mailto:'.MoModuleConstant::SUPPORT_EMAIL)
        )->toString();
        if ($license_config && $license_config->getCustomerId() === null) {
          $title = t('Authentication Required!');
          $message = t('You are not logged in to the module. Please log in to configure with the IDP.');
        } else {
          $title = t('Limit Exceeded!');
          $message = t(
            'Limit exceeded. You cannot add more items. Please reach out to us at @support to increase limits.',
            ['@support' => $support_link]
          );
        }
      } else {
        $features_link = Link::fromTextAndUrl(
          '[ENTERPRISE]',
          Url::fromUri(
            MoModuleConstant::MO_Landing_PAGE_FEATURES,
            [
              'attributes' => [
                'target' => '_blank',
                'rel' => 'noopener noreferrer',
              ],
            ]
          )
        )->toString();
        $title = t('Add New OAuth/OpenID Provider');
        $message = t(
          'You can configure only 1 application in the ' . $module_version . ' version of this module.
        Multiple OAuth/OpenID Providers are supported only in @link version of the module.',
          ['@link' => $features_link]
        );
      }

      $markup = [
        '#type' => 'markup',
        '#markup' => t('<div class="messages messages--error">' . $message . '</div>')
      ];

      $response->addCommand(new OpenModalDialogCommand($title, $markup, ['width' => '50%']));
      return $response;
    }
    $response->addCommand(new RedirectCommand(Url::fromRoute('entity.mo_client_config.add_form')->toString()));
    return $response;
  }
}
