<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;

/**
 * Defines the miniOrange client settings entity type.
 *
 * @ConfigEntityType(
 *   id = "mo_client_settings",
 *   label = @Translation("miniOrange Client Settings Entity"),
 *   label_collection = @Translation("miniOrange Client Settings Entities"),
 *   label_singular = @Translation("miniOrange Client Settings Entity"),
 *   label_plural = @Translation("miniOrange Client Settings Entities"),
 *   label_count = @PluralTranslation(
 *     singular = "@count miniOrange Client Settings Entity",
 *     plural = "@count miniOrange Client Settings Entities",
 *   ),
 *   config_prefix = "mo_client_settings",
 *   handlers = {
 *     "form" = {
 *       "edit" = "Drupal\miniorange_oauth_client\Form\MoClientSettingsForm",
 *     },
 *   },
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "id",
 *   },
 *   config_export = {
 *     "id",
 *     "auto_create_user",
 *     "redirect_unregister",
 *     "disable_new_user",
 *     "disabled_user_redirect_url",
 *     "anonymous_user_redirect_url",
 *     "login_redirect_url",
 *     "logout_redirect_url",
 *     "same_page_after_logout",
 *     "role_based_restriction",
 *     "role_restriction_attrname",
 *     "typeof_role_restriction",
 *     "list_of_restricted_roles",
 *     "redirect_url_for_role_restriction",
 *     "enable_slo",
 *     "single_logout_url",
 *     "idp_logout_redirect_uri_param",
 *     "include_id_token_in_slo",
 *     "idp_logout_token_param",
 *     "tie_token_expiry_to_usr_session",
 *     "action_after_token_expiry",
 *     "revoke_user_token_from_idp",
 *     "token_revoke_url",
 *     "token_revoke_param_name",
 *     "token_revoke_additional_params",
 *     "idp_token_expiry_param"
 *   },
 * )
 */
final class MoClientSettings extends ConfigEntityBase {
  use MoGhostInvoker;

  protected string $id;
  protected ?bool $auto_create_user = null;
  protected ?string $redirect_unregister = '';
  protected ?bool $disable_new_user = false;
  protected ?string $disabled_user_redirect_url = '';
  protected ?string $anonymous_user_redirect_url = '';
  protected ?string $login_redirect_url = '';
  protected ?bool $same_page_after_logout = false;
  protected ?string $logout_redirect_url = '';
  protected ?bool $role_based_restriction = false;
  protected ?string $role_restriction_attrname = '';
  protected ?string $typeof_role_restriction = '';
  protected ?string $list_of_restricted_roles = '';
  protected ?string $redirect_url_for_role_restriction = '';
  protected ?bool $enable_slo = false;
  protected ?string $single_logout_url = '';
  protected ?string $idp_logout_redirect_uri_param = 'post_logout_redirect_uri';
  protected ?bool $include_id_token_in_slo = false;
  protected ?string $idp_logout_token_param = 'id_token_hint';
  protected ?bool $tie_token_expiry_to_usr_session = false;
  protected ?string $idp_token_expiry_param = null;
  protected ?string $action_after_token_expiry = null;
  protected ?bool $revoke_user_token_from_idp = false;
  protected ?string $token_revoke_url = '';
  protected ?string $token_revoke_param_name = 'token';
  protected ?array $token_revoke_additional_params = [];

  /**
   * @return string
   */
  public function getId(): string
  {
    return $this->id;
  }

  /**
   * @param string $id
   * @return MoClientSettings
   */
  public function setId(string $id): MoClientSettings
  {
    $this->id = $id;
    return $this;
  }

  /**
   * @return bool|null
   * @throws \Exception
   */
  public function getAutoCreateUser(): ?bool
  {
    return is_null($this->auto_create_user) ?
      !($this->call([MoUnoLicenseTierManager::class, 'getLicenseVersion']) == MoModuleConstant::MO_FREE_VERSION) :
      $this->auto_create_user;
  }

  /**
   * @return string|null
   */
  public function getRedirectUnregister(): ?string
  {
    return $this->redirect_unregister;
  }

  /**
   * @return bool|null
   */
  public function getDisableNewUser(): ?bool
  {
    return $this->disable_new_user;
  }

  /**
   * @return string|null
   */
  public function getDisabledUserRedirectUrl(): ?string
  {
    return $this->disabled_user_redirect_url;
  }

  /**
   * @return string|null
   */
  public function getAnonymousUserRedirectUrl(): ?string
  {
    return $this->anonymous_user_redirect_url;
  }

  /**
   * @return string|null
   */
  public function getLoginRedirectUrl(): ?string
  {
    return $this->login_redirect_url;
  }

  /**
   * @return bool|null
   */
  public function getSamePageAfterLogout(): ?bool
  {
    return $this->same_page_after_logout;
  }

  /**
   * @return string|null
   */
  public function getLogoutRedirectUrl(): ?string
  {
    return $this->logout_redirect_url;
  }

  /**
   * @return bool|null
   */
  public function getRoleBasedRestriction(): ?bool
  {
    return $this->role_based_restriction;
  }

  /**
   * @return string|null
   */
  public function getRoleRestrictionAttrName(): ?string
  {
    return $this->role_restriction_attrname;
  }

  /**
   * @return string|null
   */
  public function getTypeOfRoleRestriction(): ?string
  {
    return $this->typeof_role_restriction;
  }

  /**
   * @return string|null
   */
  public function getListOfRestrictedRoles(): ?string
  {
    return $this->list_of_restricted_roles;
  }

  /**
   * @return string|null
   */
  public function getRedirectUrlForRoleRestriction(): ?string
  {
    return $this->redirect_url_for_role_restriction;
  }

  public function getEnableSlo(): ?bool
  {
    return $this->enable_slo;
  }

  public function getSingleLogoutUrl(): ?string
  {
    return $this->single_logout_url;
  }

  /**
   * @return bool|null
   */
  public function getIncludeIdTokenInSlo(): ?bool
  {
    return $this->include_id_token_in_slo;
  }

  public function getIdpLogoutRedirectUriParam(): ?string
  {
    return $this->idp_logout_redirect_uri_param;
  }

  public function getIdpLogoutTokenParam(): ?string
  {
    return $this->idp_logout_token_param;
  }
  public function getTieTokenExpiryToUsrSession(): ?bool
  {
    return $this->tie_token_expiry_to_usr_session;
  }
  public function getIdpTokenExpiryParam(): ?string
  {
    return $this->idp_token_expiry_param;
  }
  public function getActionAfterTokenExpiry(): ?string
  {
    return $this->action_after_token_expiry ?? null;
  }
  public function getRevokeUserTokenFromIdp(): ?bool
  {
    return $this->revoke_user_token_from_idp;
  }
  public function getTokenRevokeUrl(): ?string
  {
    return $this->token_revoke_url;
  }
  public function getTokenRevokeParamName(): ?string
  {
    return $this->token_revoke_param_name;
  }
  public function getTokenRevokeAdditionalParams(): ?array
  {
    return $this->token_revoke_additional_params;
  }
  public static function readMe(string $id): self
  {
    return self::load($id) ?? self::create(['id' => $id]);
  }
}
