<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\Form;

use Drupal\Component\Render\MarkupInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CssCommand;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;
use Drupal\miniorange_oauth_client\Entity\MoClientConfiguration;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoLicense\MoLibrary\MoUnoSubscriptionLibrary;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoClientConfiguration\MoAuthProtocols\MoUnoReadProtocols;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoClientConfiguration\MoGrant\MoUnoReadGrants;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoTutorial;
use Drupal\miniorange_oauth_client\MoHelper\MoClientConfigurationHelper;
use Drupal\miniorange_oauth_client\MoHelper\MoFallBackSupport;
use Drupal\miniorange_oauth_client\MoHelper\MoSupport;
use Drupal\miniorange_oauth_client\MoHelper\MoUtilities;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoLibrary\MoLogger;
use Drupal\miniorange_oauth_client\MoService\MoLayoutBuilder;

/**
 * MiniOrange Client Configuration Entity form.
 */
final class MoClientConfigurationForm extends EntityForm
{
  use MoLayoutBuilder;
  use MoGhostInvoker;

  private function addLibraries(array &$form): void
  {
    $form['libraries']['#attached']['library'] = [
      // JS component to add the authentication methods
      'miniorange_oauth_client/authentication_method_js',
      // Drupal default AJAX library
      "core/drupal.dialog.ajax",
      // Drupal new window library
      'miniorange_oauth_client/test_config_window_js',
      // Copy icon library
      'miniorange_oauth_client/copy_icon_js',
      // Common Js for hiding the message
      'miniorange_oauth_client/common_js',
    ];
  }

  /**
   * {@inheritdoc}
   * @throws \Exception
   */
  public function form(array $form, FormStateInterface $form_state): array
  {
    $form = parent::form($form, $form_state);
    $entity = $this->entity;

    $this->addLibraries($form);

    $disabled = $this->call([MoUnoSubscriptionLibrary::class, 'getModuleStatus'], ['form' => &$form]);

    if (!$entity->isNew()) {
      $form['attr_map_warning'] = MoClientConfigurationHelper::getAttributeMappingMarkup();
    }

    $form['entity_is_new'] = [
      '#type' => 'hidden',
      '#value' => (int)$entity->isNew(),
    ];

    $form['client_conf_is_enabled'] = [
      '#type' => 'hidden',
      '#value' => (int)!$disabled
    ];

    $form['callback_disabled'] = [
      '#type' => 'hidden',
      '#value' => 1
    ];

    $form['setup_guide_url'] = [
      '#type' => 'hidden',
      '#default_value' => $entity->getSetupGuideUrl(),
      '#attributes' => [
        'id' => 'setup-guide-url',
      ]
    ];

    $form['client_app'] = [
      '#title' => t('Select Application'),
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#id' => 'client_app',
      '#type' => 'select',
      '#default_value' => $entity->getClientApp(),
      '#required' => true,
      '#description' => t("Select an OAuth Server"),
      '#options' => ['' => '-- Select --'] + MoClientConfigurationHelper::getIdentityProviders(),
      '#field_suffix' => '<span id="idp-setup-guide">'.$this->buildSetupGuideLink($entity->getSetupGuideUrl()).'</span>',
      '#states' => [
        'disabled' => [
          [':input[name="entity_is_new"]' => ['value' => '0']],
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
      '#ajax' => [
        'callback' => '::updateEndpoints',
        'event' => 'change',
      ],
    ];

    $form['app_name'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#title' => $this->t('Custom App Name'),
      '#required' => TRUE,
      '#default_value' => $entity->getAppName(),
      '#id' => 'app-name',
      '#description' => $this->t('The unique name for your application.'),
      '#states' => [
        'disabled' => [
          [':input[name="entity_is_new"]' => ['value' => '0']],
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
      '#ajax' => [
        'callback' => '::changeCallback',
        'event' => 'change',
      ],
    ];

    $url_path = Url::fromUri('internal:/', ['absolute' => true])->toString() . \Drupal::service('extension.list.module')->getPath('miniorange_oauth_client') . '/assets/img/copy.png';

    $form['callback_url'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#title' => $this->t('Callback/Redirect URL'),
      '#default_value' => MoUtilities::getCallbackUrl(MoUtilities::getModifiedAppName($entity->id())),
      '#attributes' => [
        'id' => 'callback-url',
        'class' => ['callback-url'],
        'placeholder' => $this->t('Please enter the Custom App Name (currently pending).'),
      ],
      '#field_suffix' => '<span class="mo-copy-icon-span" data-target="callback-url" title="Copy">
                  <img class="mo-copy-icon" src="' . $url_path . '" alt="Copy"/>
                </span>',
      '#states' => [
        'disabled' => [
          [':input[name="callback_disabled"]' => ['value' => '1']],
        ],
      ],
    ];

    $form['client_id'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#id' => 'client_id',
      '#default_value' => $entity->getClientId(),
      '#required' => true,
      '#title' => t('Client Id'),
      '#states' => [
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $grants = $form_state->get('mo_client_config_grants') ?? [];
    !empty($grants) || $this->call([MoUnoReadGrants::class, 'getGrantTypes'], ['grants' => &$grants]);
    $form_state->set('mo_client_config_grants', $grants);

    $form['grant_type'] = [
      '#type' => 'select',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#title' => t('Grant Type'),
      '#options' => $grants,
      '#default_value' => $entity->getGrantType(),
      '#id' => 'grant_type',
      '#required' => true,
      '#states' => [
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
      '#ajax' => [
        'callback' => '::updateRadioButtons',
        'wrapper' => 'authentication-method-frame',
        'event' => 'change',
      ],
    ];

    $form['client_secret'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#default_value' => $entity->id() ? MoClientConfigurationHelper::maskClientSecret(MoClientConfiguration::load($entity->id())->getClientSecret()) : "",
      '#description' => "You will get this value from your OAuth Server",
      '#id' => 'client_secret',
      '#title' => t('Client Secret'),
      '#states' => [
        'visible' => [
          ':input[name = "grant_type"]' => ['!value' => 'implicit'],
        ],
        'required' => [
          [':input[name = "grant_type"]' => ['value' => 'authorization_code']],
          [':input[name = "grant_type"]' => ['value' => 'password']],
        ],
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['scope'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#id' => 'scope',
      '#required' => true,
      '#description' => "You can edit the value of this field, but we highly recommend not changing the default values of this field",
      '#title' => t('Scope'),
      '#default_value' => $entity->getScope(),
      '#states' => [
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];
    $login_protocols = $form_state->get('mo_login_flow_options') ?? [];
    !empty($login_protocols) || $this->call([MoUnoReadProtocols::class, 'getAuthProtocols'], ['protocols' => &$login_protocols]);
    $form_state->set('mo_login_flow_options', $login_protocols);

    $form['login_protocol_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#open' => TRUE,
      '#title' => t('Select Your Login Flow'),
      '#attributes' => [
        "style" => ['max-width: 504px;'],
      ],
      '#states' => [
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['login_protocol_frame']['login_protocol'] = [
      '#type' => 'radios',
      '#default_value' => $entity->getLoginProtocol(),
      '#options' => $login_protocols,
      '#attributes' => [
        'id' => 'login-protocol',
      ],
      '#states' => [
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['authorize_endpoint'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#id' => 'authorize-endpoint',
      '#default_value' => $entity->getAuthorizeEndpoint(),
      '#title' => t('Authorization Endpoint'),
      '#states' => [
        'invisible' => [
          ':input[name = "grant_type"]' => ['value' => 'password'],
        ],
        'optional' => [
          [':input[name="grant_type"]' => ['value' => 'password']],
        ],
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ]
    ];

    $form['access_token_endpoint'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#id' => 'access-token-endpoint',
      '#title' => t('Token Endpoint'),
      '#default_value' => $entity->getAccessTokenEndpoint(),
      '#states' => [
        'invisible' => [
          ':input[name="grant_type"]' => ['value' => 'implicit'],
        ],
        'optional' => [
          ':input[name="grant_type"]' => array('value' => 'implicit'),
        ],
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['userinfo_endpoint'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#id' => 'userinfo-endpoint',
      '#default_value' => $entity->getUserinfoEndpoint(),
      '#title' => t('User Info Endpoint'),
      '#states' => [
        'visible' => [
          ':input[name = "login_protocol"]' => ['value' => 'OAuth'],
        ],
        'invisible' => [
          ':input[name = "client_app"]' => ['value' => 'Azure AD B2C'],
          ':input[name="login_protocol"]' => ['value' => 'OpenID'],
        ],
        'optional' => [
          [':input[name = "login_protocol"]' => ['value' => 'OpenID']],
          [':input[name = "client_app"]' => ['value' => 'Azure AD B2C']],
        ],
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ]
    ];

    $form['group_info_endpoint'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#id' => 'group-info-endpoint',
      '#default_value' => $entity->getGroupInfoEndpoint(),
      '#title' => t('Group Info Endpoint'),
      '#description' => t('Fill this GroupInfo Endpoint field only if you want to fetch Groups from Azure AD'),
      '#states' => [
        'visible' => [
          ':input[name = "client_app"]' => ['value' => 'Azure AD'],
        ],
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ]
    ];

    $form['jkws_endpoint'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#default_value' => $entity->getJkwsEndpoint(),
      '#id' => 'jkws-endpoint',
      '#title' => t('JKWS URI'),
      '#states' => [
        'visible' => [
          [':input[name = "login_protocol"]' => ['value' => 'OpenID']],
          [':input[name = "client_app"]' => ['value' => 'Azure AD B2C']],
        ],
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $authentication_method_options = $form_state->get('authentication_method_options') ?? [];
    !empty($authentication_method_options) || $this->call([MoUnoReadGrants::class, 'getAuthenticationMethodOptions'], ['options' => &$authentication_method_options]);
    $form_state->set('authentication_method_options', $authentication_method_options);

    $grant_type = $form_state->getValue('grant_type') ?? $entity->getGrantType() ?? '';
    if ($grant_type !== 'authorization_code_with_pkce') {
      unset($authentication_method_options['none']);
    }

    $form['authentication_method_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#open' => true,
      "#id" => "authentication-method-frame",
      '#title' => $this->t('Send Client-Id and Secret in'),
      '#states' => [
        'invisible' => [
          ':input[name = "grant_type"]' => ['value' => 'implicit']
        ],
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
      '#attributes' => [
        "style" => ['max-width: 504px;'],
      ]
    ];

    $form['authentication_method_frame']['authentication_method_default_value'] = [
      '#type' => 'hidden',
      '#value' => $entity->getAuthenticationMethod(),
      '#attributes' => ['id' => 'authentication-method-default-value']
    ];

    $form['authentication_method_frame']['authentication_method'] = [
      '#type' => 'radios',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#id' => 'radio-buttons-wrapper',
      '#options' => $authentication_method_options,
      '#states' => [
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['settings_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#open' => true,
      '#title' => $this->t('Login Link Settings'),
      '#attributes' => [
        "style" => ['max-width: 504px;'],
      ],
      '#states' => [
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['settings_frame']['login_link_text'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#title' => $this->t('Login link text'),
      '#id' => 'login-link-text',
      '#default_value' => $entity->getLoginLinkText(),
      '#disabled' => $disabled,
      '#description' => $this->t('This text will be displayed as the login link on the user login page.'),
      '#states' => [
        'invisible' => [
          [':input[name="grant_type"]' => ['value' => 'password']],
          [':input[name="enable_login_with_oauth"]' => ['checked' => false]],
          [':input[name="display_login_link"]' => ['checked' => false]]
        ],
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['settings_frame']['enable_login_with_oauth'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#default_value' => $entity->getEnableLoginWithOauth(),
      '#title' => $this->t('Enable Login with OAuth'),
      '#states' => [
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['settings_frame']['display_login_link'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#disabled' => $disabled,
      '#title' => $this->t('Display login link on the login page'),
      '#default_value' => $entity->getDisplayLoginLink(),
      '#states' => [
        'invisible' => [
          [':input[name="grant_type"]' => ['value' => 'password']],
          [':input[name="enable_login_with_oauth"]' => ['checked' => false]]
        ],
        'disabled' => [
          [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
        ]
      ]
    ];

    //todo need to cross verify this part below it was removed here it was added can we do anything with this??
    $parent_actions = parent::actions($form, $form_state);
    $form['submit'] = $parent_actions['submit'];

    // add additional properties to the base form submit operation
    $form['submit']['#value'] = $this->t('Save Configuration');
    $form['submit']['#attributes']['class'] = ['button', 'button--primary'];
    $form['submit']['#states']['disabled'][] = [':input[name="client_conf_is_enabled"]' => ['value' => '0']];

    if (!$entity->isNew()) {
      $form['test_config']['data'] = [
        '#type' => 'button',
        '#value' => $this->t('Perform Test Configuration'),
        '#attributes' => [
          'class' => ['button', 'mo-test-config'],
          'url' => Url::fromRoute(
            $entity->getGrantType() == 'password' ? 'mo_oauth.password_grant.test_config' : 'mo_oauth.authorization_request',
            ['mo_client_config' => $entity->id(), 'test_sso' => true],
            ['absolute' => TRUE]
          )->toString(),
        ],
        '#states' => [
          'disabled' => [
            [':input[name="client_conf_is_enabled"]' => ['value' => '0']],
          ]
        ]
      ];

      // Add some additional properties to the base form delete action
      $form['delete'] = $parent_actions['delete'];
      $form['delete']['#attributes']['class'] = ['action-link', 'action-link--danger', 'use-ajax', 'action-link--icon-trash'];
      if ($disabled) {
        $form['delete']['#access'] = false;
      }
    }

    // The below part is for adding the additional layer of tutorials to the form page
    // if you don't want the tutorial part just uncomment the below line...
    // return self::buildMOLayout($form);

    $result['mo_outer_container'] = [
      "#type" => "container",
      "#attributes" => ['class' => ['mo-outer-container']]
    ];

    $this->modifyFormTitleOnTier($form);
    $result['mo_outer_container'] += self::buildMOLayout($form);

    $result['mo_outer_container']['tutorial_frame'] = self::getTutorialLayout();

    $this->call(
      [MoUnoTutorial::class, 'buildVideoTutorialFrame'],
      ['result' => &$result['mo_outer_container']['tutorial_frame']]
    );

    // This part holds the weather the tutorial should be shown or not in new configuration
    if ($entity->isNew()) {
      $result['mo_outer_container']['tutorial_frame']['#attributes']['style'] = ['display: none;'];
      $result['mo_outer_container']['tutorial_frame']['#id'] = 'mo-tutorial-frame';
      $result['mo_outer_container']['tutorial_frame']['video_plyr_frame']['#open'] = TRUE;
    }

    $this->getAdaptiveLoginGuide($result['mo_outer_container']['tutorial_frame'], $entity);

    MoSupport::buildSupportIcon($result);
    return $result;
  }

  /**
   * Ajax callback method to update the radio options
   *
   * @param array $form form array
   * @param FormStateInterface $form_state form state object
   *
   * @return array
   */
  public function updateRadioButtons(array &$form, FormStateInterface $form_state): array
  {
    return $form['mo_outer_container']['mo_outline']['authentication_method_frame'];
  }

  /**
   * This function is to update or build the callback-url
   *
   * @param array $form Form field array
   * @param FormStateInterface $form_state Form state interface
   *
   * It processes the data and finally returns the Ajax response that updates the particular column
   * @return AjaxResponse
   */
  public function changeCallback(array &$form, FormStateInterface $form_state): AjaxResponse
  {
    $ajax_response = new AjaxResponse();

    $app_name = trim($form_state->getValue('app_name', ''));
    if (!empty($app_name)) {
      $app_name = MoUtilities::getModifiedAppName($app_name);
      if (\Drupal::service('miniorange_oauth_client.uno_license_tier_manager')->getLicenseVersion() == MoModuleConstant::MO_ENTERPRISE_VERSION) {
        $callback_app_id = $app_name;
      } else {
        $callback_app_id = 'idp';
      }
      $ajax_response->addCommand(
        new InvokeCommand('#callback-url', 'val', [MoUtilities::getCallBackUrl($callback_app_id)])
      );
      $ajax_response->addCommand(
        new InvokeCommand('#login-link-text', 'val', ['Login with ' . $app_name])
      );
    }
    return $ajax_response;
  }

  /**
   * Here we are overwriting the base action method since we don't want the button to be outside the form.
   *
   * @param array $form form elements array
   * @param FormStateInterface $form_state form state object
   *
   * @return array
   */
  public function actions(array $form, FormStateInterface $form_state): array
  {
    $actions = parent::actions($form, $form_state);
    unset($actions['submit']);
    unset($actions['delete']);
    return $actions ?? [];
  }

  /**
   * {@inheritdoc}
   * @throws EntityMalformedException
   */
  public function save(array $form, FormStateInterface $form_state): int
  {
    /** @var MoClientConfiguration $entity */
    $entity = $this->entity;
    $is_new = $entity->isNew();
    if ($is_new) {
      if (\Drupal::service('miniorange_oauth_client.uno_license_tier_manager')->getLicenseVersion() == MoModuleConstant::MO_ENTERPRISE_VERSION) {
        $entity->set('id', MoUtilities::getModifiedAppName($form_state->getValue('app_name')));
      } else {
        $entity->set('id', 'idp');
      }
    }
    $client_secret = $form_state->getValue('client_secret');
    $entity->setClientSecret(
      preg_match('/\*\*/', $client_secret) ?
        MoClientConfiguration::load($entity->id())->getClientSecret() :
        $client_secret
    );
    try {
      $result = parent::save($form, $form_state);
      \Drupal::messenger()->addMessage("Application configured Successfully. Please click on the 'Perform Test Configuration' button to proceed with mapping.");
    } catch (\Throwable $exception) {
      MoLogger::error($exception->getMessage());
      \Drupal::messenger()->addError($exception->getMessage());
    }

    $is_new ?
      $form_state->setRedirectUrl($this->entity->toUrl('collection')) :
      $form_state->setRedirect(
        'mo_configuration.edit',
        ['mo_client_config' => $entity->id(), 'tab' => 'client_config']
      );
    return $result ?? 0;
  }

  /**
   * This is an AJAX callback function that helps in auto-populate the default values for the scope and endpoints
   *
   * @param array $form form element array
   * @param FormStateInterface $form_state form state object
   *
   * @return AjaxResponse
   * @throws \Exception
   */
  public function updateEndpoints(array $form, FormStateInterface $form_state): AjaxResponse
  {
    $selected_value = $form_state->getValue('client_app');
    $response = new AjaxResponse();

    $data = MoFallBackSupport::idpInfoHelper('clientName', $selected_value);

    $response->addCommand(
      new InvokeCommand('#scope', 'val', [$data['scope'] ?? ''])
    );
    $response->addCommand(
      new InvokeCommand('#authorize-endpoint', 'val', [$data['authorizationEndpoint'] ?? ''])
    );
    $response->addCommand(
      new InvokeCommand('#access-token-endpoint', 'val', [$data['tokenEndpoint'] ?? ''])
    );
    $response->addCommand(
      new InvokeCommand('#userinfo-endpoint', 'val', [$data['userInfoEndpoint'] ?? ''])
    );
    $response->addCommand(
      new InvokeCommand('#group-info-endpoint', 'val', [$data['groupInfoEndpoint'] ?? ''])
    );
    $response->addCommand(
      new InvokeCommand('#jwks-endpoint', 'val', [$data['jwksEndpoint'] ?? ''])
    );
    $guide_url = $this->buildSetupGuideLink($data['setupGuideLink']);
    $response->addCommand(
      new InvokeCommand('#setup-guide-url', 'val', [$data['setupGuideLink'] ?? ''])
    );
    $response->addCommand(new HtmlCommand('#idp-setup-guide', $guide_url));

    $response->addCommand(new CssCommand('#mo-tutorial-frame', ['display' => 'block']));

    if ($selected_value == 'Azure AD B2C') {
      $response->addCommand(
        new InvokeCommand('input[name="login_protocol"][value="OpenID"]', 'prop', ['checked', true])
      );
    }

    $this->call([MoUnoTutorial::class, 'ajaxEnableVideoTutorialVisibility'], ['response' => &$response, 'entity' => $this->entity]);

    return $response;
  }

  /**
   * This function helps in building setup guide link to show in the form
   *
   * @param string|null $link setup guide link
   *
   * @return MarkupInterface|null
   */
  public function buildSetupGuideLink(?string $link): ?MarkupInterface
  {
    if (!empty($link)) {
      $url = Link::fromTextAndUrl(
        "Setup Guide",
        Url::fromUri(
          $link,
          [
            'attributes' =>
              [
                'target' => '_blank',
                'rel' => 'noopener noreferrer',
              ]
          ]
        )
      )->toRenderable();
      return \Drupal::service('renderer')->renderPlain($url);
    }
    return null;
  }

  public function getAdaptiveLoginGuide(array &$form, $entity): void
  {
    if (!$entity->isNew()) {
      $form['add_login_link_in_multisite'] = [
        '#type' => 'fieldset',
        '#open' => true,
        '#title' => $this->t('Want to add login link to different pages in your Drupal site?'),
      ];
      $form['add_login_link_in_multisite']['intro'] = [
        '#markup' => '<p>' . $this->t(
          'After completing your configurations, by default you will see a login link on your Drupal site\'s login page.
          However, if you want to add the login link somewhere else, please follow the steps below:'
        ) . '</p>',
      ];

      $form['add_login_link_in_multisite']['steps_set_1'] = [
        '#theme' => 'item_list',
        '#items' => [
          $this->t('Go to @structure → @blocks', [
            '@structure' => Markup::create('<strong>Structure</strong>'),
            '@blocks' => Markup::create('<strong>Blocks</strong>'),
          ]),
          $this->t('Click on @add_block', [
            '@add_block' => Markup::create('<strong>Add block</strong>'),
          ]),
          $this->t('Enter @title and @desc', [
            '@title' => Markup::create('<strong>Block Title</strong>'),
            '@desc' => Markup::create('<strong>Block description</strong>'),
          ]),
          $this->t('Under the @body, add the following URL to create a login link:', [
            '@body' => Markup::create('<strong>Block body</strong>'),
          ]),
        ]
      ];

      $form['add_login_link_in_multisite']['url'] = [
        '#markup' => Markup::create(
          '
            <strong><em>&lt;a href="' .
          Url::fromRoute('mo_oauth.authorization_request',['mo_client_config' => $entity->id()], ['absolute' => TRUE])->toString() .
          '"&gt;Login&lt;/a&gt;</em></strong>'
        ),
      ];

      $form['add_login_link_in_multisite']['steps_set_2'] = [
        '#theme' => 'item_list',
        '#items' => [
          $this->t('From the text format dropdown, select either @filtered or @full', [
            '@filtered' => Markup::create('<strong>Filtered HTML</strong>'),
            '@full' => Markup::create('<strong>Full HTML</strong>'),
          ]),
          $this->t('Under @region_settings, select where you want the login link to appear', [
            '@region_settings' => Markup::create('<strong>REGION SETTINGS</strong>'),
          ]),
          $this->t('Click the @save to save your settings', [
            '@save' => Markup::create('<strong>SAVE block</strong>'),
          ]),
        ],
      ];
    }
  }
}
