<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\miniorange_oauth_client\Entity\MoClientConfiguration;
use Drupal\miniorange_oauth_client\Entity\MoClientSettings;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoDTO\MoModuleSettings;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoTestConfigAttributeListBuilder;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoTutorial;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoLibrary\MoLogger;
use Drupal\miniorange_oauth_client\MoService\MoAjaxTables;
use Drupal\miniorange_oauth_client\MoService\MoLayoutBuilder;
use Drupal\Core\Url;

/**
 * MiniOrange Client Settings Entity form.
 */
final class MoClientSettingsForm extends EntityForm
{

  use MoGhostInvoker;
  use MoLayoutBuilder;

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array
  {
    $form = parent::buildForm($form, $form_state);
    $disabled = \Drupal::service('miniorange_oauth_client.uno_module_subscription_library')->getModuleStatus($form);

    /** @var MoClientSettings $entity */
    $entity = $this->getEntity();

    $form['entity_key'] = [
      '#type' => 'hidden',
      '#value' => $entity->id(),
    ];

    $form['client_settings_is_enabled'] = [
      '#type' => 'hidden',
      '#value' => (int)!$disabled
    ];

    $this->buildUserCreationFunctionality($form, $entity, $disabled);

    $this->buildRedirectUserAfterLoginAndLogoutFunctionality($form, $entity, $disabled);

    $this->buildAttributeOrRoleBasedRestrictionFunctionality($form, $entity);

    $this->buildSLOFunctionality($form, $entity, $disabled, $form_state);

    $this->buildSessionManagementFunctionality($form, $entity);

    // add additional properties to the base form submit operation
    if (!$entity->isNew()) {
      $form['actions']['delete'] = [
        '#type' => 'submit',
        '#tier' => MoModuleConstant::MO_STANDARD_VERSION_TIER,
        '#value' => $this->t('Reset'),
        "#weight" => 2,
        '#attributes' => ['class' => ['button', 'button--danger']],
        '#submit' => ['::delete'],
        '#states' => [
          'disabled' => [
            [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
          ]
        ],
      ];
    }

    $form['actions']['submit']['#attributes']['class'] = ['button', 'button--primary'];
    $form['actions']['submit']['#weight'] = 1;
    $form['actions']['submit']['#states']['disabled'][] = [':input[name="client_settings_is_enabled"]' => ['value' => '0']];
    $form['actions']['submit']['#tier'] = MoModuleConstant::MO_STANDARD_VERSION_TIER;

    $result['mo_outer_container'] = [
      "#type" => "container",
      "#attributes" => ['class' => ['mo-outer-container']]
    ];

    $this->modifyFormTitleOnTier($form);
    $result['mo_outer_container'] += $this->buildMOLayout($form);

    $result['mo_outer_container']['tutorial_frame'] = self::getTutorialLayout();

    $this->call(
      [MoUnoTutorial::class, 'buildVideoTutorialFrame'],
      ['result' => &$result['mo_outer_container']['tutorial_frame']]
    );
    $this->call(
      [MoUnoTestConfigAttributeListBuilder::class, 'buildTestConfigurationFrame'],
      ['result' => &$result['mo_outer_container']['tutorial_frame'], 'entity_id' => $entity->id()]
    );

    return $result;
  }

  public function buildUserCreationFunctionality(array &$form, MoClientSettings $entity, bool $disabled): void
  {

    $form['auto_create_user'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_STANDARD_VERSION_TIER,
      '#default_value' => $entity->getAutoCreateUser(),
      '#title' => t('Enable auto creation of users'),
      '#description' => t('Users will be auto-created in Drupal on first SSO login (if they don\'t already have an account).'),
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['redirect_unregister'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Redirect URL for Unregistered Users'),
      '#attributes' => ['placeholder' => 'Enter complete URL'],
      '#default_value' => $entity->getRedirectUnregister(),
      '#description' => t('If empty, the un-registered users will be redirected to same page from where they initiated the login.'),
      '#states' => [
        'visible' => [
          [
            ':input[name = "auto_create_user"]' => ['checked' => False],
          ],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['disable_new_user'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#disabled' => $disabled,
      '#default_value' => $entity->getDisableNewUser(),
      '#title' => t('Create new users in Blocked Status'),
      '#states' => [
        'visible' => [
          [':input[name = "auto_create_user"]' => ['checked' => true]],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['disabled_user_redirect_url'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Redirect URL for Blocked Users'),
      '#attributes' => ['placeholder' => 'Enter complete URL'],
      '#default_value' => $entity->getDisabledUserRedirectUrl(),
      '#description' => t('If empty, users will be redirected to same page from where they initiated the login.'),
      '#states' => [
        'visible' => [
          ':input[name="disable_new_user"]' => ['checked' => true],
          'and',
          ':input[name="auto_create_user"]' => ['checked' => true],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];
  }

  public function buildRedirectUserAfterLoginAndLogoutFunctionality(array &$form, MoClientSettings $entity, bool $disabled): void
  {
    $form['redirect_url_login_logout_frame'] = [
      '#type' => 'details',
      '#title' => t('Redirection after SSO login and logout'),
      '#open' => true,
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['redirect_url_login_logout_frame']['anonymous_user_redirect_url'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Redirect URL for Anonymous User'),
      '#attributes' => ['placeholder' => 'Enter complete URL'],
      '#default_value' => $entity->getAnonymousUserRedirectUrl(),
      '#description' => t('Users will be redirected to this URL if the SSO login fails or is canceled.'),
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['redirect_url_login_logout_frame']['login_redirect_url'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_STANDARD_VERSION_TIER,
      '#title' => t('Redirect URL after login'),
      '#default_value' => $entity->getLoginRedirectUrl(),
      '#attributes' => ['placeholder' => 'Enter complete URL'],
      '#description' => t('Users will be taken to this URL after performing SSO.'),
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['redirect_url_login_logout_frame']['same_page_after_logout'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#default_value' => $entity->getSamePageAfterLogout(),
      '#title' => t('Keep users on the same page after logout'),
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['redirect_url_login_logout_frame']['logout_redirect_url'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_STANDARD_VERSION_TIER,
      '#disabled' => $disabled,
      '#title' => t('Redirect URL after logout'),
      '#default_value' => $entity->getLogoutRedirectUrl(),
      '#attributes' => ['placeholder' => 'Enter complete URL'],
      '#description' => t('Users will be taken to this URL after performing logout.'),
      '#states' => [
        'disabled' => [
          [':input[name = "same_page_after_logout"]' => ['checked' => true]],
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];
  }

  public function buildAttributeOrRoleBasedRestrictionFunctionality(array &$form, MoClientSettings $entity): void
  {
    // Attribute/Role Based Restriction
    $form['attribute_role_base_restriction'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Attribute Based Restriction'),
      '#open' => true,
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['attribute_role_base_restriction']['role_based_restriction'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Enable Attribute Based Restriction'),
      '#default_value' => $entity->getRoleBasedRestriction() ?? 0,
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['attribute_role_base_restriction']['role_restriction_attrname'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Attribute Name'),
      '#default_value' => $entity->getRoleRestrictionAttrName() ?? '',
      '#description' => '<b>'. t('Note:'). '</b> ' .t('Please fill this'). ' <b>' . t('Attribute Name field') . '</b> '.
        t('to enable Attribute based restriction. Users will be restricted based on this attribute coming from OAuth Provider.'),
      '#states' => [
        'required' => [
          ':input[name="role_based_restriction"]' => ['checked' => true],],
        'visible' => [
          ':input[name="role_based_restriction"]' => ['checked' => TRUE],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['attribute_role_base_restriction']['restriction_type_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Select the restriction type'),
      '#open' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="role_based_restriction"]' => ['checked' => TRUE],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ]
    ];

    $form['attribute_role_base_restriction']['restriction_type_frame']['typeof_role_restriction'] = [
      '#type' => 'radios',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#options' => ['whitelist' => 'Allow below Attributes', 'blacklist' => 'Restrict below Attributes'],
      '#default_value' => $entity->getTypeofRoleRestriction() ?? 'whitelist',
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['attribute_role_base_restriction']['list_of_restricted_roles'] = [
      '#type' => 'textarea',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Enter the attribute values'),
      '#placeholder' => t('Enter the list of semicolon (;) separated Attribute values coming from OAuth Provider to validate.'),
      '#description' => t('Enter the list of semicolon (;) separated Attribute values coming from OAuth Provider to validate.'),
      '#default_value' => $entity->getListOfRestrictedRoles() ?? '',
      '#states' => [
        'required' => [
          ':input[name="role_based_restriction"]' => ['checked' => true],
        ],
        'visible' => [
          ':input[name="role_based_restriction"]' => ['checked' => TRUE],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['attribute_role_base_restriction']['redirect_url_for_role_restriction'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#maxlength' => 512,
      '#title' => t('Redirect URL for Restricted Users'),
      '#description' => t('If empty, users will be redirected to the login page (<i>' . Url::fromRoute('user.login')->toString() . '</i>).'),
      '#default_value' => $entity->getRedirectUrlForRoleRestriction() ?? '',
      '#attributes' => [
        'placeholder' => 'Enter the URL on which you wish to Redirect the restricted users',
      ],
      '#states' => [
        'visible' => [
          ':input[name="role_based_restriction"]' => ['checked' => TRUE],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ]
    ];
  }

  public function buildSLOFunctionality(array &$form, MoClientSettings $entity, bool $disabled, FormStateInterface $form_state): void
  {
    $form['logout_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('Log out Settings'),
      '#open' => true,
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $module_settings = MoModuleSettings::readMe();
    $token_storage_info = $module_settings->getTokenStorageKeys();

    $mod_settings_page_url = Link::fromTextAndUrl(
      t('Module Settings'),
      Url::fromRoute('mo_module.settings')
    )->toString();

    // slo functionality
    $form['logout_frame']['slo'] = [
      '#type' => 'fieldset',
      '#title' => t('Single Log Out'),
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      // this piece is needed if you want to take disabled functionality into fieldset based on name.
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['logout_frame']['slo']['enable_slo'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('Enable single log out'),
      '#default_value' => $entity->getEnableSlo(),
      '#description' => t('Automatically destroys session in your IDP on logout.'),
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['logout_frame']['slo']['single_logout_url'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('IDP End Session Endpoint'),
      '#default_value' => $entity->getSingleLogoutUrl(),
      '#attributes' => ['placeholder' => 'Enter complete URL'],
      '#description' => t('URL to log out user from the Identity Provider when they log out from the Drupal site.'),
      '#states' => [
        'invisible' => [
          [':input[name="enable_slo"]' => ['checked' => FALSE]],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['logout_frame']['slo']['idp_logout_redirect_uri_param'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('Redirect URI Param Name of Identity Provider'),
      '#default_value' => $entity->getIdpLogoutRedirectUriParam(),
      '#states' => [
        'invisible' => [
          [':input[name="single_logout_url"]' => ['value' => '']],
          [':input[name="enable_slo"]' => ['checked' => FALSE]],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $is_store_id_tok_in_ses_enabled = $module_settings->getStoreTokenInSession() && $token_storage_info['id_token']['is_include'] == 1;

    $form['logout_frame']['slo']['store_id_tok_in_ses_enabled'] = [
      '#type' => 'hidden',
      '#value' => intval($is_store_id_tok_in_ses_enabled)
    ];

    $form['logout_frame']['slo']['inc_id_tok_in_slo_cont'] = [
      '#type' => 'container',
      '#states' => [
        'invisible' => [
          [':input[name="single_logout_url"]' => ['value' => '']],
          [':input[name="enable_slo"]' => ['checked' => FALSE]],
        ],
      ]
    ];

    $form['logout_frame']['slo']['inc_id_tok_in_slo_cont']['inc_id_tok_in_slo_error'] = [
      '#type' => 'markup',
      '#markup' => '<div class="mo-register-message">' .
        t(
          'To avail the below feature, goto @module_settings tab scroll down to "Token Storage" then enable "Store Token in User Session" and also enable the "Id Token" checkbox in the displayed table.',
          ['@module_settings' => $mod_settings_page_url]
        ) .
        '</div>',
      '#access' => !$is_store_id_tok_in_ses_enabled,
    ];

    $form['logout_frame']['slo']['inc_id_tok_in_slo_cont']['include_id_token_in_slo'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('Include ID Token in End Session Endpoint'),
      '#default_value' => $entity->getIncludeIdTokenInSlo(),
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
          [':input[name="store_id_tok_in_ses_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['logout_frame']['slo']['inc_id_tok_in_slo_cont']['idp_logout_token_param'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('Name the parameter carrying the Id-Token'),
      '#default_value' => $entity->getIdpLogoutTokenParam(),
      '#states' => [
        'invisible' => [
          [':input[name="include_id_token_in_slo"]' => ['checked' => FALSE]],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
          [':input[name="store_id_tok_in_ses_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    // token revoke functionality
    $form['logout_frame']['token_revoke'] = [
      '#type' => 'fieldset',
      '#title' => t('Revoke Token'),
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      // this piece is needed if you want to take disabled functionality into fieldset based on name.
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $is_store_ref_tok_in_ses_enabled = $module_settings->getStoreTokenInSession() && $token_storage_info['refresh_token']['is_include'] == 1;
    $form['logout_frame']['token_revoke']['store_ref_tok_in_ses_enabled'] = [
      '#type' => 'hidden',
      '#value' => intval($is_store_ref_tok_in_ses_enabled)
    ];

    $form['logout_frame']['token_revoke']['token_revoke_disabled_error'] = [
      '#type' => 'markup',
      '#markup' => '<div class="mo-register-message">' .
        t(
          'To avail this feature, goto %module_settings tab scroll down to "Token Storage" then enable "Store Token in User Session" and also enable the "Refresh Token" checkbox in the displayed table.',
          ['%module_settings' => $mod_settings_page_url]
        ) .
        '</div>',
      '#access' => !$is_store_ref_tok_in_ses_enabled,
    ];

    $form['logout_frame']['token_revoke']['revoke_user_token_from_idp'] = [
      '#type' => 'checkbox',
      '#title' => t('Revoke user tokens after logout'),
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#default_value' => $entity->getRevokeUserTokenFromIdp(),
      '#description' => t('The Refresh Token provided by the IDP can be revoked on logout if it is stored in the user session.'),
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
          [':input[name="store_ref_tok_in_ses_enabled"]' => ['value' => '0']],
        ]
      ]
    ];

    $form['logout_frame']['token_revoke']['token_revoke_url'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('IDP Token Revoke Endpoint'),
      '#attributes' => ['placeholder' => 'Enter complete URL'],
      '#default_value' => $entity->getTokenRevokeUrl(),
      '#states' => [
        'required' => [
          ':input[name="revoke_user_token_from_idp"]' => ['checked' => true],],
        'visible' => [
          ':input[name="revoke_user_token_from_idp"]' => ['checked' => TRUE],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
          [':input[name="store_ref_tok_in_ses_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['logout_frame']['token_revoke']['token_revoke_param_name'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Name the parameter carrying the token'),
      '#default_value' => $entity->getTokenRevokeParamName() ?? '',
      '#description' => t('This field helps you in customizing the parameter name to pass the refresh token to the IDP.'),
      '#states' => [
        'required' => [
          ':input[name="revoke_user_token_from_idp"]' => ['checked' => true],],
        'invisible' => [
          [':input[name="revoke_user_token_from_idp"]' => ['checked' => FALSE]],
          [':input[name="token_revoke_url"]' => ['value' => '']],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
          [':input[name="store_ref_tok_in_ses_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $additional_params_header = function () {
      return [
        [
          'data' => t('Param Name'),
          'width' => '45%',
        ],
        [
          'data' => t('Param Value'),
          'width' => '45%',
        ],
        [
          'data' => t('Action'),
          'width' => '10%',
        ],
      ];
    };

    $additional_params_fields = function () {
      return [
        'param_name' => [
          'type' => 'textfield',
        ],
        'param_value' => [
          'type' => 'textfield',
        ],
        'delete_button_role' => [
          'type' => 'submit',
          'submit' => [$this, 'removeTokenRevokeAdditionalParamRow'],
          'callback' => [$this, 'tokenRevokeAdditionalParamAjaxCallback'],
          'wrapper' => 'token-revoke-additional-params-mapping-table',
        ],
      ];
    };

    $additional_param_rows = $entity->getTokenRevokeAdditionalParams();

    $unique_id_array = MoAjaxTables::getUniqueID($form_state->get('token_revoke_additional_params_mapping_uid'), $additional_param_rows);
    $form_state->set('token_revoke_additional_params_mapping_uid', $unique_id_array);

    $form['logout_frame']['token_revoke']['token_revoke_additional_params_frame'] = [
      '#type' => 'item',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Additional Params To Pass on Token Revocation'),
      '#states' => [
        'required' => [
          ':input[name="revoke_user_token_from_idp"]' => ['checked' => true],],
        'invisible' => [
          [':input[name="revoke_user_token_from_idp"]' => ['checked' => FALSE]],
          [':input[name="token_revoke_url"]' => ['value' => '']],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
          [':input[name="store_ref_tok_in_ses_enabled"]' => ['value' => '0']],
        ]
      ],
      'mapping' => [
        'token_revoke_additional_params' => MoAjaxTables::generateTables(
          'token-revoke-additional-params-mapping-table',
          $additional_params_fields(),
          $unique_id_array,
          $additional_param_rows,
          $additional_params_header()
        ),
        'add_row' => MoAjaxTables::generateAddButton(
          'Add',
          [$this, 'addTokenRevokeAdditionalParamRow'],
          [$this, 'tokenRevokeAdditionalParamAjaxCallback'],
          'token-revoke-additional-params-mapping-table',
          '',
          $disabled || !in_array(
            $this->call([MoUnoLicenseTierManager::class, 'getLicenseVersion']),
            MoModuleConstant::MO_ENTERPRISE_VERSION_TIER
          )
        )
      ]
    ];
  }

  public function submitForm(&$form, FormStateInterface $form_state): void
  {
    $form_state->setRedirect(
      'mo_configuration.edit', ['mo_client_config' => $this->entity->id(), 'tab' => 'client_setting']
    );
    parent::submitForm($form, $form_state);
    \Drupal::messenger()->addStatus(t('Client settings saved successfully.'));
  }

  public function delete(array $form, FormStateInterface $form_state): void
  {
    try {
      $this->entity->delete();
      \Drupal::messenger()->addStatus(t('Client settings have been deleted successfully.'));
    } catch (\Exception $exception) {
      MoLogger::error($exception->getMessage());
      \Drupal::messenger()->addError(t('Something went wrong in saving your client settings.'));
    }
  }

  private function buildSessionManagementFunctionality(array &$form, MoClientSettings $entity): void
  {
    $form['session_management_frame'] = [
      '#type' => 'details',
      '#open' => true,
      '#title' => 'Session Management',
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['session_management_frame']['tie_token_expiry_to_usr_session'] = [
      '#type' => 'checkbox',
      '#title' => t('Synchronize Drupal user session lifetime with IdP token expiry'),
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#default_value' => $entity->getTieTokenExpiryToUsrSession(),
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $config_entity = MoClientConfiguration::load($entity->id());
    $user_token_keys = array_merge(['' => "-- Select --"], $config_entity->getServerTokenAttributes());

    $form['session_management_frame']['idp_token_expiry_param'] = [
      '#type' => 'select',
      '#options' => $user_token_keys,
      '#title' => t('Select token expiry parameter'),
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#default_value' => $entity->getIdpTokenExpiryParam(),
      '#description' => "Select the parameter that provides the token expiry time in the IDPs token response.",
      '#states' => [
        'invisible' => [
          ':input[name="tie_token_expiry_to_usr_session"]' => ['checked' => FALSE],
        ],
        'required' => [
          ':input[name="tie_token_expiry_to_usr_session"]' => ['checked' => TRUE],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $options = [
      'logout' => t('Terminate Drupal user session upon expiry.'),
      'renew' => t('Automatically renew token using the refresh token.'),
    ];

    $form['session_management_frame']['action_after_token_expiry'] = [
      '#type' => 'radios',
      '#options' => $options,
      '#title' => t('What to do after token expiry?'),
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#default_value' => $entity->getActionAfterTokenExpiry() ?? 'logout',
      '#states' => [
        'invisible' => [
          ':input[name="tie_token_expiry_to_usr_session"]' => ['checked' => FALSE],
        ],
        'required' => [
          ':input[name="tie_token_expiry_to_usr_session"]' => ['checked' => TRUE],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];
  }

  public function tokenRevokeAdditionalParamAjaxCallback($form)
  {
    return $form['mo_outer_container']['mo_outline']['logout_frame']['token_revoke']['token_revoke_additional_params_frame']['mapping']['token_revoke_additional_params'];
  }

  public function addTokenRevokeAdditionalParamRow(array &$form, FormStateInterface $form_state): void
  {
    $add_button = ($form_state->getTriggeringElement()['#name']);
    $rows = $form_state->getValue('total_rows_' . $add_button);
    $unique_array_id = $form_state->get('token_revoke_additional_params_mapping_uid');
    for ($count = 1; $count <= $rows; $count++) {
      $uuid_service = \Drupal::service('uuid');
      $uuid = $uuid_service->generate();
      $unique_array_id[] = $uuid;
    }
    $form_state->set('token_revoke_additional_params_mapping_uid', $unique_array_id);
    $form_state->setRebuild();
  }

  public function removeTokenRevokeAdditionalParamRow(array &$form, FormStateInterface $form_state): void
  {
    $id = $form_state->getTriggeringElement()['#name'];
    $var_value = 'token_revoke_additional_params_mapping_uid';
    $unique_id = $form_state->get($var_value);
    $unique_id = array_diff($unique_id, [$id]);
    if (empty($unique_id)) {
      $uuid_service = \Drupal::service('uuid');
      $unique_id[0] = $uuid_service->generate();
    }
    $form_state->set($var_value, $unique_id);
    $form_state->setRebuild();

  }
}
