<?php

namespace Drupal\miniorange_oauth_client\Form;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Url;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoHelper\MoUtilities;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoLibrary\MoLogger;

/**
 * Customer support request form.
 */
class MoCustomerSupport extends FormBase
{
  use MoGhostInvoker;

  public function getFormId()
  {
    return 'mo_customer_request_form';
  }

  public function buildForm(array $form, FormStateInterface $form_state, $params = [])
  {
    $form['#prefix'] = '<div id="modal_support_form">';
    $form['#suffix'] = '</div>';
    $form_state->set('mo_form_type', $params['form_type'] ?? '');
    $config = \Drupal::service('miniorange_oauth_client.uno_module_license_entity')->readMe();
    $email = $config?->getCustomerEmail() ?? \Drupal::config('system.site')->get('mail');
    $mobile = $config?->getCustomerPhoneNo() ?? '';


    $form['email_address'] = [
      '#type' => 'email',
      '#title' => $this->t('Email'),
      '#default_value' => $email,
      '#required' => TRUE,
      '#attributes' => [
        'placeholder' => $this->t('Enter valid email'),
        'style' => 'width:99%;margin-bottom:1%;',
      ],
    ];

    $form['support_number'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Phone Number'),
      '#default_value' => $mobile,
      '#attributes' => [
        'placeholder' => $this->t('Enter your Phone Number'),
        'style' => 'width:99%;margin-bottom:1%;',
      ],
    ];

    if ($form_state->get('mo_form_type') == 'customer-support') {
      $form['support_method'] = [
        '#type' => 'select',
        '#title' => t('What are you looking for'),
        '#attributes' => ['style' => 'width:99%;margin-bottom:1%;'],
        '#options' => [
          'I need Technical Support' => t('I need Technical Support'),
          'I want to Schedule a Setup Call/Demo' => t('I want to Schedule a Setup Call/Demo'),
          'I have Sales enquiry' => t('I have Sales enquiry'),
          'I have a custom requirement' => t('I have a custom requirement'),
          'My reason is not listed here' => t('My reason is not listed here'),
        ],
      ];
      foreach (MoModuleConstant::ZONES as $key => $value) {
        $time_zones[$value] = $key;
      }

      $form['date_and_time'] = [
        '#type' => 'container',
        '#states' => [
          'visible' => [
            ':input[name = "support_method"]' => ['value' => 'I want to Schedule a Setup Call/Demo'],
          ],
        ],
      ];

      $form['date_and_time']['time_zone'] = [
        '#type' => 'select',
        '#title' => t('Select Timezone'),
        '#options' => $time_zones ?? [],
        '#default_value' => 'Etc/GMT',
      ];

      $form['date_and_time']['meeting_time'] = [
        '#type' => 'datetime',
        '#title' => t('Date and Time'),
        '#format' => '',
        '#default_value' => DrupalDateTime::createFromTimestamp(time()),
      ];
    }

    $form['support_query'] = [
      '#type' => 'textarea',
      '#required' => TRUE,
      '#title' => $this->t('Query \ Issue \ Use Case'),
      '#attributes' => [
        'placeholder' => $this->t('Describe your Message here!'),
        'style' => 'width:99%',
      ],
      '#rows' => 2
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['send'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
      '#attributes' => [
        'class' => [
          'use-ajax',
          'button--primary',
        ],
      ],
      '#ajax' => [
        'callback' => [$this, 'submitModalFormAjax'],
        'event' => 'click',
      ],
    ];

    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';

    return $form;
  }

  public function validateForm(array &$form, FormStateInterface $form_state): void
  {
    $form_state->clearErrors();
  }


  /**
   * AJAX callback for modal form submission.
   *
   * @param array $form
   *   The form array.
   * @param FormStateInterface $form_state
   *   The form state.
   *
   * @return AjaxResponse
   *   The AJAX response.
   */
  public function submitModalFormAjax(array &$form, FormStateInterface $form_state): AjaxResponse
  {
    $form_values = $form_state->getValues();
    $response = new AjaxResponse();

    $email = filter_var($form_values['email_address'], FILTER_SANITIZE_EMAIL);
    $phone = htmlspecialchars($form_values['support_number'] ?? '', ENT_QUOTES, 'UTF-8');
    $query = htmlspecialchars($form_values['support_query'] ?? '', ENT_QUOTES, 'UTF-8');

    if (empty($email) || empty($query) || $invalid_email = !filter_var($email, FILTER_VALIDATE_EMAIL)) {
      if ($invalid_email ?? false) {
        $response->addCommand(new MessageCommand('Please enter a valid email.', NULL, ['type' => 'error']));
      }
      if (empty($email)) {
        $response->addCommand(new MessageCommand('\'Email\' field is required!', NULL, ['type' => 'error']));
      }
      if (empty($query)) {
        $response->addCommand(new MessageCommand('\'Query \ Issue \ Use Case\' field is required!', NULL, ['type' => 'error']));
      }
    } else {
      $form_type = $form_state->get('mo_form_type');

      $result = $this->sendSupportQuery($email, $phone, $query, $form_type, $form_values);
      if ($result) {
        $type = 'status';
        if ($form_type == 'customer-support') {
          $message = 'We received your query. We will get back to you with in next 24 hours.';
        } elseif ($form_type == 'notify-us') {
          $message = 'Thank you for using our module, need assistance refer to our official documentation.';
        } elseif ($form_type == 'trial-request') {
          $message = 'Success! The trial request was successfully sent. We will send you an email including the steps to activate the trial shortly. Please check your inbox for further instructions.';
        } else {
          $message = 'Notified Successfully. We will get back to you shortly.';
        }
      } else {
        $message = 'There was a problem in notifying us. Please reach out to us at <a href="mailto:' . MoModuleConstant::SUPPORT_EMAIL . '">' . MoModuleConstant::SUPPORT_EMAIL . '</a>';
        $type = 'error';
      }
      $response->addCommand(new MessageCommand($this->t($message), NULL, ['type' => $type]));
    }
    $response->addCommand(new CloseModalDialogCommand());
    return $response;
  }

  public function submitForm(array &$form, FormStateInterface $form_state) {}

  /**
   * Send support query to MiniOrange support.
   *
   * @param string $email
   *   User email address.
   * @param string $phone
   *   User phone number.
   * @param string $query
   *   User query/message.
   *
   * @return bool
   *   TRUE if successful, FALSE otherwise.
   */
  private function sendSupportQuery(string $email, string $phone, string $query, string $form_type, array $form_state): bool
  {
    try {
      $modules_version = $this->call([MoUnoLicenseTierManager::class, 'getModuleVersion']);
      $php_version = phpversion();

      $headers = [
        'Content-Type' => 'application/json',
      ];
      $base_url = Url::fromUri('internal:/', ['absolute' => true])->toString();

      $support_method = $form_state['support_method'] ?? '';

      switch ($form_type) {
        case 'trial-request':
          $prefix = 'Trial request';
          break;
        case 'notify-us':
          $prefix = 'Usage notification';
          break;
        default:
          $get_prefix = function () use ($form_state, $support_method) {
            if ($support_method == 'I want to Schedule a Setup Call/Demo') {
              $prefix = 'Setup Meeting/Call request';
              $form_type = 'call-request';
            }
            return [$prefix ?? 'Support request', $form_type ?? 'customer-support'];
          };
          [$prefix, $form_type] = $get_prefix();
          break;
      }
      $subject = $prefix . ' for Drupal-' . \DRUPAL::VERSION . ' OAuth Client Module | ' . $modules_version . ' | ' . phpversion() . ' - ' . $email;

      $content = '<div >Hello, <br><br>Company :<a href="' . $base_url . '" target="_blank" >' . $base_url .
        '</a><br><br>Phone Number:' . $phone . '<br><br>Email:<a href="mailto:' . $email . '" target="_blank">' .
        $email . '</a>';
      if ($form_type == 'call-request') {
        $date_time = $form_state['meeting_time']->format('Y-m-d h:i a');
        $time_zone = $form_state['time_zone'];
        $content .= '<br><br> Timezone: <b>' . $time_zone . '</b><br><br> Date & Time: <b>' . $date_time . '</b>';
      } elseif ($form_type == 'customer-support') {
        $content .= '<br><br> Support needed for: <b>' . $support_method . '</b>';
      }

      $content .= '<br><br>' . $prefix . ' for:[DRUPAL ' . \Drupal::VERSION . ' OAuth Client | ' . $modules_version .
        ' | PHP ' . phpversion() . ' ] ' .
        $query .
        '</div>';

      $customerKey = "16555";
      $apikey = "fFd2XcvTGDemZvbw1bcUesNJWEqKbbUq";
      $currentTimeInMillis = MoUtilities::getOAuthTimestamp();
      $stringToHash = $customerKey . $currentTimeInMillis . $apikey;
      $hashValue = hash("sha512", $stringToHash);

      $headers += [
        'Customer-Key' => $customerKey,
        'Timestamp' => $currentTimeInMillis,
        'Authorization' => $hashValue,
      ];

      $fields = [
        'customerKey' => $customerKey,
        'sendEmail' => TRUE,
        'email' => [
          'customerKey' => $customerKey,
          'fromEmail' => $email,
          'fromName' => 'miniOrange',
          'toEmail' => MoModuleConstant::SUPPORT_EMAIL,
          'toName' => MoModuleConstant::SUPPORT_EMAIL,
          'subject' => $subject,
          'content' => $content,
        ],
      ];

      $url = MoModuleConstant::LICENSE_SERVER_BASE_URL . '/moas/api/notify/send';

      $field_string = json_encode($fields);

      $response = MoUtilities::callService($url, $field_string, $headers);

      return !empty($response);
    } catch (\Exception $e) {
      MoLogger::error('Support query failed: @message', ['@message' => $e->getMessage()]);
      return FALSE;
    }
  }

}
