<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InsertCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoDTO\MoModuleSettings;
use Drupal\miniorange_oauth_client\MoHelper\MoSupport;
use Drupal\miniorange_oauth_client\MoService\MoLayoutBuilder;

class MoLoggerSettingsForm extends ConfigFormBase
{
  use MoLayoutBuilder;

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string
  {
    return 'mo_module_log_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array
  {
    return ['miniorange_oauth_client.module_settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array
  {
    $form = parent::buildForm($form, $form_state);

    $form['markup_library'] = [
      '#attached' => [
        'library' => [
          "core/drupal.dialog.ajax",
        ]
      ],
    ];

    $form['message_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'ajax-message'],
    ];

    $entity = MoModuleSettings::readMe();

    $form['enable_logs'] = [
      '#type' => 'checkbox',
      '#title' => t('Enable logs'),
      '#default_value' => $entity->getEnableLogs(),
      '#description' => t('Enable logging for this module'),
    ];

    $form['log_severity'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Select the severity for logging'),
      '#options' => self::getLogSeverityMap(),
      '#default_value' => $entity->getLogSeverity(),
      '#description' => $this->t('Select the severity levels to enable.'),
      '#states' => [
        'visible' => [
          ':input[name="enable_logs"]' => ['checked' => TRUE],
          ],
      ]
    ];
    $form['actions']['export_logs'] = [
      '#type' => 'submit',
      '#value' => t('Export Logs'),
      '#name' => 'export_logs',
      '#attributes' => ['class' => ['button']],
      '#ajax' => [
        'callback' => '::exportLogs',
        'progress' => ['type' => 'throbber'],
      ],
      '#states' => [
        'disabled' => [
          ':input[name="enable_logs"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $result = $this->buildMOLayout($form);
    MoSupport::buildSupportIcon($result);

    return $result;
  }

  /**
   * @throws \Exception
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void
  {
    if ($form_state->getTriggeringElement()['#name'] !== "export_logs") {
      $values = $form_state->cleanValues()->getValues();
      $values['log_severity'] = array_keys(array_filter($values['log_severity']));
      MoModuleSettings::save($values);
      $this->messenger()->addStatus(t('Log settings saved successfully.'));
    }
  }

  public function exportLogs(array $form, FormStateInterface $form_state): AjaxResponse
  {
    $entity = MoModuleSettings::readMe();
    if($entity->getEnableLogs() != 1){
      $this->messenger()->addWarning($this->t('Please enable logs to export'));
      $message = ['#type' => 'status_messages'];
      $response = new AjaxResponse();
      $response->addCommand(new InsertCommand('#ajax-message', \Drupal::service('renderer')->renderRoot($message)));
      return $response;
    }
    $connection = \Drupal::database();
    $arr = $connection
      ->select('watchdog', 'w')
      ->fields('w', ['severity', 'timestamp', 'message', 'variables', 'link', 'location', 'referer'])
      ->condition('w.type', 'miniorange_oauth_client')
      ->orderBy('w.wid', 'DESC')
      ->execute()
      ->fetchAll();

    $data = json_encode($arr, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    $response = new AjaxResponse();

    if (!empty($data)) {
      \Drupal::service('tempstore.private')->get(MoModuleConstant::MODULE_NAME)->set('download_module_logs_data', $data);

      $filename = 'miniorange_oauth_logs-' . date('Y-m-d_H-i-s') . '.json';
      $this->messenger()->addStatus($this->t('Logs are ready, Download will start shortly'));

      $download_url = URL::fromRoute(
        'mo_oauth.download_data', ['config_id' => 'download_module_logs_data', 'file_name' => $filename],
        ['absolute' => true]
      )->toString();

      $iframe_html = "<iframe style='display:none' src='$download_url'></iframe>";
      $response->addCommand(new InvokeCommand('body', 'append', [$iframe_html]));
    } else {
      $this->messenger()->addWarning($this->t('No logs available to Download'));
    }

    $message = ['#type' => 'status_messages'];
    $response->addCommand(new InsertCommand('#ajax-message', \Drupal::service('renderer')->renderRoot($message)));

    return $response;
  }

  public static function getLogSeverityMap(): array
  {
    return [
      'emergency' => t('Emergency'),
      'alert' => t('Alert'),
      'critical' => t('Critical'),
      'error' => t('Error'),
      'warning' => t('Warning'),
      'notice' => t('Notice'),
      'info' => t('Info'),
      'debug' => t('Debug')
    ];
  }
}
