<?php

namespace Drupal\miniorange_oauth_client\MoEntityListBuilder;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\miniorange_oauth_client\Entity\MoClientConfiguration;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoLicense\MoDTO\MoUnoLicense;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoHelper\MoClientConfigurationHelper;
use Drupal\miniorange_oauth_client\MoHelper\MoSupport;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoService\MoLayoutBuilder;

class MoClientConfigurationListBuilder extends FormBase
{
use MoGhostInvoker;
use MoLayoutBuilder;
  /**
   * {@inheritdoc}
   */
  public function getFormId()
  {
    return 'mo_client_configuration_manage_order_form';
  }

  /**
   * {@inheritdoc}
   * @throws EntityMalformedException
   */
  public function buildForm(array $form, FormStateInterface $form_state)
  {
    // General AJAX library
    $form['libraries']['#attached']['library'] = [
      'core/drupal.dialog.ajax',
      'miniorange_oauth_client/notify_us_popup',
      'miniorange_oauth_client/common_js',
      // Drupal new window library
      'miniorange_oauth_client/test_config_window_js'
    ];
    $entities = MoClientConfiguration::loadActive(true, null, null);

    $build_draggable_rows =
      $this->call([MoUnoLicenseTierManager::class, 'getLicenseVersion']) == MoModuleConstant::MO_ENTERPRISE_VERSION &&
      count($entities) > 1 ;

    $form += $this->buildTopBar(count($entities));
    $form['table'] = [
      '#type' => 'table',
      '#header' => $this->buildHeader($build_draggable_rows),
      '#attributes' => [
        'width' => '100%',
      ],
      '#empty' => "There are no applications configured."
    ];

    if ($build_draggable_rows) {
      $form['table']['#tabledrag'] = [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'entities-weight',
        ],
      ];
    }

    foreach ($entities as $id => $entity) {
      $form['table'][$id] = [
        'app_name' => ['#markup' => Link::createFromRoute($entity->getAppName(), 'mo_configuration.edit', ['mo_client_config' => $entity->id()])->toString()],
        'provider' => ['#plain_text' =>$entity->getClientApp()],
        'client_id' => ['#plain_text' =>strlen($entity->getClientId()) > 25 ? substr($entity->getClientId(), 0, 24) . '...' : $entity->getClientId()],
        'test_button' => $this->getTestConfigMarkup($entity),
        'mapping' => $this->buildMappingOperations($entity),
        'action' => $this->buildOperations($entity),
      ];
      if ($build_draggable_rows) {
        $form['table'][$id] += [
          'weight' => [
            '#type' => 'weight',
            '#title' => $this->t('Weight for @title', ['@title' => $entity->id()]),
            '#title_display' => 'invisible',
            '#default_value' => $entity->getWeight(),
            '#attributes' => ['class' => ['entities-weight']],
          ],
          '#attributes' => [
            'class' => ['draggable']
          ]
        ];
      }
    }
    if ($build_draggable_rows) {
      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->t('Save Order'),
        '#attributes' => ['class' => ['button', 'button--primary', 'button--small']]
      ];

      if (\Drupal::service('miniorange_oauth_client.uno_module_subscription_library')->getModuleStatus()) {
        $form['actions']['submit']['#attributes'] += ['disabled' => true];
      }

      $form['actions']['submit_description'] = [
        '#type' => 'item',
        '#description' => t('The above "Save Order" feature prioritizes the Identity Provider login link on the login page.'). '<br>'.
        '<strong class="mo-strong mo-red mo-space-right">' . t('Note:') . '</strong>'.t('This setting is not applicable to Password Grant flow.')
      ];
    }

    $result = self::buildMOLayout($form);
    MoSupport::buildSupportIcon($result);
    return $result;
  }

  /**
   * This function is to build the top nav bar either with the string add new button.
   *
   * @param int $my_list
   * @return array
   * @throws \Exception
   */
  public function buildTopBar(int $avail_count): array
  {
    $add_new = [];

    $license_config = $this->call([MoUnoLicense::class, 'readMe']);
    $provided_count = $license_config?->getNoOfSp() ?? 1;

    \Drupal::service('miniorange_oauth_client.uno_module_subscription_library')->getModuleStatus($add_new);

    if ($avail_count > 0) {
      $add_new['attr_map_warning'] = MoClientConfigurationHelper::getAttributeMappingMarkup();
    }
    $add_new['client_config_add_button'] = [
      '#type' => 'link',
      '#id' => 'client-config-add-button',
      '#title' => $this->t('+ Add New ( ') . (($count = $provided_count - $avail_count) > 0 ? $count : 0) . $this->t(' Remaining )'),
      '#url' => Url::fromRoute('mo_oauth.add_client_config'),
      '#attributes' => [
        'class' => ['button', 'button--primary', 'use-ajax'],
        'data-dialog-type' => 'modal',
      ],
    ];

    return $add_new;
  }
  public function buildHeader(bool $build_draggable_rows): array
  {
    $header['app_name'] = $this->t('App Name');
    $header['provider'] = $this->t('Provider');
    $header['client_id'] = $this->t('Client Id');
    $header['test_button'] = $this->t('Test');
    $header['mapping'] = $this->t('Mappings');
    $header['action'] = $this->t('Action');
    if ($build_draggable_rows) {
      $header['weight'] = $this->t('Weight');
    }
    return $header;
  }


  public static function getTestConfigMarkup(MoClientConfiguration $entity): array
  {
    $button = [
      '#type' => 'button',
      '#value' => t('Perform Test Configuration'),
      '#attributes' => [
        'class' => ['button', 'button--small', 'mo-test-config'],
        'url' => Url::fromRoute(
          $entity->getGrantType() == 'password' ? 'mo_oauth.password_grant.test_config' : 'mo_oauth.authorization_request',
          ['mo_client_config' => $entity->id(), 'test_sso' => true],
          ['absolute' => TRUE]
        )->toString(),
      ],
    ];
    if (\Drupal::service('miniorange_oauth_client.uno_module_subscription_library')->getModuleStatus()) {
      $button['#attributes'] += ['disabled' => true];
    }
    return $button;
  }

  /**
   * @throws EntityMalformedException
   */
  public function buildOperations(EntityInterface $entity): array
  {
    return [
      '#type' => 'operations',
      '#links' => [
          'edit' => [
            'title' => t('Edit'),
            'weight' => 10,
            'url' => Url::fromRoute('mo_configuration.edit', ['mo_client_config' => $entity->id()])
          ],
        'delete' => [
          'title' => t('Delete'),
          'weight' => 10,
          'url' => $entity->toUrl('delete-form'),
          'attributes' => [
            'data-dialog-type' => 'modal',
            'data-action' => 'delete',
            'data-dialog-options' => '{"width":880}',
            'class' => ['use-ajax']
          ]
        ],
        ]
    ];
  }
  public function buildMappingOperations(EntityInterface $entity): array
  {
    return [
      '#type' => 'operations',
      '#links' => [
        'attribute_mapping' => [
          'title' => t('Attribute Mapping'),
          'weight' => 10,
          'url' => Url::fromRoute(
            'mo_configuration.edit', ['mo_client_config' => $entity->id(), 'tab' => 'attribute_mapping']
          )
        ],
        'role_mapping' => [
          'title' => t('Role Mapping'),
          'weight' => 20,
          'url' => Url::fromRoute(
            'mo_configuration.edit', ['mo_client_config' => $entity->id(), 'tab' => 'role_mapping']
          )
        ],
        'profile_mapping' => [
          'title' => t('Profile Mapping'),
          'weight' => 20,
          'url' => Url::fromRoute(
            'mo_configuration.edit', ['mo_client_config' => $entity->id(), 'tab' => 'profile_mapping']
          )
        ],
        'group_mapping' => [
          'title' => t('Group Mapping'),
          'weight' => 20,
          'url' => Url::fromRoute(
            'mo_configuration.edit', ['mo_client_config' => $entity->id(), 'tab' => 'group_mapping']
          )
        ],
        'client_setting' => [
          'title' => t('Client Settings'),
          'weight' => 20,
          'url' => Url::fromRoute(
            'mo_configuration.edit', ['mo_client_config' => $entity->id(), 'tab' => 'client_setting']
          )
        ]
      ]
    ];
  }
  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void
  {
    $values = $form_state->getValue('table');
    foreach ($values as $id => $row) {
      $entity = MoClientConfiguration::load($id);
      if ($entity) {
        $entity->set('weight', $row['weight']);
        $entity->save();
      }
    }
    $this->messenger()->addMessage($this->t('The order has been updated.'));
  }
}
