<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\MoEntityListBuilder;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Url;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoHelper\MoSupport;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoService\MoLayoutBuilder;

/**
 * Provides a list controller for the miniOrange login reports entity type.
 */
final class MoLoginReportListBuilder extends EntityListBuilder
{
  use MoLayoutBuilder;
  use MoGhostInvoker;

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array
  {
    $header['username'] = $this->t('User Name');
    $header['email'] = $this->t('Email');
    $header['application'] = $this->t('Application');
    $header['status'] = $this->t('Status');
    $header['created'] = $this->t('Initiated At');
    $header['ip_address'] = $this->t('IP Address');
    $header['navigation_url'] = $this->t('Navigation URL');

    return $header;
  }

  public function load()
  {
    $query = $this->getStorage()->getQuery()
      ->accessCheck(TRUE)
      ->sort($this->entityType->getKey('sort'), 'DESC');

    $ids = $query->execute();
    return $this->storage->loadMultiple($ids);
  }
  /**
   * @throws \Exception
   */
  public function render(): array
  {
    $form = [];
    $disabled = \Drupal::service('miniorange_oauth_client.uno_module_subscription_library')->getModuleStatus($form);
    $current_lic_ver = $this->call([MoUnoLicenseTierManager::class, 'getLicenseVersion']);
    if (!in_array($current_lic_ver, MoModuleConstant::MO_ENTERPRISE_VERSION_TIER)) {
      \Drupal::messenger()->addWarning($this->t('This feature is available in the enterprise version of the module.'));
    }
    if (!empty($this->getEntityIds())) {
      $form['top_bar'] = [
        '#type' => 'link',
        '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
        '#title' => t('Clear All'),
        '#attributes' => [
          'class' => ['button', 'button--danger', 'use-ajax'],
          'data-dialog-type' => 'modal',
          'data-dialog-options' => Json::encode([
            'width' => 800,
          ]),
        ],
        '#attached' => [
          'library' => ['core/drupal.dialog.ajax'],
        ],
        '#url' => Url::fromRoute('entity.mo_login_reports.delete_multiple'),
      ];
      if ($disabled) {
        $form['top_bar']['#access'] = false;
      }
    }
    $form += parent::render();
    $this->modifyFormTitleOnTier($form);
    $result = $this->buildMOLayout($form);
    MoSupport::buildSupportIcon($result);
    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array
  {
    $row['username'] = $entity->get('username')->value;
    $row['email'] = $entity->get('email')->value;
    $row['application'] = $entity->get('application')->value;
    $row['status'] = $entity->get('status')->value;
    $row['created'] = \Drupal::service('date.formatter')->format($entity->get('created_at')->value);
    $row['ip_address'] = $entity->get('ip_address')->value;
    $row['navigation_url'] = [
      'data' => $entity->get('navigation_url')->value,
      'style' => 'word-break: break-all;',
    ];
    return $row;
  }

}
