<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\MoHelper\JWTHandler\Utils;

use Drupal\miniorange_oauth_client\MoHelper\JWTHandler\Utils\RSAUtils\Crypt_RSA;
use Drupal\miniorange_oauth_client\MoHelper\JWTHandler\Utils\RSAUtils\Math\Math_BigInteger;
use Drupal\miniorange_oauth_client\MoHelper\MoUtilities;
use Drupal\miniorange_oauth_client\MoLibrary\MoLogger;

class JWTUtils {

	const HEADER  = 'HEADER';
	const PAYLOAD = 'PAYLOAD';
	const SIGN    = 'SIGN';
	private $decoded_jwt;
  private $jwt;
	private $session;

  /**
   * @throws \Exception
   */
  public function __construct(
      string $jwt
      ) {
    $this->jwt = explode( '.', $jwt );
		if ( 3 > count( $this->jwt ) ) {
      throw new \Exception('The ID Token Received is not a valid JWT');
		}
    $this->session =\Drupal::service('session');

		$header            = $this->get_jwt_claim( '', self::HEADER );
		$payload           = $this->get_jwt_claim( '', self::PAYLOAD );
		$this->decoded_jwt = array(
			'header'  => $header,
			'payload' => $payload,
		);
	}

	private function get_jwt_claim( $claim = '', $part = '' ) {
		$target = '';
		switch ( $part ) {
			case self::HEADER:
				$target = $this->jwt[0];
				break;
			case self::PAYLOAD:
				$target = $this->jwt[1];
				break;
			case self::SIGN:
				return $this->jwt[2];
			default:
        throw new \Exception('Cannot Find ' . $part . ' in the id_token');
		}
		$target = json_decode(MoUtilities::base64urlDecode($target), true );
		if ( ! $target || empty( $target ) ) {
			return null;
		}
		return empty( $claim ) ? $target : ( isset( $target[ $claim ] ) ? $target[ $claim ] : null );
	}

  /**
   * @throws \Exception
   */
  public function verify($secret = '' ){
		if ( empty( $secret ) ) {
			return false;
		}
		$sign_verify = new JWSVerify($this->get_jwt_claim( 'alg', self::HEADER ));
		$payload_to_verify = $this->get_header() . '.' . $this->get_payload();
		return $sign_verify->verify($payload_to_verify, $secret, base64_decode( strtr( $this->get_jwt_claim( false, self::SIGN ), '-_', '+/' ) ));
	}

  /**
   * @throws \Exception
   */
  public function verify_from_jwks($jwks_uri = '', $algo = 'RS256' ){
		$jwks_resposnse = MoUtilities::buildGetRequest( NULL, $jwks_uri);

		$verified = false;
		if (json_last_error() !== JSON_ERROR_NONE){
			return $verified;
		}

		if (!isset( $jwks_resposnse['keys'])){
			return $verified;
		}
		foreach ( $jwks_resposnse['keys'] as $key => $value ) {
			if( ! isset( $value['kty'] ) || 'RSA' !== $value['kty'] || ! isset( $value['e'] ) || ! isset( $value['n'] ) ) {
				continue;
			}
			$verified = $verified || $this->verify(
				$this->jwks_to_pem(
					[
						'n' => new Math_BigInteger(MoUtilities::base64urlDecode($value['n']), 256 ),
						'e' => new Math_BigInteger(MoUtilities::base64urlDecode($value['e']), 256 ),
					]
				)
			);
			if ( true === $verified ) {
				break;
			}
		}
		return $verified;
	}

	public function verify_exp_and_nonce( )
  {
		$expiry = $this->get_jwt_claim( 'exp', self::PAYLOAD );
		if ( is_null( $expiry ) || time() > $expiry ) {
      MoLogger::error('Current Timestamp => '.time());
      MoLogger::error('Timestamp in JWT exp claim => '.$expiry);
      MoLogger::error('JWT has been expired. Please try again.');

      throw new \Exception('JWT has been expired. Please try try Logging in again.');
		}
		$nonce = $this->get_jwt_claim( 'nonce', self::PAYLOAD );
    $nonce_stored = $this->session->get('nonce');

		$this->session->remove('nonce');
    if(is_null( $nonce ) || $nonce != $nonce_stored){
          MoLogger::error('nonce stored => '.$nonce_stored);
          MoLogger::error('nonce recieved => '.$nonce);
          throw new \Exception('Invalid nonce claim. Please try again.');
    }
    $time_stored = $this->session->get('mo_sso_request_time');
    if($time_stored>time() || $time_stored+1800 < time()){
      MoLogger::error('Current Timestamp => '.time());
      MoLogger::error('The request was sent at Timestamp => '.$time_stored);
      throw new \Exception('The request has been expired. Please try again.');
		}
    return true;
	}


	private function jwks_to_pem( $jwks = [] ) {
		$rsa = new Crypt_RSA();
		$rsa->loadKey( $jwks );
		return $rsa->getPublicKey();
	}

	public function get_decoded_header() {
		return $this->decoded_jwt['header'];
	}

	public function get_decoded_payload() {
		if(isset($this->decoded_jwt['payload']))
		return $this->decoded_jwt['payload'];
	}

	public function get_header() {
		return $this->jwt[0];
	}

	public function get_payload() {
		return $this->jwt[1];
	}
}
