<?php

namespace Drupal\miniorange_saml\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\miniorange_saml\Utilities;

/**
 * Showing Support form info.
 */
class MiniorangeSupport extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'miniorange_SAML_support';
  }

  /**
   * Builds the form for support query.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The rendered form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $email = $this->config('miniorange_saml_settings')->get('miniorange_saml_customer_admin_email');
    $phone = $this->config('miniorange_saml_settings')->get('miniorange_saml_customer_admin_phone');
    $form['miniorange_saml_support'] = [
      '#type' => 'container',
      '#prefix' => '<div id="modal_support_form">',
      '#suffix' => '</div>',
    ];
    $form['miniorange_saml_support']['mo_otp_verification_script'] = [
      '#attached' => [
        'library' => [
          'core/drupal.dialog.ajax',
          'miniorange_saml/miniorange_saml.admin',
        ],
      ],
    ];
    $form['miniorange_saml_support']['mo_otp_verification_status_messages'] = [
      '#type' => 'status_messages',
      '#weight' => -10,
    ];
    $form['miniorange_saml_support']['mo_saml_markup_1'] = [
      '#markup' => t('<div class="mo_saml_highlight_background_note_1">
                    We are here to help you with configuring this module on your site. Feel free to reach out with your query, and we\'ll be happy to assist you promptly.</div>'),
    ];
    $form['miniorange_saml_support']['mo_saml_support_email_address'] = [
      '#type' => 'email',
      '#title' => $this->t('Email'),
      '#default_value' => $email,
      '#required' => TRUE,
      '#attributes' => [
        'placeholder' => t('Enter your email'),
        'style' => 'width:99%;margin-bottom:1%;',
      ],
    ];
    $form['miniorange_saml_support']['mo_saml_support_query'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Query'),
      '#required' => TRUE,
      '#attributes' => [
        'placeholder' => $this->t('Describe your query here (Maximum 500 characters allowed)'),
        'style' => 'width:99%',
        'maxlength' => 500,
        'minlength' => 1,
      ],
      '#suffix' => '<br>',
    ];
    $form['miniorange_saml_support']['actions'] = [
      '#type' => 'actions',
    ];
    $form['miniorange_saml_support']['actions']['submit'] = [
      '#type' => 'submit',
      '#button_type' => 'primary',
      '#value' => $this->t('Submit query'),
      '#attributes' => [
        'class' => [
          'use-ajax',
        ],
      ],
      '#ajax' => [
        'callback' => '::submitQuery',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Sending Query...'),
        ],
      ],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {}

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {}

  /**
   * Handles the submission of a support query and returns an AJAX response.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The response containing the AJAX commands.
   */
  public function submitQuery(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    if ($form_state->hasAnyErrors()) {
      $response->addCommand(new ReplaceCommand('#modal_support_form', $form));
    }
    else {
      $form_values = $form_state->getValues();
      $email = $form_values['mo_saml_support_email_address'];
      $phone = '';
      $query = $form_values['mo_saml_support_query'] . '<br><br>' . 'Timezone: ' . Utilities::getFormattedTimezone();
      $queryType = 'Support';
      $support_response = Utilities::sendSupportQuery($email, $phone, $query, $queryType);
      $ajax_form = Utilities::getModalFormAfterQuery($support_response, $email, FALSE);
      $response->addCommand($ajax_form);
    }
    return $response;
  }

}
