<?php

namespace Drupal\miniorange_saml\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;

/**
 * Provides a form for the miniorange_saml bundle plan promotion.
 */
class MiniorangeBundlePlan extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'miniorange_saml_bundle_plan';
  }

  /**
   * Builds the bundle plan promotion form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    global $base_url;
    $module_path = $base_url . '/' . \Drupal::service('extension.list.module')->getPath("miniorange_saml");

    // Attach CSS library.
    $form['#attached']['library'][] = 'miniorange_saml/miniorange_saml.admin';

    // Main fieldset container.
    $form['bundle_plan_fieldset'] = [
      '#type' => 'fieldset',
      '#title' => '',
      '#attributes' => [
        'class' => ['bundle-plan-container'],
      ],
    ];

    // Left section with text content.
    $form['bundle_plan_fieldset']['left_section'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['bundle-plan-left-section'],
      ],
    ];

    // Headline.
    $form['bundle_plan_fieldset']['left_section']['headline'] = [
      '#markup' => Markup::create('<h3 class="bundle-plan-headline">' . $this->t('Beyond your SSO setup with User Provisioning!') . '</h3>'),
    ];

    // Sub-headline/Description.
    $form['bundle_plan_fieldset']['left_section']['description'] = [
      '#markup' => Markup::create('<div class="bundle-plan-description">' . $this->t('Manage user access and automate all user CRUD operations — directly inside your Drupal SAML Plugin.') . '</div>'),
    ];

    // Features list.
    $features = [
      $this->t('Automated user provisioning & sync'),
      $this->t('Real-time roles & groups mapping'),
      $this->t('Role-based user provisioning'),
      $this->t('Seamless integration with any IdP — Azure, Okta, Salesforce, and more'),
    ];

    $form['bundle_plan_fieldset']['left_section']['features_list'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['bundle-plan-features-list'],
      ],
    ];

    foreach ($features as $index => $feature) {
      $form['bundle_plan_fieldset']['left_section']['features_list']['feature_' . $index] = [
        '#type' => 'checkbox',
        '#title' => $feature,
        '#default_value' => 1,
      ];
    }

    // Action buttons container.
    $form['bundle_plan_fieldset']['left_section']['actions'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['bundle-plan-actions'],
      ],
    ];

    // Upgrade to Bundle Plan button.
    $form['bundle_plan_fieldset']['left_section']['actions']['upgrade_button'] = [
      '#type' => 'link',
      '#title' => $this->t('Upgrade to Bundle Plan'),
      '#url' => Url::fromUri('https://plugins.miniorange.com/drupal-saml-single-sign-on-sso', ['external' => TRUE]),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
        'target' => '_blank',
      ],
    ];

    // Setup Guide button.
    $form['bundle_plan_fieldset']['left_section']['actions']['setup_guide_button'] = [
      '#type' => 'link',
      '#title' => $this->t('Setup Guide'),
      '#url' => Url::fromUri('https://plugins.miniorange.com/drupal/setup-guides?category=user-provisioning', ['external' => TRUE]),
      '#attributes' => [
        'class' => ['button'],
        'target' => '_blank',
      ],
    ];

    // Right section with visual illustration.
    $form['bundle_plan_fieldset']['right_section'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['bundle-plan-right-section'],
      ],
    ];

    $form['bundle_plan_fieldset']['right_section']['illustration'] = [
      '#markup' => Markup::create('<div class="bundle-plan-image-container"><img class="bundle-plan-image" src="' . $module_path . '/includes/images/bundle-plan.png" alt="' . $this->t('Bundle Plan') . '"></div>'),
    ];

    // SCIM Server Configuration fieldset.
    $form['scim_config_fieldset'] = [
      '#type' => 'details',
      '#title' => $this->t('Configure Drupal as a SCIM Server'),
      '#attributes' => [
        'class' => ['scim-config-container'],
      ],
      '#open' => TRUE,
    ];

    // Description.
    $form['scim_config_fieldset']['description'] = [
      '#markup' => Markup::create('<p>' . $this->t('Share the following URL and token with your IdP to enable secure user provisioning in Drupal.') . '</p>'),
    ];

    // Select Your Client dropdown.
    $form['scim_config_fieldset']['select_client'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Your Client:'),
      '#options' => [
        'okta' => $this->t('Okta'),
        'azure' => $this->t('Azure'),
        'salesforce' => $this->t('Salesforce'),
      ],
      '#default_value' => 'okta',
    ];

    // SCIM Base URL.
    $form['scim_config_fieldset']['scim_base_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SCIM Base URL:'),
      '#default_value' => $base_url . '/scim',
      '#attributes' => [
        'readonly' => 'readonly',
      ],
    ];

    // SCIM Bearer Token container.
    $form['scim_config_fieldset']['token_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['scim-token-container'],
      ],
    ];

    // SCIM Bearer Token field.
    $form['scim_config_fieldset']['token_container']['scim_bearer_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SCIM Bearer Token:'),
      '#default_value' => '81cbbb0087a04c80846a1efc5d348e9468a1cc6787bbc4d9b74e59b92275629a',
      '#attributes' => [
        'readonly' => 'readonly',
      ],
    ];

    // Generate New button.
    $form['scim_config_fieldset']['token_container']['generate_token'] = [
      '#type' => 'button',
      '#value' => $this->t('Generate New'),
      '#attributes' => [
        'class' => ['button', 'button--secondary'],
      ],
      '#disabled' => TRUE,
    ];

    // Attribute Mapping fieldset.
    $form['attribute_mapping_fieldset'] = [
      '#type' => 'details',
      '#title' => $this->t('Attribute Mapping'),
      '#attributes' => [
        'class' => ['attribute-mapping-container'],
      ],
    ];

    // Basic Attribute Mapping subsection.
    $form['attribute_mapping_fieldset']['basic_attribute_mapping'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['basic-attribute-mapping'],
      ],
    ];

    $form['attribute_mapping_fieldset']['basic_attribute_mapping']['title'] = [
      '#markup' => Markup::create('<h4>' . $this->t('Basic Attribute Mapping:') . '</h4>'),
    ];

    $form['attribute_mapping_fieldset']['basic_attribute_mapping']['description'] = [
      '#markup' => Markup::create('<p>' . $this->t('Predefined mappings for common user fields like Email and Name.') . '</p>'),
    ];

    // Basic attribute fields container.
    $form['attribute_mapping_fieldset']['basic_attribute_mapping']['fields_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['basic-attribute-fields'],
      ],
    ];

    // Username Attribute field.
    $form['attribute_mapping_fieldset']['basic_attribute_mapping']['fields_container']['username_attribute'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username Attribute:'),
      '#default_value' => 'userName',
      '#disabled' => TRUE,
    ];

    // Email Attribute field.
    $form['attribute_mapping_fieldset']['basic_attribute_mapping']['fields_container']['email_attribute'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Email Attribute:'),
      '#default_value' => 'emails.0.value',
      '#disabled' => TRUE,
    ];

    // Custom Attribute Mapping subsection.
    $form['attribute_mapping_fieldset']['custom_attribute_mapping'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['custom-attribute-mapping'],
      ],
    ];

    $form['attribute_mapping_fieldset']['custom_attribute_mapping']['title'] = [
      '#markup' => Markup::create('<h4>' . $this->t('Custom Attribute Mapping:') . '</h4>'),
    ];

    $form['attribute_mapping_fieldset']['custom_attribute_mapping']['description'] = [
      '#markup' => Markup::create('<p>' . $this->t('Add mappings for additional or organization-specific fields.') . '</p>'),
    ];

    // Custom attribute mapping table.
    $form['attribute_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_table'] = [
      '#type' => 'table',
      '#header' => [
        [
          'data' => $this->t('SCIM Client Attribute Name'),
          'width' => '40%',
        ],
        [
          'data' => $this->t('Drupal Attribute'),
          'width' => '40%',
        ],
        [
          'data' => $this->t('Operation'),
          'width' => '20%',
        ],
      ],
    ];

    // First row of the table.
    $form['attribute_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_table'][0]['scim_client_attribute'] = [
      '#type' => 'select',
      '#options' => [
        '' => $this->t('-Select IDP Attribute-'),
      ],
      '#disabled' => TRUE,
    ];

    $form['attribute_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_table'][0]['drupal_attribute'] = [
      '#type' => 'select',
      '#options' => [
          '' => $this->t('-Select Attribute Value-'),
      ],
      '#disabled' => TRUE,
    ];

    $form['attribute_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_table'][0]['delete_button'] = [
      '#markup' => Markup::create('<span class="button">' . $this->t('Remove') . '</span>'),
    ];

    // Add button and number input.
    $form['attribute_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_add_button'] = [
      '#prefix' => '<div class="container-inline">',
      '#type' => 'button',
      '#value' => $this->t('Add'),
      '#disabled' => TRUE,
    ];

    $form['attribute_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_total_items'] = [
      '#type' => 'number',
      '#default_value' => 1,
      '#min' => 1,
      '#max' => 50,
      '#suffix' => '&nbsp;&nbsp;' . $this->t('more rows') . '</div>',
    ];

    // Role Mapping fieldset.
    $form['role_mapping_fieldset'] = [
      '#type' => 'details',
      '#title' => $this->t('Role Mapping'),
      '#attributes' => [
        'class' => ['role-mapping-container'],
      ],
    ];

    // Basic Role Mapping subsection.
    $form['role_mapping_fieldset']['basic_role_mapping'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['basic-role-mapping'],
      ],
    ];

    $form['role_mapping_fieldset']['basic_role_mapping']['title'] = [
      '#markup' => Markup::create('<h4>' . $this->t('Basic Role Mapping:') . '</h4>'),
    ];

    $form['role_mapping_fieldset']['basic_role_mapping']['description'] = [
      '#markup' => Markup::create('<p>' . $this->t('Predefined mappings for common user fields like Email and Name.') . '</p>'),
    ];

    // Enable Role Mapping checkbox.
    $form['role_mapping_fieldset']['basic_role_mapping']['enable_role_mapping'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Role Mapping'),
      '#default_value' => 0,
      '#description' => $this->t('Note : Enable this first if you want to use below features.'),
      '#disabled' => TRUE,
    ];

    // Update existing roles checkbox.
    $form['role_mapping_fieldset']['basic_role_mapping']['update_existing_roles'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Update the users existing roles.'),
      '#default_value' => 0,
      '#description' => $this->t('Note : Enable this first if you want to use below features.'),
      '#disabled' => TRUE,
    ];

    // Select default group for new users.
    $form['role_mapping_fieldset']['basic_role_mapping']['default_group'] = [
      '#type' => 'select',
      '#title' => $this->t('Select default group for new users'),
      '#options' => [
        'authenticated' => $this->t('authenticated user'),
        'administrator' => $this->t('administrator'),
        'content-editor' => $this->t('content editor')
      ],
      '#default_value' => 'authenticated',
      '#description' => $this->t('Note: This role will be assigned to user at the time of user creation in Drupal site.'),
      '#disabled' => TRUE,
    ];

    // Custom Attribute Mapping subsection.
    $form['role_mapping_fieldset']['custom_attribute_mapping'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['custom-attribute-mapping'],
      ],
    ];

    $form['role_mapping_fieldset']['custom_attribute_mapping']['title'] = [
      '#markup' => Markup::create('<h4>' . $this->t('Custom Attribute Mapping:') . '</h4>'),
    ];

    $form['role_mapping_fieldset']['custom_attribute_mapping']['description'] = [
      '#markup' => Markup::create('<p>' . $this->t('Add mappings for additional or organization-specific fields.') . '</p>'),
    ];

    // Attribute mapping table.
    $form['role_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_table'] = [
      '#type' => 'table',
      '#header' => [
        [
          'data' => $this->t('SCIM Client Attribute Name'),
          'width' => '40%',
        ],
        [
          'data' => $this->t('Drupal Attribute'),
          'width' => '40%',
        ],
        [
          'data' => $this->t('Operation'),
          'width' => '20%',
        ],
      ],
    ];

    // First row of the table.
    $form['role_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_table'][0]['scim_client_attribute'] = [
      '#type' => 'select',
      '#options' => [
        '' => $this->t('-Select IDP Attribute-'),
      ],
      '#disabled' => TRUE,
    ];

    $form['role_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_table'][0]['drupal_attribute'] = [
      '#type' => 'select',
      '#options' => [
        '' => $this->t('-Select Attribute Value-'),
      ],
      '#disabled' => TRUE,
    ];

    $form['role_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_table'][0]['delete_button'] = [
      '#markup' => Markup::create('<span class="button">' . $this->t('Remove') . '</span>'),
    ];

    // Add button and number input.
    $form['role_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_add_button'] = [
      '#prefix' => '<div class="container-inline">',
      '#type' => 'button',
      '#value' => $this->t('Add'),
      '#disabled' => TRUE,
    ];

    $form['role_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_total_items'] = [
      '#type' => 'number',
      '#default_value' => 1,
      '#min' => 1,
      '#max' => 50,
      '#suffix' => '&nbsp;&nbsp;' . $this->t('more rows') . '</div>',
    ];

    // Additional Features fieldset.
    $form['additional_features_fieldset'] = [
      '#type' => 'details',
      '#title' => $this->t('Additional Features'),
      '#attributes' => [
        'class' => ['additional-features-container'],
      ],
    ];

    // First radio button group - User deletion action.
    $form['additional_features_fieldset']['user_deletion_action'] = [
      '#type' => 'radios',
      '#title' => $this->t('Perform the following action on Drupal user list when user is deleted from IDP (SCIM Client):'),
      '#options' => [
        'do_nothing' => $this->t('Do Nothing'),
        'delete_users' => $this->t('Delete Users'),
        'deactivate_users' => $this->t('Deactivate Users'),
        'delete_and_assign' => $this->t('Delete users and assign content to an anonymous user'),
      ],
      '#default_value' => 'do_nothing',
      '#attributes' => [
        'class' => ['container-inline'],
        ],
      '#disabled' => TRUE,
    ];

    // Second radio button group - User deletion action (second group).
    $form['additional_features_fieldset']['user_deletion_action_2'] = [
      '#type' => 'radios',
      '#title' => $this->t('Perform the following action on Drupal user list when user is deleted from IDP (SCIM Client):'),
      '#options' => [
        'allowed' => $this->t('Allowed'),
        'not_allowed' => $this->t('Not Allowed'),
      ],
      '#default_value' => 'allowed',
      '#attributes' => [
          'class' => ['container-inline'],
      ],
      '#disabled' => TRUE,
    ];

    // Checkbox - Disabled new Group creation.
    $form['additional_features_fieldset']['disabled_group_creation'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disabled new Group creation'),
      '#default_value' => 0,
      '#disabled' => TRUE,
    ];

    $form['save_configuration_button'] = [
        '#type' => 'submit',
        '#value' => $this->t('Save Configuration'),
        '#attributes' => [
            'class' => ['button', 'button--primary'],
        ],
        '#disabled' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // No form submission needed for promotional page.
  }

}
