/**
 * @file
 * Dynamic pricing functionality for plan cards.
 */

(function ($, Drupal) {
  'use strict';

  // Pricing table based on number of instances
  const pricingTable = {
    1: { standard: 249, premium: 399, enterprise: 449 },
    2: { standard: 479, premium: 749, enterprise: 849 },
    3: { standard: 699, premium: 1119, enterprise: 1249 },
    4: { standard: 929, premium: 1419, enterprise: 1549 },
    5: { standard: 1079, premium: 1659, enterprise: 1849 },
    6: { standard: 1279, premium: 1899, enterprise: 2099 },
    7: { standard: 1479, premium: 2129, enterprise: 2349 },
    8: { standard: 1689, premium: 2349, enterprise: 2549 },
    9: { standard: 1899, premium: 2499, enterprise: 2749 },
    10: { standard: 2019, premium: 2599, enterprise: 2899 }
  };

  /**
   * Update prices for all cards based on selected instances.
   */
  function updatePrices(selectedInstances) {
    const instances = parseInt(selectedInstances);
    const prices = pricingTable[instances];

    if (!prices) {
      return;
    }

    // Update Standard card price
    const standardCard = document.querySelector('.mo-saml-plan-card[data-plan-type="standard"]');
    if (standardCard) {
      const priceElement = standardCard.querySelector('.mo-saml-price-amount');
      if (priceElement) {
        priceElement.textContent = '$' + prices.standard;
      }
    }

    // Update Premium card price
    const premiumCard = document.querySelector('.mo-saml-plan-card[data-plan-type="premium"]');
    if (premiumCard) {
      const priceElement = premiumCard.querySelector('.mo-saml-price-amount');
      if (priceElement) {
        priceElement.textContent = '$' + prices.premium;
      }
    }

    // Update Enterprise card price
    const enterpriseCard = document.querySelector('.mo-saml-plan-card[data-plan-type="enterprise"]');
    if (enterpriseCard) {
      const priceElement = enterpriseCard.querySelector('.mo-saml-price-amount');
      if (priceElement) {
        priceElement.textContent = '$' + prices.enterprise;
      }
    }
  }

  /**
   * Update all instance dropdowns to the same value.
   */
  function updateAllInstanceDropdowns(selectedValue) {
    const allInstanceSelects = document.querySelectorAll('.mo-saml-instance-select');
    allInstanceSelects.forEach(function(select) {
      if (select.value !== selectedValue) {
        select.value = selectedValue;
      }
    });
  }

  /**
   * Update all version dropdowns to the same value.
   */
  function updateAllVersionDropdowns(selectedValue) {
    const allVersionSelects = document.querySelectorAll('.mo-saml-version-select');
    allVersionSelects.forEach(function(select) {
      if (select.value !== selectedValue) {
        select.value = selectedValue;
      }
    });
  }

  /**
   * Attach behavior for dynamic pricing.
   */
  Drupal.behaviors.moSamlDynamicPricing = {
    attach: function (context, settings) {
      // Find all instance select dropdowns
      const instanceSelects = context.querySelectorAll('.mo-saml-instance-select');
      
      instanceSelects.forEach(function(select) {
        // Remove existing event listeners to avoid duplicates
        $(select).off('change.moSamlPricing');
        
        // Add change event listener
        $(select).on('change.moSamlPricing', function() {
          const selectedValue = $(this).val();
          
          // Update all instance dropdowns
          updateAllInstanceDropdowns(selectedValue);
          
          // Update prices for all cards
          updatePrices(selectedValue);
        });
      });

      // Initialize prices based on default selected instance value
      if (instanceSelects.length > 0) {
        const initialValue = instanceSelects[0].value;
        updatePrices(initialValue);
      }

      // Find all version select dropdowns
      const versionSelects = context.querySelectorAll('.mo-saml-version-select');
      
      versionSelects.forEach(function(select) {
        // Remove existing event listeners to avoid duplicates
        $(select).off('change.moSamlVersion');
        
        // Add change event listener
        $(select).on('change.moSamlVersion', function() {
          const selectedValue = $(this).val();
          
          // Update all version dropdowns
          updateAllVersionDropdowns(selectedValue);
        });
      });
    }
  };

})(jQuery, Drupal);

