<?php

namespace Drupal\miniorange_saml\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\miniorange_saml\MoSupportIconForm;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for the miniorange_saml bundle plan promotion.
 */
class MiniorangeBundlePlan extends FormBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a MiniorangeBundlePlan object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory, MessengerInterface $messenger, LoggerChannelFactoryInterface $logger_factory) {
    $this->configFactory = $config_factory;
    $this->messenger = $messenger;
    $this->loggerFactory = $logger_factory;
    $this->logger = $logger_factory->get('miniorange_saml');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('messenger'),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'miniorange_saml_bundle_plan';
  }

  /**
   * Builds the bundle plan promotion form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Attach dialog library for modal functionality.
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';

    // Bundle Plan Overview Form.
    $this->bundlePlanOverviewForm($form, $form_state);

    // Drupal as SCIM Server Form
    $this->drupalAsScimServerForm($form, $form_state);

    // Additional Feature Form
    $this->additionalFeatureForm($form, $form_state);

    // Attribute Mapping fieldset.
    $this->attributeMappingForm($form, $form_state);

    // Role Mapping fieldset.
    $this->roleMappingForm($form, $form_state);

    $form['save_configuration'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save Configuration'),
      '#disabled' => TRUE,
    ];

    MoSupportIconForm::buildSupportIcon($form);
    return $form;
  }

  public function bundlePlanOverviewForm(array &$form, FormStateInterface $form_state)
  {
      global $base_url;
      $module_path = $base_url . '/' . \Drupal::service('extension.list.module')->getPath("miniorange_saml");

      // Attach CSS library.
      $form['#attached']['library'][] = 'miniorange_saml/miniorange_saml.admin';

      // Main fieldset container.
      $form['bundle_plan_fieldset'] = [
          '#type' => 'fieldset',
          '#title' => '',
          '#attributes' => [
              'class' => ['bundle-plan-container'],
          ],
      ];

      // Left section with text content.
      $form['bundle_plan_fieldset']['left_section'] = [
          '#type' => 'container',
          '#attributes' => [
              'class' => ['bundle-plan-left-section'],
          ],
      ];

      // Headline.
      $form['bundle_plan_fieldset']['left_section']['headline'] = [
          '#markup' => Markup::create('<h3 class="bundle-plan-headline">' . $this->t('Level Up With User Provisioning!') . '</h3>'),
      ];

      // Sub-headline/Description.
      $form['bundle_plan_fieldset']['left_section']['description'] = [
          '#markup' => Markup::create('<div class="bundle-plan-description">' . $this->t('Manage user access and automate all user creation, update, deletion, and deactivation processes — directly inside your Drupal SAML Plugin.') . '</div>'),
      ];

      // Features list.
      $features = [
          $this->t('Automated user provisioning & sync'),
          $this->t('Real-time roles & groups mapping'),
          $this->t('Role-based user provisioning'),
          $this->t('Seamless integration with any IdP — Azure, Okta, Keycloak, Salesforce, and more'),
      ];

      $form['bundle_plan_fieldset']['left_section']['features_list'] = [
          '#type' => 'container',
          '#attributes' => [
              'class' => ['bundle-plan-features-list'],
          ],
      ];

      foreach ($features as $index => $feature) {
          $form['bundle_plan_fieldset']['left_section']['features_list']['feature_' . $index] = [
              '#markup' => Markup::create('<div class="bundle-plan-feature-item"><span class="bundle-plan-checkmark">✓</span><span class="bundle-plan-feature-text">' . $feature . '</span></div>'),
          ];
      }

      // Action buttons container.
      $form['bundle_plan_fieldset']['left_section']['actions'] = [
          '#type' => 'container',
          '#attributes' => [
              'class' => ['bundle-plan-actions'],
          ],
      ];

      // Upgrade to Bundle Plan button.
      $form['bundle_plan_fieldset']['left_section']['actions']['upgrade_button'] = [
          '#type' => 'link',
          '#title' => $this->t('Upgrade to Suite Plan'),
          '#url' => Url::fromUri('https://plugins.miniorange.com/drupal-saml-single-sign-on-sso', ['external' => TRUE]),
          '#attributes' => [
              'class' => ['button', 'button--primary'],
              'target' => '_blank',
          ],
      ];

      // Setup Guide button.
      $form['bundle_plan_fieldset']['left_section']['actions']['setup_guide_button'] = [
          '#type' => 'link',
          '#title' => $this->t('Setup Guides'),
          '#url' => Url::fromUri('https://plugins.miniorange.com/drupal/setup-guides?category=user-provisioning', ['external' => TRUE]),
          '#attributes' => [
              'class' => ['button'],
              'target' => '_blank',
          ],
      ];

      // Right section with visual illustration.
      $form['bundle_plan_fieldset']['right_section'] = [
          '#type' => 'container',
          '#attributes' => [
              'class' => ['bundle-plan-right-section'],
          ],
      ];

      $form['bundle_plan_fieldset']['right_section']['illustration'] = [
          '#markup' => Markup::create('<div class="bundle-plan-image-container"><img class="bundle-plan-image" src="' . $module_path . '/includes/images/bundle-plan.png" alt="' . $this->t('Bundle Plan') . '"></div>'),
      ];

      return $form;
  }

  public function drupalAsScimServerForm(array &$form, FormStateInterface $form_state)
  {
      global $base_url;
      $module_path = $base_url . '/' . \Drupal::service('extension.list.module')->getPath("miniorange_saml");
      // SCIM Server Configuration fieldset.
      $form['scim_config_fieldset'] = [
          '#type' => 'details',
          '#title' => $this->t('Configure Drupal for SCIM or REST API-Based Provisioning'),
          '#attributes' => [
              'class' => ['scim-config-container'],
          ],
          '#open' => TRUE,
      ];

      // Get selected client and provisioning method from form state.
      $selected_client = $form_state->getValue('select_client', '');
      $selected_provisioning = $form_state->getValue('provisioning_method', '');

      // Build radio button options based on selected IdP.
      $provisioning_options = [];
      // Okta, Azure, Salesforce support both SCIM and API.
      if (in_array($selected_client, ['okta', 'azure', 'salesforce'])) {
          $provisioning_options['scim'] = $this->t('SCIM based Provisioning');
          $provisioning_options['api'] = $this->t('API based Provisioning');
      }
      elseif (in_array($selected_client, ['aws', 'auth0', 'cyberark', 'onelogin', 'pingone', 'miniorange', 'custom'])) {
          $provisioning_options['scim'] = $this->t('SCIM based Provisioning');
      }
      // Keycloak supports only API.
      elseif ($selected_client === 'keycloak') {
          $provisioning_options['api'] = $this->t('API based Provisioning');
      }

      // Auto-select if only one option is available.
      if (count($provisioning_options) === 1 && empty($selected_provisioning)) {
          $selected_provisioning = key($provisioning_options);
          $form_state->setValue('provisioning_method', $selected_provisioning);
      }

      // Application selection wrapper (for AJAX updates - includes provisioning method).
      $form['scim_config_fieldset']['application_selection_wrapper'] = [
          '#type' => 'container',
          '#attributes' => [
              'id' => 'application-selection-wrapper',
              'class' => ['application-selection-wrapper'],
          ],
      ];

      // Container for dropdown and setup guide button (inside AJAX wrapper).
      $form['scim_config_fieldset']['application_selection_wrapper']['select_client_container'] = [
          '#type' => 'container',
          '#attributes' => [
              'class' => ['select-client-container'],
              'style' => 'display: flex; align-items: flex-end; gap: 10px;',
          ],
      ];

      // Select Your Application dropdown.
      $form['scim_config_fieldset']['application_selection_wrapper']['select_client_container']['select_client'] = [
          '#type' => 'select',
          '#title' => $this->t('Select Your Application:'),
          '#options' => [
              '' => $this->t('- Select IdP -'),
              'okta' => $this->t('Okta'),
              'azure' => $this->t('Microsoft Entra ID'),
              'keycloak' => $this->t('Keycloak'),
              'salesforce' => $this->t('Salesforce'),
              'aws' => $this->t('AWS'),
              'auth0' => $this->t('Auth0'),
              'cyberark' => $this->t('CyberArk'),
              'onelogin' => $this->t('OneLogin'),
              'pingone' => $this->t('PingOne'),
              'miniorange' => $this->t('miniOrange'),
              'custom' => $this->t('Custom IdP')
          ],
          '#ajax' => [
              'callback' => [$this, 'updateApplicationSelection'],
              'wrapper' => 'application-selection-wrapper',
              'event' => 'change',
          ],
          '#default_value' => $form_state->getValue('select_client', ''),
      ];

      // Setup Guide button next to dropdown - show when provisioning method is selected.
      if (!empty($selected_provisioning)) {
          // Setup guide URLs mapping for each IdP and provisioning method.
          $setup_guide_urls = [
              'okta' => [
                  'scim' => 'https://www.drupal.org/docs/contributed-modules/user-sync-provisioning-in-drupal/user-provisioning-sync-changes-in-drupal/user-sync-changes-from-okta-to-drupal',
                  'api' => 'https://plugins.miniorange.com/drupal-okta-user-sync',
              ],
              'azure' => [
                  'scim' => 'https://www.drupal.org/docs/contributed-modules/user-sync-provisioning-in-drupal/user-provisioning-sync-changes-in-drupal/microsoft-entra-id-as-scim-client',
                  'api' => 'https://plugins.miniorange.com/setup-drupal-azure-synchronization',
              ],
              'keycloak' => [
                  'scim' => 'https://plugins.miniorange.com/drupal-keycloak-user-sync',
                  'api' => 'https://plugins.miniorange.com/drupal-keycloak-user-sync',
              ],
              'salesforce' => [
                  'scim' => 'https://plugins.miniorange.com/drupal/setup-guides?category=user-provisioning',
                  'api' => 'https://plugins.miniorange.com/drupal/setup-guides?category=user-provisioning',
              ],
              'aws' => [
                  'scim' => 'https://plugins.miniorange.com/user-provisioning-and-sync-between-aws-and-drupal',
                  'api' => 'https://plugins.miniorange.com/drupal/setup-guides?category=user-provisioning',
              ],
              'auth0' => [
                  'scim' => 'https://plugins.miniorange.com/drupal-to-auth0-user-sync',
                  'api' => 'https://plugins.miniorange.com/drupal/setup-guides?category=user-provisioning',
              ],
              'cyberark' => [
                  'scim' => 'https://plugins.miniorange.com/drupal-scim-cyberark-user-provisioning',
                  'api' => 'https://plugins.miniorange.com/drupal/setup-guides?category=user-provisioning',
              ],
              'onelogin' => [
                  'scim' => 'https://plugins.miniorange.com/drupal-scim-onelogin-user-provisioning',
                  'api' => 'https://plugins.miniorange.com/drupal/setup-guides?category=user-provisioning',
              ],
              'pingone' => [
                  'scim' => 'https://plugins.miniorange.com/drupal-scim-pingone-user-provisioning',
                  'api' => 'https://plugins.miniorange.com/drupal/setup-guides?category=user-provisioning',
              ],
              'miniorange' => [
                  'scim' => 'https://plugins.miniorange.com/drupal-to-miniorange-user-sync',
                  'api' => 'https://plugins.miniorange.com/drupal/setup-guides?category=user-provisioning',
              ],
              'custom' => [
                  'scim' => 'https://plugins.miniorange.com/drupal/setup-guides?category=user-provisioning',
                  'api' => 'https://plugins.miniorange.com/drupal/setup-guides?category=user-provisioning'
              ]
          ];

          // Get setup guide URL based on selected IdP and provisioning method.
          if (isset($setup_guide_urls[$selected_client][$selected_provisioning])) {
              $setup_guide_url = $setup_guide_urls[$selected_client][$selected_provisioning];

              if($selected_client !== 'salesforce') {
                  $form['scim_config_fieldset']['application_selection_wrapper']['select_client_container']['setup_guide'] = [
                      '#type' => 'link',
                      '#title' => $this->t('Setup Guide'),
                      '#url' => Url::fromUri($setup_guide_url, ['external' => TRUE]),
                      '#attributes' => [
                          'class' => ['button', 'button--primary'],
                          'target' => '_blank',
                          'rel' => 'noopener noreferrer',
                          'style' => 'margin-bottom: 24px;',
                      ],
                  ];
              }
              else {
                  // For Salesforce, show "Coming Soon" modal instead of redirecting to wrong URL.
                  $form['scim_config_fieldset']['application_selection_wrapper']['select_client_container']['setup_guide'] = [
                      '#type' => 'button',
                      '#value' => $this->t('Setup Guide'),
                      '#attributes' => [
                          'class' => ['button', 'button--primary'],
                          'style' => 'margin-bottom: 24px;',
                      ],
                      '#ajax' => [
                          'callback' => '::showComingSoonModal',
                          'event' => 'click',
                      ],
                  ];
              }
          }
      }

      // Radio buttons for provisioning method selection - only show if there are multiple options.
      if (count($provisioning_options) > 1) {
          $form['scim_config_fieldset']['application_selection_wrapper']['provisioning_method'] = [
              '#type' => 'radios',
              '#title' => $this->t('Select Provisioning Method:'),
              '#options' => $provisioning_options,
              '#default_value' => $selected_provisioning,
              '#ajax' => [
                  'callback' => [$this, 'updateProvisioningMethodSelection'],
                  'wrapper' => 'application-selection-wrapper',
                  'event' => 'change',
              ],
          ];
      }

      // Show SCIM fields when scim is selected.
      if (
          ($selected_provisioning === 'scim' && $selected_client !== '' && $selected_client !== 'keycloak') ||
          ($selected_provisioning === 'api' && in_array($selected_client, ['aws', 'pingone', 'cyberark', 'auth0', 'miniorange', 'custom']))
      ) {
          // SCIM Base URL.
          $form['scim_config_fieldset']['application_selection_wrapper']['scim_base_url'] = [
              '#type' => 'textfield',
              '#title' => $this->t('SCIM Base URL:'),
              '#disabled' => TRUE,
              '#default_value' => $base_url . '/scim',
              '#attributes' => [
                  'readonly' => 'readonly',
              ],
          ];

          // Get or generate SCIM Bearer Token.
          $config = $this->configFactory->getEditable('miniorange_saml.settings');
          $bearer_token = $config->get('mo_scim_bearer_token');
          if (empty($bearer_token)) {
              $bearer_token = $this->generateToken();
          }

          // SCIM Bearer Token field (textfield with password masking).
          $form['scim_config_fieldset']['application_selection_wrapper']['scim_bearer_token'] = [
              '#type' => 'textfield',
              '#title' => $this->t('SCIM Bearer Token:'),
              '#default_value' => $bearer_token,
              '#disabled' => TRUE,
              '#attributes' => [
                  'readonly' => 'readonly',
                  'id' => 'scim-bearer-token-field',
                  'class' => ['scim-bearer-token-password'],
                  'data-token-value' => $bearer_token,
              ],
          ];

          // Generate New button.
          $form['scim_config_fieldset']['application_selection_wrapper']['generate_token'] = [
              '#type' => 'submit',
              '#value' => $this->t('Generate New'),
              '#attributes' => [
                  'class' => ['button', 'button--secondary'],
              ],
              '#submit' => [[$this, 'generateTokenSubmit']],
              '#disabled' => TRUE,
          ];
      }

      // Show API fields when api is selected.
      if (($selected_provisioning === 'api' && !in_array($selected_client, ['aws', 'pingone', 'cyberark', 'auth0', 'miniorange', 'custom'])) || ($selected_provisioning === 'scim' && $selected_client === 'keycloak')) {
          // Show heading based on selected IdP.
          $api_heading = '';
          if ($selected_client === 'okta') {
              $this->buildOktaFormFields($form, ['scim_config_fieldset', 'application_selection_wrapper']);
              // $api_heading = $this->t('Okta API Form');
          }
          elseif ($selected_client === 'azure') {
              $this->buildAzureFormFields($form, ['scim_config_fieldset', 'application_selection_wrapper']);
              // $api_heading = $this->t('Azure API Form');
          }
          elseif ($selected_client === 'keycloak') {
              $this->buildKeycloakFormFields($form, ['scim_config_fieldset', 'application_selection_wrapper']);
              // $api_heading = $this->t('Keycloak API Form');
          }
          elseif ($selected_client === 'salesforce') {
              $this->buildSalesforceFormFields($form, ['scim_config_fieldset', 'application_selection_wrapper']);
              // $api_heading = $this->t('Salesforce API Form');
          }
          else {
              $api_heading = $this->t('API Form');
          }

          $form['scim_config_fieldset']['application_selection_wrapper']['api_heading'] = [
              '#markup' => Markup::create('<h2>' . $api_heading . '</h2>'),
          ];

          // Save and Test Configuration button for all API based provisioning.
          $form['scim_config_fieldset']['application_selection_wrapper']['save_and_test'] = [
              '#type' => 'submit',
              '#value' => $this->t('Save and Test Configuration'),
              '#attributes' => [
                  'class' => ['button', 'button--primary'],
              ],
              '#submit' => [[$this, 'saveAndTestConfiguration']],
              '#disabled' => TRUE,
          ];
      }
  }

  public function additionalFeatureForm(array &$form, FormStateInterface $form_state)
  {
      $form['mo_user_provisioning_scim_server_feature_setting'] = [
          '#type' => 'details',
          '#title' => t('Additional Features '),
          '#attributes' => ['class' => ['mo_details_css']],
      ];

      $form['mo_user_provisioning_scim_server_feature_setting']['set_of_radiobuttons']['miniorange_scim_options'] = [
          '#title' => t('Perform the following action on Drupal user list when user is deleted from provider:'),
          '#type' => 'radios',
          '#tree' => TRUE,
          '#options' => [
              'NONE' => t('Do Nothing'),
              'DELETE' => t('Delete Users'),
              'DEACTIVATE' => t('Deactivate Users'),
          ],
          '#disabled' => TRUE,
          '#prefix' => '<div class="container-inline">',
          '#suffix' => '</div>',
      ];


      $form['mo_user_provisioning_scim_server_feature_setting']['enable_group_provisioning'] = [
          '#title' => t('Enable Group Provisioning'),
          '#type' => 'checkbox',
          '#default_value' => 0,
          '#description' => t('Enable this option to allow SCIM to sync group updates between Drupal and your application.'),
          '#disabled' => TRUE,
      ];

      $form['mo_user_provisioning_scim_server_feature_setting']['group_modifications'] = [
          '#type' => 'fieldset',
          '#title' => t('Group Provisioning Settings'),
          '#states' => [
              'visible' => [
                  ':input[name="enable_group_provisioning"]' => ['checked' => FALSE],
              ],
          ],
      ];

      $form['mo_user_provisioning_scim_server_feature_setting']['group_modifications']['mo_user_provisioning_group_create'] = [
          '#title' => t('Create Groups'),
          '#type' => 'checkbox',
          '#default_value' => 0,
          '#description' => t('Creates a matching Drupal role when a new group is added on the SCIM client.'),
          '#disabled' => TRUE,
      ];

      $form['mo_user_provisioning_scim_server_feature_setting']['group_modifications']['mo_user_provisioning_group_delete'] = [
          '#title' => t('Delete Groups'),
          '#type' => 'checkbox',
          '#default_value' => 0,
          '#description' => t('Deletes the Drupal role when the corresponding group is removed on the SCIM client.'),
          '#disabled' => TRUE,
      ];

      $form['mo_user_provisioning_scim_server_feature_setting']['group_modifications']['mo_user_provisioning_group_update'] = [
          '#title' => t('Update Groups'),
          '#type' => 'checkbox',
          '#default_value' => 0,
          '#description' => t('Applies group changes (e.g., rename) from the SCIM client to Drupal.'),
          '#disabled' => TRUE,
      ];

      return $form;
  }

  public function attributeMappingForm(array &$form, FormStateInterface $form_state)
  {
      $form['attribute_mapping_fieldset'] = [
          '#type' => 'details',
          '#title' => $this->t('Attribute Mapping'),
          '#attributes' => [
              'class' => ['attribute-mapping-container'],
          ],
      ];

      // Description for Attribute Mapping.
      $form['attribute_mapping_fieldset']['description'] = [
          '#markup' => Markup::create('<div class="mo_user_provisioning_background_note">' . $this->t('This feature allows you to map user attributes from your Identity Provider to corresponding user profile fields in Drupal.') . '</div>'),
      ];

      // Basic attribute fields container.
      $form['attribute_mapping_fieldset']['fields_container'] = [
          '#type' => 'container',
          '#attributes' => [
              'class' => ['basic-attribute-fields'],
          ],
      ];

      // Username Attribute field.
      $form['attribute_mapping_fieldset']['fields_container']['username_attribute'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Username Attribute:'),
          '#default_value' => 'userName',
          '#disabled' => TRUE,
      ];

      // Email Attribute field.
      $form['attribute_mapping_fieldset']['fields_container']['email_attribute'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Email Attribute:'),
          '#default_value' => 'emails.0.value',
          '#disabled' => TRUE,
      ];

      // Custom Attribute Mapping subsection.
      $form['attribute_mapping_fieldset']['custom_attribute_mapping'] = [
          '#type' => 'container',
          '#attributes' => [
              'class' => ['custom-attribute-mapping'],
          ],
      ];

      $form['attribute_mapping_fieldset']['custom_attribute_mapping']['title'] = [
          '#markup' => Markup::create('<strong>' . $this->t('Custom Attribute Mapping:') . '</strong>'),
      ];

      // Custom attribute mapping table.
      $form['attribute_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_table'] = [
          '#type' => 'table',
          '#header' => [
              [
                  'data' => $this->t('SCIM Client Attribute Name'),
                  'width' => '40%',
              ],
              [
                  'data' => $this->t('Drupal Attribute'),
                  'width' => '40%',
              ],
              [
                  'data' => $this->t('Operation'),
                  'width' => '20%',
              ],
          ],
      ];

      // First row of the table.
      $form['attribute_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_table'][0]['scim_client_attribute'] = [
          '#type' => 'select',
          '#options' => [
              '' => $this->t('-Select IdP Attribute-'),
          ],
          '#disabled' => TRUE,
      ];

      $form['attribute_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_table'][0]['drupal_attribute'] = [
          '#type' => 'select',
          '#options' => [
              '' => $this->t('-Select Attribute Value-'),
          ],
          '#disabled' => TRUE,
      ];

      $form['attribute_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_table'][0]['delete_button'] = [
        '#type' => 'button',
        '#value' => $this->t('Remove'),
        '#disabled' => TRUE,
      ];

      // Add button and number input.
      $form['attribute_mapping_fieldset']['custom_attribute_mapping']['attribute_mapping_add_button'] = [
          '#prefix' => '<div class="container-inline">',
          '#type' => 'button',
          '#value' => $this->t('Add'),
          '#disabled' => TRUE,
      ];

      return $form;
  }

    public function roleMappingForm(array &$form, FormStateInterface $form_state)
    {
        $form['role_mapping_fieldset'] = [
            '#type' => 'details',
            '#title' => $this->t('Role Mapping'),
            '#attributes' => [
                'class' => ['role-mapping-container'],
            ],
        ];

        // Description for Role Mapping.
        $form['role_mapping_fieldset']['description'] = [
            '#markup' => Markup::create('<div class="mo_user_provisioning_background_note">' . $this->t('This feature allows you to assign and map user roles in Drupal dynamically using user attributes received from the IdP.') . '</div>'),
        ];

        // Enable Role Mapping checkbox.
        $form['role_mapping_fieldset']['enable_role_mapping'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Enable Role Mapping'),
            '#default_value' => 0,
            '#disabled' => TRUE,
        ];

        // Update existing roles checkbox.
        $form['role_mapping_fieldset']['update_existing_roles'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Update the users existing roles'),
            '#default_value' => 0,
            '#disabled' => TRUE,
        ];

        // Select default group for new users.
        $form['role_mapping_fieldset']['default_group'] = [
            '#type' => 'select',
            '#title' => $this->t('Select default role for new users'),
            '#options' => [
                'select' => $this->t('- Select Default Role -'),
                'authenticated' => $this->t('authenticated user'),
                'administrator' => $this->t('administrator'),
                'content-editor' => $this->t('content editor')
            ],
            '#description' => $this->t('Note: This role will be assigned to user at the time of user creation in Drupal site.'),
            '#disabled' => TRUE,
        ];

        // Custom Role Mapping fieldset.
        $form['role_mapping_fieldset']['custom_role_mapping'] = [
          '#markup' => Markup::create('<strong>' . $this->t('Custom Role Mapping:') . '</strong>'),
        ];

        // Role Attribute textfield.
        $form['role_mapping_fieldset']['custom_role_mapping']['role_attribute'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Role Attribute'),
            '#attributes' => [
                'placeholder' => 'Enter Role Attribute',
            ],
            '#disabled' => TRUE,
        ];

        // Role mapping table.
        $form['role_mapping_fieldset']['custom_role_mapping']['role_mapping_table'] = [
            '#type' => 'table',
            '#header' => [
                [
                    'data' => $this->t('Drupal Roles'),
                    'width' => '40%',
                ],
                [
                    'data' => $this->t('SCIM Client Attribute Value'),
                    'width' => '40%',
                ],
                [
                    'data' => $this->t('Operation'),
                    'width' => '20%',
                ],
            ],
        ];

        // First row of the table.
        $form['role_mapping_fieldset']['custom_role_mapping']['role_mapping_table'][0]['drupal_role'] = [
            '#type' => 'select',
            '#options' => [
                '' => $this->t('- Select Drupal Role -'),
            ],
            '#disabled' => TRUE,
        ];

        $form['role_mapping_fieldset']['custom_role_mapping']['role_mapping_table'][0]['scim_attribute_value'] = [
            '#type' => 'select',
            '#options' => [
                '' => $this->t('- Select Attribute Value -'),
            ],
            '#disabled' => TRUE,
        ];

        $form['role_mapping_fieldset']['custom_role_mapping']['role_mapping_table'][0]['remove_button'] = [
            '#type' => 'button',
            '#value' => $this->t('Remove'),
            '#disabled' => TRUE,
        ];

        // Add Only button.
        $form['role_mapping_fieldset']['custom_role_mapping']['add_only_button'] = [
            '#type' => 'button',
            '#value' => $this->t('Add'),
            '#disabled' => TRUE,
        ];

        return $form;
    }

  /**
   * AJAX callback to update application selection (setup guide button and provisioning method).
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form element to replace.
   */
  public function updateApplicationSelection(array &$form, FormStateInterface $form_state) {
    // Reset provisioning method when IdP changes.
    $form_state->setValue('provisioning_method', '');
    
    return $form['scim_config_fieldset']['application_selection_wrapper'];
  }

  /**
   * AJAX callback to update provisioning method selection (SCIM or API).
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form element to replace.
   */
  public function updateProvisioningMethodSelection(array &$form, FormStateInterface $form_state) {
    return $form['scim_config_fieldset']['application_selection_wrapper'];
  }

  /**
   * Generate bearer_token and store it in variable.
   *
   * @return string
   *   The generated bearer token.
   */
  public function generateToken() {
    try {
      $bearer_token = bin2hex(random_bytes(32));
      $config = $this->configFactory->getEditable('miniorange_saml.settings');
      $config->set('mo_scim_bearer_token', $bearer_token)->save();
      return $bearer_token;
    } catch (\Exception $e) {
      $this->logger->debug($e->getMessage());
      $this->messenger->addError($this->t('An error occurred while generating the new token. Please try again.'));
      return '';
    }
  }

  /**
   * Submit handler for Generate New token button.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function generateTokenSubmit(array &$form, FormStateInterface $form_state) {
    $this->generateToken();
    $this->messenger->addMessage($this->t('New token has been generated successfully.'));
    
    // Redirect back to the same page - form state will preserve values.
    $current_route = \Drupal::routeMatch()->getRouteName();
    $url = Url::fromRoute($current_route);
    $form_state->setRedirectUrl($url);
  }

  /**
   * Submit handler for Save and Test Configuration button.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function saveAndTestConfiguration(array &$form, FormStateInterface $form_state) {
    $selected_client = $form_state->getValue('select_client', 'okta');
    
    // Get form values based on selected IdP.
    if ($selected_client === 'okta') {
      $portal_url = $form_state->getValue('okta_portal_url');
      $bearer_token = $form_state->getValue('okta_bearer_token');
      $test_email = $form_state->getValue('okta_test_email');
      
      // TODO: Save configuration and test connection.
      $this->messenger->addMessage($this->t('Okta configuration saved and tested successfully.'));
    }
    elseif ($selected_client === 'azure') {
      $tenant_id = $form_state->getValue('azure_tenant_id');
      $client_id = $form_state->getValue('azure_client_id');
      $client_secret = $form_state->getValue('azure_client_secret');
      $tenant_name = $form_state->getValue('azure_tenant_name');
      $redirect_uri = $form_state->getValue('azure_redirect_uri');
      $test_upn = $form_state->getValue('azure_test_upn');
      
      // TODO: Save configuration and test connection.
      $this->messenger->addMessage($this->t('Azure configuration saved and tested successfully.'));
    }
    elseif ($selected_client === 'keycloak') {
      $domain = $form_state->getValue('keycloak_domain');
      $client_id = $form_state->getValue('keycloak_client_id');
      $client_secret = $form_state->getValue('keycloak_client_secret');
      $realm = $form_state->getValue('keycloak_realm');
      
      // TODO: Save configuration and test connection.
      $this->messenger->addMessage($this->t('Keycloak configuration saved and tested successfully.'));
    }
    elseif ($selected_client === 'salesforce') {
      $domain = $form_state->getValue('salesforce_domain');
      $application_id = $form_state->getValue('salesforce_application_id');
      $client_secret = $form_state->getValue('salesforce_client_secret');
      $redirect_url = $form_state->getValue('salesforce_redirect_url');
      $scope = $form_state->getValue('salesforce_scope');
      $test_username = $form_state->getValue('salesforce_test_username');
      
      // TODO: Save configuration and test connection.
      $this->messenger->addMessage($this->t('Salesforce configuration saved and tested successfully.'));
    }
    
    // Redirect back to the same page - form state will preserve values.
    $current_route = \Drupal::routeMatch()->getRouteName();
    $url = Url::fromRoute($current_route);
    $form_state->setRedirectUrl($url);
  }

  /**
   * Build Keycloak form fields.
   *
   * @param array $form
   *   The form array.
   * @param array $wrapper_path
   *   The array path to the wrapper element where fields should be added.
   */
  protected function buildKeycloakFormFields(array &$form, array $wrapper_path) {
    $form_ref = &$form;
    foreach ($wrapper_path as $key) {
      $form_ref = &$form_ref[$key];
    }
    
    $form_ref['keycloak_description'] = [
      '#markup' => Markup::create('<p>' . $this->t('Configure the following settings to register your Keycloak application in Drupal.') . '</p>'),
    ];

    // Keycloak Domain field.
    $form_ref['keycloak_domain'] = [
      '#type' => 'textfield',
      '#title' => Markup::create($this->t('Keycloak Domain')),
      '#default_value' => '',
      '#disabled' => TRUE,
      '#description' => $this->t('Enter the base URL of your Keycloak server (e.g., https://keycloak.example.com)'),
      '#attributes' => [
        'placeholder' => 'https://keycloak.example.com/auth',
      ],
    ];

    // Client ID field.
    $form_ref['keycloak_client_id'] = [
      '#type' => 'textfield',
      '#title' => Markup::create($this->t('Client ID')),
      '#default_value' => '',
      '#disabled' => TRUE,
      '#description' => $this->t('Enter the client ID for authentication'),
      '#attributes' => [
        'placeholder' => 'userSync',
      ],
    ];

    // Client Secret field.
    $form_ref['keycloak_client_secret'] = [
      '#type' => 'textfield',
      '#title' => Markup::create($this->t('Client Secret')),
      '#default_value' => '',
      '#disabled' => TRUE,
      '#description' => $this->t('Enter the client Secret'),
      '#attributes' => [
        'placeholder' => 'Enter your client secret',
      ],
    ];

    // Keycloak Realm field.
    $form_ref['keycloak_realm'] = [
      '#type' => 'textfield',
      '#title' => Markup::create($this->t('Keycloak Realm')),
      '#default_value' => '',
      '#disabled' => TRUE,
      '#description' => $this->t('Enter the Keycloak Realm'),
      '#attributes' => [
        'placeholder' => 'user-sync-keycloak',
      ],
    ];
  }

  /**
   * Build Okta API form fields.
   *
   * @param array $form
   *   The form array.
   * @param array $wrapper_path
   *   The array path to the wrapper element where fields should be added.
   */
  protected function buildOktaFormFields(array &$form, array $wrapper_path) {
    $form_ref = &$form;
    foreach ($wrapper_path as $key) {
      $form_ref = &$form_ref[$key];
    }
    
    $form_ref['okta_description'] = [
      '#markup' => Markup::create('<p>' . $this->t('Configure the following settings to register your Okta application in Drupal.') . '</p>'),
    ];

    // Okta Portal URL.
    $form_ref['okta_portal_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Okta Portal URL:'),
      '#disabled' => TRUE,
      '#placeholder' => 'https://xxxxxx.okta.com',
      '#description' => $this->t('Enter the site url of your Okta Portal e.g https://xxxxxx.okta.com'),
    ];

    // Okta Bearer Token.
    $form_ref['okta_bearer_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Okta Bearer Token:'),
      '#disabled' => TRUE,
      '#description' => $this->t('You can create Token under Security → API → Tokens in your Okta portal.'),
    ];

    // Test Your Configuration.
    $form_ref['okta_test_email'] = [
      '#type' => 'textfield',
      '#disabled' => TRUE,
      '#title' => $this->t('Test Your Configuration (Enter user email:)'),
      '#description' => $this->t('Enter Email of user which is present in Okta.'),
    ];
  }

  /**
   * Build Azure API form fields.
   *
   * @param array $form
   *   The form array.
   * @param array $wrapper_path
   *   The array path to the wrapper element where fields should be added.
   */
  protected function buildAzureFormFields(array &$form, array $wrapper_path) {
    global $base_url;
    
    $form_ref = &$form;
    foreach ($wrapper_path as $key) {
      $form_ref = &$form_ref[$key];
    }

    // Tenant ID.
    $form_ref['azure_tenant_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tenant ID:'),
      '#disabled' => TRUE,
      '#description' => Markup::create($this->t('You can find the <strong>Tenant ID</strong> in the <strong>Overview</strong> tab of your Azure application.')),
    ];

    // Application (Client) ID.
    $form_ref['azure_client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Application (Client) ID:'),
      '#disabled' => TRUE,
      '#description' => Markup::create($this->t('You can find the <strong>Application ID</strong> in the <strong>Overview</strong> tab of your Azure application.')),
    ];

    // Client Secret.
    $form_ref['azure_client_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client Secret:'),
      '#disabled' => TRUE,
      '#description' => Markup::create($this->t('You can find the <strong>Client Secret</strong> value in the <strong>Certificates & Secrets</strong> tab of your Azure application.')),
    ];

    // Tenant Name/Primary Domain.
    $form_ref['azure_tenant_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tenant Name/Primary Domain:'),
      '#disabled' => TRUE,
      '#description' => Markup::create($this->t('You can find the <strong>Tenant Name</strong> under primary domain in the <strong>Overview</strong> tab of your Azure application.')),
    ];

    // Redirect URI (Optional).
    $form_ref['azure_redirect_uri'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Redirect URI: (Optional)'),
      '#default_value' => $base_url,
      '#disabled' => TRUE,
    ];

    // Test UPN/ID.
    $form_ref['azure_test_upn'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Test UPN/ID:'),
      '#description' => Markup::create($this->t('Please note that to test the configuration of your application you will need <strong>UserPrincipalName/ID</strong> of User. The <strong>User Principle Name / Object ID</strong> is present in the <strong>Users</strong> tab of your Azure Portal.')),
      '#disabled' => TRUE,
    ];
  }

  /**
   * Build Salesforce API form fields.
   *
   * @param array $form
   *   The form array.
   * @param array $wrapper_path
   *   The array path to the wrapper element where fields should be added.
   */
  protected function buildSalesforceFormFields(array &$form, array $wrapper_path) {
    global $base_url;
    
    $form_ref = &$form;
    foreach ($wrapper_path as $key) {
      $form_ref = &$form_ref[$key];
    }

    $form_ref['salesforce_description'] = [
      '#markup' => Markup::create('<p>' . $this->t('Configure the following settings to register your Salesforce application in Drupal.') . '</p>'),
    ];

    // Domain name field.
    $form_ref['salesforce_domain'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Domain name:'),
      '#default_value' => '',
      '#disabled' => TRUE,
      '#description' => $this->t('Enter your Salesforce domain name (e.g., https://yourinstance.salesforce.com)'),
      '#attributes' => [
        'placeholder' => 'https://yourinstance.salesforce.com',
      ],
    ];

    // Application ID field.
    $form_ref['salesforce_application_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Application ID:'),
      '#default_value' => '',
      '#disabled' => TRUE,
      '#description' => $this->t('Enter the Application (Client) ID from your Salesforce Connected App'),
      '#attributes' => [
        'placeholder' => 'Enter your Application ID',
      ],
    ];

    // Client Secret field.
    $form_ref['salesforce_client_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client Secret:'),
      '#default_value' => '',
      '#disabled' => TRUE,
      '#description' => $this->t('Enter the Client Secret from your Salesforce Connected App'),
      '#attributes' => [
        'placeholder' => 'Enter your Client Secret',
      ],
    ];

    // Redirect URL field.
    $form_ref['salesforce_redirect_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Redirect URL:'),
      '#default_value' => $base_url,
      '#disabled' => TRUE,
      '#description' => $this->t('Enter the Redirect URL configured in your Salesforce Connected App'),
      '#attributes' => [
        'placeholder' => $base_url,
      ],
    ];

    // Scope field.
    $form_ref['salesforce_scope'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Scope:'),
      '#default_value' => 'api refresh_token',
      '#disabled' => TRUE,
      '#description' => $this->t('Enter the OAuth scopes (e.g., api refresh_token)'),
      '#attributes' => [
        'placeholder' => 'api refresh_token',
      ],
    ];

    // Test username field.
    $form_ref['salesforce_test_username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Test username:'),
      '#disabled' => TRUE,
      '#description' => $this->t('Enter the username of a user in Salesforce to test the configuration'),
      '#attributes' => [
        'placeholder' => 'Enter test username',
      ],
    ];
  }

  /**
   * AJAX callback to show "Coming Soon" modal for Salesforce setup guide.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response with modal dialog command.
   */
  public function showComingSoonModal(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    
    // Get the selected provisioning method.
    $selected_provisioning = $form_state->getValue('provisioning_method', '');
    $provisioning_label = ($selected_provisioning === 'scim') ? 'SCIM' : 'API';
    
    $content = Markup::create(
      '<div style="padding: 20px; text-align: center;">
        <p style="font-size: 14px; margin-top: 20px;">
          If you need help in the meantime, feel free to reach out at <a href="mailto:drupalsupport@xecurify.com" style="color: #0073aa;">drupalsupport@xecurify.com</a>.
        </p>
      </div>'
    );
    
    $response->addCommand(new OpenModalDialogCommand(
      $this->t('Coming Soon'),
      $content,
      ['width' => '600px']
    ));
    
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // No form submission needed for promotional page.
  }

}