<?php declare(strict_types=1);

namespace Drupal\minisite\Plugin\Validation\Constraint;

use Drupal\Core\Archiver\ArchiverInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\file\Plugin\Validation\Constraint\BaseFileConstraintValidator;
use Drupal\minisite\Minisite;
use Drupal\minisite\MinisiteValidator;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

/**
 * Validates the minisite archive constraint.
 */
class MinisiteArchiveConstraintValidator extends BaseFileConstraintValidator implements ContainerInjectionInterface {

  public function __construct(
    protected MinisiteValidator $minisiteValidator,
  ) {}

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('minisite.validator'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function validate(mixed $value, Constraint $constraint): void {
    $file = $this->assertValueIsFile($value);
    if (!$constraint instanceof MinisiteArchiveConstraint) {
      throw new UnexpectedTypeException($constraint, MinisiteArchiveConstraint::class);
    }

    // Check the physical file exists
    if (!is_readable($file->getFileUri())) {
      throw new MissingArchiveException($file->getFileUri());
    }

    $extensions = $constraint->extensions;

    if ($this->minisiteValidator->hasInvalidArchiveExtension($file)) {
      $this->context->addViolation($constraint->archiveExtensionConstraint);
    }

    // Construct an archiver to pass to the following contraint validations
    // File URI is different from file name is this is a file being uploaded,
    // so we must provide both to correctly instantiate the archiver.
    $archiver = Minisite::getArchiver($file->getFileUri(), $file->getFilename());
    if (!$archiver) {
      $this->context->buildViolation($constraint->invalidArchiveConstraint, [
        '%file' => $file->getFilename(),
      ])->addViolation();
      return;
    }

    $this->minisiteValidator->archiver = $archiver;

    if ($this->minisiteValidator->notValidArchiveStructure()) {
      $this->context->addViolation($constraint->archiveStructureConstraint);
    }

    if ($this->minisiteValidator->isMissingIndexFile()) {
      $this->context->addViolation($constraint->archiveIndexFileConstraint);
    }

    if ($this->minisiteValidator->containsDisallowedExtension($extensions)) {
      $this->context->addViolation($constraint->fileExtensionConstraint);
    }

    if ($this->minisiteValidator->exceedsMaximumPathLength()) {
      $this->context->addViolation($constraint->filePathLengthConstraint);
    }
  }

}
