<?php

namespace Drupal\mjml_preview\Controller;

use Drupal\Component\Utility\Html;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Cache\CacheableResponse;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\Controller\EntityViewController;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\key\KeyInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\node\NodeInterface;
use Qferrer\Mjml\Http\CurlApi;
use Qferrer\Mjml\Renderer\ApiRenderer;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;

/**
 * Controller that renders a node with MJML.
 */
class MjmlController extends EntityViewController {

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  private ConfigFactoryInterface $configFactory;

  /**
   * Key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  private KeyRepositoryInterface $keyRepository;

  /**
   * Request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  private RequestStack $requestStack;

  /**
   * Entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  private EntityDisplayRepositoryInterface $displayRepository;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): MjmlController {
    $instance = parent::create($container);

    $instance->configFactory = $container->get('config.factory');
    $instance->keyRepository = $container->get('key.repository');
    $instance->requestStack = $container->get('request_stack');
    $instance->displayRepository = $container->get('entity_display.repository');

    return $instance;
  }

  /**
   * Get MJML template for a node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   Node.
   *
   * @return string
   *   MJML template.
   *
   * @throws \Exception
   */
  private function getNodeMjml(NodeInterface $node): string {
    $build = $this->view($node, 'mjml');
    $currentRequest = $this->requestStack->getCurrentRequest();
    assert($currentRequest instanceof Request);

    return Html::transformRootRelativeUrlsToAbsolute(
      $this->renderer->render($build),
      $currentRequest->getSchemeAndHttpHost()
    );
  }

  /**
   * Render MJML template to HTML.
   *
   * @param \Drupal\node\NodeInterface $node
   *   Node.
   *
   * @return \Drupal\Core\Cache\CacheableResponse
   *   Response containing rendered HTML.
   *
   * @throws \Exception
   */
  public function render(NodeInterface $node): CacheableResponse {
    $config = $this->configFactory->get('mjml_preview.settings');
    $secretKey = $this->keyRepository->getKey($config->get('secret_key'));
    assert($secretKey instanceof KeyInterface);

    $api = new CurlApi($config->get('app_id'), $secretKey->getKeyValue());
    $mjmlRenderer = new ApiRenderer($api);
    $html = $mjmlRenderer->render($this->getNodeMjml($node));

    $response = new CacheableResponse($html);
    $response->addCacheableDependency($node)
      ->addCacheableDependency($config)
      ->addCacheableDependency($secretKey);

    return $response;
  }

  /**
   * Download MJML template.
   *
   * @param \Drupal\node\NodeInterface $node
   *   Node.
   *
   * @return \Drupal\Core\Cache\CacheableResponse
   *   Response containing MJML.
   *
   * @throws \Exception
   */
  public function download(NodeInterface $node): CacheableResponse {
    $response = new CacheableResponse($this->getNodeMjml($node));
    $response->headers->set('Content-Type', 'text/html');
    $response->headers->set(
      'Content-Disposition',
      $response->headers->makeDisposition(
        ResponseHeaderBag::DISPOSITION_ATTACHMENT,
        $node->id() . '.mjml'
      )
    );
    $response->addCacheableDependency($node);

    return $response;
  }

  /**
   * Check access to MJML routes.
   *
   * @param \Drupal\node\NodeInterface $node
   *   Node.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   Access result.
   */
  public function access(NodeInterface $node): AccessResultInterface {
    $viewModes = $this->displayRepository->getViewModeOptionsByBundle($node->getEntityTypeId(), $node->bundle());
    $display = $this->displayRepository->getViewDisplay($node->getEntityTypeId(), $node->bundle(), 'mjml');

    $result = AccessResult::allowedIf(isset($viewModes['mjml']))
      ->addCacheableDependency($display);

    $config = $this->configFactory->get('mjml_preview.settings');

    return $result->andIf(
      // The route would not work if this config is not set.
      AccessResult::allowedIf($config->get('app_id') && $config->get('secret_key'))
        ->addCacheableDependency($config)
    );
  }

}
