<?php

namespace Drupal\mobile_native_share\Form;

use Drupal\Core\Entity\ContentEntityType;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Mobile Native Share settings.
 */
class MobileNativeShareSettings extends ConfigFormBase {

  /**
   * Default entity types supporting Mobile Native Share.
   */
  const DEFAULT_ENTITY_TYPES = [
    'comment',
    'node',
    'taxonomy_term',
  ];

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The entity type bundle info.
   */
  protected EntityTypeBundleInfoInterface $entityTypeBundleInfo;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): MobileNativeShareSettings {
    $instance = parent::create($container);
    $instance->moduleHandler = $container->get('module_handler');
    $instance->entityTypeBundleInfo = $container->get('entity_type.bundle.info');
    $instance->entityTypeManager = $container->get('entity_type.manager');

    return $instance;
  }

  /**
   * Provides Configuration Form name.
   */
  public function getFormId(): string {
    return 'mobile_native_share_settings';
  }

  /**
   * Provides Configuration Page name for Accessing the values.
   */
  protected function getEditableConfigNames(): array {
    return [
      'mobile_native_share.settings',
    ];
  }

  /**
   * Creates a Form for Configuring the Module.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('mobile_native_share.settings');

    $form['style'] = [
      '#type' => 'select',
      '#options' => [
        'default' => $this->t('Default'),
        'fixed-icon' => $this->t('Fixed'),
      ],
      '#default_value' => $config->get('style') ?? 'default',
      '#title' => $this->t('Display style'),
    ];

    $form['icon'] = [
      '#type' => 'textfield',
      '#default_value' => $config->get('icon'),
      '#title' => $this->t('Icon'),
    ];

    $form['entity_types'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Entity types'),
    ];

    $form['entities'] = [
      '#tree' => TRUE,
    ];

    $entities = $this->getAvailableEntities();

    foreach ($entities as $entity) {
      $entity_id = $entity->id();

      $form['entities'][$entity_id] = [
        '#type' => 'details',
        '#title' => $entity->getLabel(),
        '#group' => 'entity_types',
      ];

      // Get all available bundles for the current entity.
      $bundles = $this->entityTypeBundleInfo
        ->getBundleInfo($entity_id);

      foreach ($bundles as $machine_name => $bundle) {
        $label = $bundle['label'];
        // Some labels are TranslatableMarkup objects (such as the File entity).
        if ($label instanceof TranslatableMarkup) {
          $label = $label->render();
        }

        $form['entities'][$entity_id][$machine_name] = [
          '#type' => 'details',
          '#title' => $this->t('@entity', ['@entity' => $label]),
        ];

        $form['entities'][$entity_id][$machine_name]['enable'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Enable'),
          '#default_value' => $config->get("entities.{$entity_id}.{$machine_name}.enable"),
        ];

        $form['entities'][$entity_id][$machine_name]['title'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Title'),
          '#default_value' => $config->get("entities.{$entity_id}.{$machine_name}.title"),
        ];

        $form['entities'][$entity_id][$machine_name]['description'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Description'),
          '#default_value' => $config->get("entities.{$entity_id}.{$machine_name}.description"),
        ];
      }

    }

    if ($this->moduleHandler->moduleExists('token')) {
      $form['tokens'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['node'],
        '#global_types' => TRUE,
        '#click_insert' => TRUE,
        '#show_restricted' => FALSE,
        '#recursion_limit' => 3,
        '#text' => $this->t('Browse available tokens'),
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * Submits the Configuration Form.
   *
   * @param array $form
   *   Form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    $this->config('mobile_native_share.settings')
      ->set('style', $values['style'])
      ->set('icon', $values['icon'])
      ->set('entities', $values['entities'])
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Get available entities.
   *
   * @return \Drupal\Core\Entity\ContentEntityType[]
   *   Array of available entities.
   */
  private function getAvailableEntities(): array {
    $content_entity_types = [];
    $entity_types = self::DEFAULT_ENTITY_TYPES;
    $this->moduleHandler->alter('mobile_native_share_entity_types', $entity_types);
    $entity_type_definitions = $this->entityTypeManager->getDefinitions();
    foreach ($entity_type_definitions as $definition) {
      if ($definition instanceof ContentEntityType && in_array($definition->id(), $entity_types)) {
        $content_entity_types[] = $definition;
      }
    }

    return $content_entity_types;
  }

}
