<?php

namespace Drupal\mobile_native_share\Form;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Entity\ContentEntityType;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Mobile Native Share settings.
 */
class MobileNativeShareSettings extends ConfigFormBase {

  /**
   * Default entity types supporting Mobile Native Share.
   */
  const DEFAULT_ENTITY_TYPES = [
    'comment',
    'node',
    'taxonomy_term',
  ];

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The entity type bundle info.
   */
  protected EntityTypeBundleInfoInterface $entityTypeBundleInfo;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): MobileNativeShareSettings {
    $instance = parent::create($container);
    $instance->moduleHandler = $container->get('module_handler');
    $instance->entityTypeBundleInfo = $container->get('entity_type.bundle.info');
    $instance->entityTypeManager = $container->get('entity_type.manager');

    return $instance;
  }

  /**
   * Provides Configuration Form name.
   */
  public function getFormId(): string {
    return 'mobile_native_share_settings';
  }

  /**
   * Provides Configuration Page name for Accessing the values.
   */
  protected function getEditableConfigNames(): array {
    return [
      'mobile_native_share.settings',
    ];
  }

  /**
   * Creates a Form for Configuring the Module.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('mobile_native_share.settings');

    $form['button_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Button settings'),
      '#open' => TRUE,
    ];

    $form['button_settings']['display_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Display mode'),
      '#description' => $this->t('Choose how the share button should be rendered: with icon and text, only icon, or only text.'),
      '#options' => [
        'icon_and_text' => $this->t('Icon and text'),
        'icon_only' => $this->t('Icon only'),
        'text_only' => $this->t('Text only'),
      ],
      '#default_value' => $config->get('display_mode') ?? 'icon_and_text',
    ];

    $form['button_settings']['style'] = [
      '#type' => 'select',
      '#title' => $this->t('Button style'),
      '#description' => $this->t('Select the visual style for the button. "Default" uses standard styling, "Fixed" forces a fixed icon layout.'),
      '#options' => [
        'default' => $this->t('Default'),
        'fixed-icon' => $this->t('Fixed'),
      ],
      '#default_value' => $config->get('style') ?? 'default',
    ];

    $form['button_settings']['icon'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom icon URL'),
      '#description' => $this->t('Provide a valid URL or relative path to a custom icon image. Leave empty to use the default icon.'),
      '#default_value' => $config->get('icon'),
    ];

    $form['entity_types'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Entity types and bundles'),
      '#description' => $this->t('Select the entity types and specific bundles where the Mobile Native Share button should be available. For example, you can enable sharing for nodes, taxonomy terms, comments, or extend it to custom entities.'),
    ];

    $form['entities'] = [
      '#tree' => TRUE,
    ];

    $entities = $this->getAvailableEntities();

    foreach ($entities as $entity) {
      $entity_id = $entity->id();

      $form['entities'][$entity_id] = [
        '#type' => 'details',
        '#title' => $entity->getLabel(),
        '#group' => 'entity_types',
      ];

      // Get all available bundles for the current entity.
      $bundles = $this->entityTypeBundleInfo
        ->getBundleInfo($entity_id);

      foreach ($bundles as $machine_name => $bundle) {
        $label = $bundle['label'];
        // Some labels are TranslatableMarkup objects (such as the File entity).
        if ($label instanceof TranslatableMarkup) {
          $label = $label->render();
        }

        $form['entities'][$entity_id][$machine_name] = [
          '#type' => 'details',
          '#title' => $this->t('@entity', ['@entity' => $label]),
        ];

        $form['entities'][$entity_id][$machine_name]['enable'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Enable'),
          '#default_value' => $config->get("entities.{$entity_id}.{$machine_name}.enable"),
        ];

        $form['entities'][$entity_id][$machine_name]['title'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Title'),
          '#default_value' => $config->get("entities.{$entity_id}.{$machine_name}.title"),
        ];

        $form['entities'][$entity_id][$machine_name]['description'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Description'),
          '#default_value' => $config->get("entities.{$entity_id}.{$machine_name}.description"),
        ];
      }

    }

    if ($this->moduleHandler->moduleExists('token')) {
      $form['tokens'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['node'],
        '#global_types' => TRUE,
        '#click_insert' => TRUE,
        '#show_restricted' => FALSE,
        '#recursion_limit' => 3,
        '#text' => $this->t('Browse available tokens'),
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * Form validation handler.
   *
   * @param array $form
   *   Form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $icon = $form_state->getValue('icon');

    // Check if the icon field is not empty.
    if (!empty($icon)) {
      $safe_icon = UrlHelper::stripDangerousProtocols($icon);

      // If the stripped URL differs, it means a dangerous protocol was used.
      if ($safe_icon !== $icon) {
        $form_state->setErrorByName('icon', $this->t('The icon URL contains a dangerous protocol.'));
      }
      // If the URL is not valid according to UrlHelper, set an error.
      elseif (!UrlHelper::isValid($icon, TRUE)) {
        $form_state->setErrorByName('icon', $this->t('Please enter a valid URL or leave the field empty.'));
      }
      else {
        // Check if the URL points to an image file by extension.
        $allowed_extensions = ['png', 'jpg', 'jpeg', 'gif', 'svg'];
        $path = parse_url($icon, PHP_URL_PATH);
        $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));

        // If the extension is not one of the allowed image types, set an error.
        if (!in_array($extension, $allowed_extensions, TRUE)) {
          $form_state->setErrorByName('icon', $this->t('The icon URL must point to an image file (png, jpg, jpeg, gif, svg).'));
        }
      }

    }
    parent::validateForm($form, $form_state);
  }

  /**
   * Submits the Configuration Form.
   *
   * @param array $form
   *   Form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    $this->config('mobile_native_share.settings')
      ->set('display_mode', $values['display_mode'])
      ->set('style', $values['style'])
      ->set('icon', $values['icon'])
      ->set('entities', $values['entities'])
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Get available entities.
   *
   * @return \Drupal\Core\Entity\ContentEntityType[]
   *   Array of available entities.
   */
  private function getAvailableEntities(): array {
    $content_entity_types = [];
    $entity_types = self::DEFAULT_ENTITY_TYPES;
    $this->moduleHandler->alter('mobile_native_share_entity_types', $entity_types);
    $entity_type_definitions = $this->entityTypeManager->getDefinitions();
    foreach ($entity_type_definitions as $definition) {
      if ($definition instanceof ContentEntityType && in_array($definition->id(), $entity_types)) {
        $content_entity_types[] = $definition;
      }
    }

    return $content_entity_types;
  }

}
