<?php

namespace Drupal\model_viewer_formatter\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'model_viewer' formatter.
 *
 * @FieldFormatter(
 *   id = "model_viewer",
 *   label = @Translation("Model Viewer (Google)"),
 *   field_types = {
 *     "file"
 *   }
 * )
 */
class ModelViewerFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * Constructs a ModelViewerFormatter object.
   *
   * @param string $pluginId
   *   The plugin_id for the formatter.
   * @param mixed $pluginDefinition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $fieldDefinition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $viewMode
   *   The view mode.
   * @param array $thirdPartySettings
   *   Any third party settings.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $fileUrlGenerator
   *   The file URL generator.
   */
  public function __construct($pluginId, $pluginDefinition, FieldDefinitionInterface $fieldDefinition, array $settings, $label, $viewMode, array $thirdPartySettings, $fileUrlGenerator) {
    parent::__construct($pluginId, $pluginDefinition, $fieldDefinition, $settings, $label, $viewMode, $thirdPartySettings);
    $this->fileUrlGenerator = $fileUrlGenerator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $pluginId, $pluginDefinition) {
    return new static(
      $pluginId,
      $pluginDefinition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('file_url_generator')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'width' => '100%',
      'height' => '600px',
      'auto_rotate' => TRUE,
      'camera_controls' => TRUE,
      'background_color' => '#f0f0f0',
      'poster' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $formState) {
    $elements = parent::settingsForm($form, $formState);

    $elements['width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Width'),
      '#default_value' => $this->getSetting('width'),
      '#description' => $this->t('Width of the viewer (e.g., 100%, 800px).'),
      '#required' => TRUE,
    ];

    $elements['height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Height'),
      '#default_value' => $this->getSetting('height'),
      '#description' => $this->t('Height of the viewer (e.g., 600px, 50vh).'),
      '#required' => TRUE,
    ];

    $elements['auto_rotate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-rotate'),
      '#default_value' => $this->getSetting('auto_rotate'),
      '#description' => $this->t('Enable automatic rotation of the model.'),
    ];

    $elements['camera_controls'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Camera controls'),
      '#default_value' => $this->getSetting('camera_controls'),
      '#description' => $this->t('Enable camera controls for zooming and panning.'),
    ];

    $elements['background_color'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Background color'),
      '#default_value' => $this->getSetting('background_color'),
      '#description' => $this->t('Background color in hex format (e.g., #f0f0f0).'),
    ];

    $elements['poster'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Poster image'),
      '#default_value' => $this->getSetting('poster'),
      '#description' => $this->t('Optional URL to a poster image shown before the model loads.'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $summary[] = $this->t('Width: @width', ['@width' => $this->getSetting('width')]);
    $summary[] = $this->t('Height: @height', ['@height' => $this->getSetting('height')]);
    $summary[] = $this->getSetting('auto_rotate') ? $this->t('Auto-rotate: Enabled') : $this->t('Auto-rotate: Disabled');
    $summary[] = $this->getSetting('camera_controls') ? $this->t('Camera controls: Enabled') : $this->t('Camera controls: Disabled');
    $summary[] = $this->t('Background: @color', ['@color' => $this->getSetting('background_color')]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      $file = File::load($item->target_id);

      if (!$file) {
        continue;
      }

      $fileUri = $file->getFileUri();
      $fileUrl = $this->fileUrlGenerator->generateString($fileUri);

      // Check if file is an OBJ file.
      $mimeType = $file->getMimeType();
      $extension = pathinfo($file->getFilename(), PATHINFO_EXTENSION);

      if (!in_array(strtolower($extension), ['obj', 'gltf', 'glb'])) {
        continue;
      }

      $elements[$delta] = [
        '#theme' => 'model_viewer_formatter',
        '#file_url' => $fileUrl,
        '#width' => $this->getSetting('width'),
        '#height' => $this->getSetting('height'),
        '#auto_rotate' => $this->getSetting('auto_rotate'),
        '#camera_controls' => $this->getSetting('camera_controls'),
        '#background_color' => $this->getSetting('background_color'),
        '#poster' => $this->getSetting('poster'),
        '#attached' => [
          'library' => [
            'model_viewer_formatter/model-viewer',
          ],
        ],
      ];
    }

    return $elements;
  }

}

