<?php

namespace Drupal\moderated_content_bulk_publish;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\RevisionLogInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Session\AccountInterface;
use Drupal\moderated_content_bulk_publish\Form\SettingsForm;

/**
 * A Helper Class to assist with the publishing, archiving and unpublishing
 * bulk action.
 *   - Called by Publish Latest Revision, Unpublish Latest Revision and Draft
 * Current Revision Bulk Operations
 *   - Easy one-stop shop to make modifications to these bulk actions.
 */
class AdminModeratedContent {

  use MessengerTrait;
  use LoggerChannelTrait;

  /**
   * The entity.
   *
   * @var \Drupal\Core\Entity\EntityInterface|null $entity
   */
  private ?EntityInterface $entity;

  /**
   * The id. Default is 0.
   *
   * @var int $id
   */
  private int $id;

  /**
   * The status. Default is 0.
   *
   * @var int $status
   */
  private int $status;

  /**
   * The config for the module.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $moderatedContentBulkSettings;

  /**
   * The config for the module.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * Entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The time interface.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $dateTime;

  /**
   * The time interface.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The module handler.
   * @var \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The AdminHelper class.
   *
   * @var \Drupal\moderated_content_bulk_publish\AdminHelper
   */
  protected AdminHelper $adminHelper;

  /**
   * Construct for RevisionActionBase.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Component\Datetime\TimeInterface $dateTime
   *   The time interface.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager interface.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\moderated_content_bulk_publish\AdminHelper $adminHelper
   *   AdminHelper class.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    AccountInterface $currentUser,
    EntityTypeManagerInterface $entityTypeManager,
    TimeInterface $dateTime,
    LanguageManagerInterface $languageManager,
    ModuleHandlerInterface $moduleHandler,
    AdminHelper $adminHelper
  ) {
    $this->moderatedContentBulkSettings = $configFactory->get(SettingsForm::SETTINGS);
    $this->currentUser = $currentUser;
    $this->entityTypeManager = $entityTypeManager;
    $this->dateTime = $dateTime;
    $this->languageManager = $languageManager;
    $this->moduleHandler = $moduleHandler;
    $this->adminHelper = $adminHelper;
  }

  /**
   * Initialize values.
   *
   * @param \Drupal\Core\Entity\EntityInterface|null $entity
   *   The entity.
   * @param int $status
   *   The status.
   *
   * @return void
   */
  public function setValues(?EntityInterface $entity, int $status): void {
    $this->entity = !is_null($entity) ? $entity : NULL;
    $this->status = $status ?: 0;
    $this->id = $this->entity ? $this->entity->id() : 0;
  }

  /**
   * Draft current revision.
   */
  public function draft(bool $currentTranslationOnly = FALSE) {
    $currentLang = $this->adminHelper->getCurrentLangcode();
    $allLanguages = $this->adminHelper->getAllLanguagesToUpdate($this->entity, $currentTranslationOnly);
    foreach ($allLanguages as $langcode => $languageName) {
      if ($this->entity->hasTranslation($langcode)) {
        $this->getLogger('moderated_content_bulk_publish')
          ->notice(mb_convert_encoding("Unpublish $langcode for " . $this->id . " in moderated_content_bulk_publish", 'UTF-8'));
        $this->entity = $this->entity->getTranslation($langcode);
        $this->entity->set('moderation_state', $this->moderatedContentBulkSettings->get('workflow_state_draft'));
        if ($this->entity instanceof RevisionLogInterface) {
          $this->setRevisionData('Bulk operation create drafted revision');
        }
        $this->entity->setSyncing(TRUE);
        $this->entity->setRevisionTranslationAffected(TRUE);
        if ($this->currentUser->hasPermission('moderated content bulk draft')) {
          if ($langcode === $currentLang || $currentTranslationOnly) {
            $this->entity->save();
          }
          else {
            drupal_register_shutdown_function('Drupal\moderated_content_bulk_publish\$this->adminHelper->bulkPublishShutdown', $this->entity, $langcode, $this->moderatedContentBulkSettings->get('workflow_state_unpublished'));
          }
        }
        else {
          $this->getLogger('moderated_content_bulk_publish')
            ->notice(mb_convert_encoding("Bulk unpublish not permitted, check permissions", 'UTF-8'));
        }
      }
    }
    foreach ($allLanguages as $langcode => $languageName) {
      if ($this->entity->hasTranslation($langcode)) {
        $this->entity = $this->entity->getTranslation($langcode);
        $this->entity->set('moderation_state', $this->moderatedContentBulkSettings->get('workflow_state_draft'));
        if ($this->entity instanceof RevisionLogInterface) {
          $this->setRevisionData('Bulk operation create draft revision');
        }
        $this->entity->setSyncing(TRUE);
        $this->entity->setRevisionTranslationAffected(TRUE);
        if ($this->currentUser->hasPermission('moderated content bulk draft')) {
          if ($langcode === $currentLang || $currentTranslationOnly) {
            $this->entity->save();
          }
          else {
            drupal_register_shutdown_function('Drupal\moderated_content_bulk_publish\$this->adminHelper->bulkPublishShutdown', $this->entity, $langcode, 'draft');
          }
        }
        else {
          $this->getLogger('moderated_content_bulk_publish')
            ->notice(mb_convert_encoding("Bulk unpublish not permitted, check permissions.", 'UTF-8'));
        }
      }
    }
    return $this->entity;
  }


  /**
   * Publish Latest Revision.
   */
  public function publish(&$errorMessage = '', &$msgIsToken = '', &$msgIsPublished = '', &$msgIsAbsoluteURL = '', bool $currentTranslationOnly = FALSE) {

    // Initialize.
    $validateFailure = FALSE;
    $allLanguages = $this->adminHelper->getAllLanguagesToUpdate($this->entity, $currentTranslationOnly);

    foreach ($allLanguages as $langcode => $languageName) {
      if ($this->entity->hasTranslation($langcode)) {
        $this->getLogger('moderated_content_bulk_publish')
          ->notice(mb_convert_encoding("Publish latest revision $langcode for " . $this->id . " in moderated_content_bulk_publish", 'UTF-8'));
        $latest_revision = $this->getLatestRevision($this->entity, $this->entity->id(), $vid, $langcode);
        if (!$latest_revision === FALSE) {
          $this->entity = $latest_revision;
        }
        // Add a hook that allows verifications outside of moderated_content_bulk_publish.
        $this->entity = $this->entity->getTranslation($langcode);
        $bodyfields = $this->entity->getFields();
        $bodyFieldVal = NULL;
        if (isset($bodyfields['body'])) {
          $bodyfield = $bodyfields['body'];
          $bodyFieldVal = $bodyfield->getValue();
        }

        $hookObject = new HookObject();
        $hookObject->nid = $this->entity->id();
        $hookObject->bundle = $this->entity->bundle();
        $hookObject->langcode = $langcode;
        $hookObject->body_field_val = $bodyFieldVal[0]['value'] ?? NULL;
        $hookObject->validate_failure = $validateFailure;
        $this->moduleHandler->invokeAll('moderated_content_bulk_publish_verify_publish', [$hookObject]);
        if ($hookObject->validate_failure) {
          $errorMessage = $hookObject->error_message;
          $msgIsToken = $hookObject->msgdetail_isToken;
          $msgIsPublished = $hookObject->msgdetail_isPublished;
          $msgIsAbsoluteURL = $hookObject->msgdetail_isAbsoluteURL;
          return NULL;
        }

        $this->entity->set('moderation_state', $this->moderatedContentBulkSettings->get('workflow_state_published'));
        if ($this->entity instanceof RevisionLogInterface) {
          $this->setRevisionData('Bulk operation publish revision');
        }
        $this->entity->setSyncing(TRUE);
        $this->entity->setRevisionTranslationAffected(TRUE);
        if ($this->currentUser->hasPermission('moderated content bulk publish')) {
          $this->entity->save();
        }
        else {
          $this->getLogger('moderated_content_bulk_publish')
            ->notice(mb_convert_encoding("Bulk publish not permitted, check permissions.", 'UTF-8'));
        }
      }
    }
    return $this->entity;
  }

  /**
   * Unpublish current revision.
   */
  public function unpublish(&$errorMessage = '', &$markup = '', bool $currentTranslationOnly = TRUE) {

    $allLanguages = $this->adminHelper->getAllLanguagesToUpdate($this->entity, $currentTranslationOnly);

    // Initialize.
    foreach ($allLanguages as $langcode => $languageName) {
      if ($this->entity->hasTranslation($langcode)) {
        // Add a hook that allows verifications outside of moderated_content_bulk_publish.
        $bundle = $this->entity->bundle();
        $nid = $this->entity->id();
        $hookObject = new HookObject();
        $hookObject->nid = $nid;
        $hookObject->bundle = $bundle;
        $hookObject->langcode = $langcode;
        $hookObject->show_button = TRUE;
        $hookObject->markup = $markup;
        $hookObject->error_message = $errorMessage;
        $this->moduleHandler->invokeAll('moderated_content_bulk_publish_verify_' . $this->moderatedContentBulkSettings->get('workflow_state_unpublished'), [$hookObject]);
        if (!$hookObject->show_button) {
          $markup = $hookObject->markup;
          $errorMessage = $hookObject->error_message;
          return NULL;
        }
        $this->getLogger('moderated_content_bulk_publish')
          ->notice(mb_convert_encoding("Unpublish $langcode for " . $this->id . " in moderated_content_bulk_publish", 'UTF-8'));
        $this->entity = $this->entity->getTranslation($langcode);
        $this->entity->set('moderation_state', $this->moderatedContentBulkSettings->get('workflow_state_unpublished'));
        if ($this->entity instanceof RevisionLogInterface) {
          $this->setRevisionData('Bulk operation create unpublished revision');
        }
        // $this->entity->setSyncing(TRUE);  Removing and using shutdown call to complete save of alt lang.

        $this->entity->setRevisionTranslationAffected(TRUE);
        if ($this->currentUser->hasPermission('moderated content bulk unpublish')) {
          $this->entity->save();
        }
        else {
          $this->getLogger('moderated_content_bulk_publish')
            ->notice(mb_convert_encoding("Bulk unpublished not permitted, check permissions", 'UTF-8'));
        }
      }
    }
    return $this->entity;
  }

  /**
   * Change owner current revision.
   */
  public function changeOwner($currentTranslationOnly = FALSE, $ownerUid = 0) {

    $this->getLogger('moderated_content_bulk_publish')
      ->notice(mb_convert_encoding('Change owner action in moderated_content_bulk_publish', 'UTF-8'));
    $allLanguages = $this->adminHelper->getAllLanguagesToUpdate($this->entity, $currentTranslationOnly);

    foreach ($allLanguages as $langcode => $languageName) {
      if ($this->entity->hasTranslation($langcode)) {
        $this->entity = $this->entity->getTranslation($langcode);
        $this->entity->setOwnerId($ownerUid);
        if ($this->entity instanceof RevisionLogInterface) {
          $this->setRevisionData('Bulk operation change owner content');
        }
        if ($this->currentUser->hasPermission('moderated content bulk change owner')) {
          $this->entity->save();
        }
        else {
          $this->getLogger('moderated_content_bulk_publish')
            ->notice(mb_convert_encoding("Bulk change owner not permitted, check permissions.", 'UTF-8'));
        }
      }
    }
    return $this->entity;
  }

  /**
   * Save current revision.
   */
  public function saveContent($currentTranslationOnly = FALSE) {

    $this->getLogger('moderated_content_bulk_publish')
      ->notice(mb_convert_encoding('Save content action in moderated_content_bulk_publish', 'UTF-8'));
    $allLanguages = $this->adminHelper->getAllLanguagesToUpdate($this->entity, $currentTranslationOnly);

    foreach ($allLanguages as $langcode => $languageName) {
      if ($this->entity->hasTranslation($langcode)) {
        $this->entity = $this->entity->getTranslation($langcode);
        $this->entity->setChangedTime($this->dateTime->getRequestTime());
        if ($this->entity instanceof RevisionLogInterface) {
          $this->setRevisionData('Bulk operation save content');
        }
        if ($this->currentUser->hasPermission('moderated content bulk save content')) {
          $this->entity->save();
        }
        else {
          $this->getLogger('moderated_content_bulk_publish')
            ->notice(mb_convert_encoding("Bulk save content not permitted, check permissions.", 'UTF-8'));
        }
      }
    }
    return $this->entity;
  }

  /**
   * Get the latest revision.
   */
  public function getLatestRevision($entity, $entityId, &$vid, $langcode = NULL) {

    // Can be removed once we move to Drupal >= 8.6.0 , currently on 8.5.0.
    // See change record here: https://www.drupal.org/node/2942013 .
    $lang = $langcode;
    if (!isset($lang)) {
      $lang = $this->adminHelper->getCurrentLangcode();
    }
    $latestRevisionResult = $this->entityTypeManager
      ->getStorage($entity->getEntityType()->id())
      ->getQuery()
      ->latestRevision()
      ->condition($entity->getEntityType()->getKey('id'), $entityId, '=')
      ->accessCheck(TRUE)
      ->execute();
    if (count($latestRevisionResult)) {
      $nodeRevisionId = key($latestRevisionResult);
      if ($nodeRevisionId == $vid) {
        // There is no pending revision, the current revision is the latest.
        return FALSE;
      }
      $vid = $nodeRevisionId;
      $latestRevision = $this->entityTypeManager
        ->getStorage($entity->getEntityType()->id())
        ->loadRevision($nodeRevisionId);
      if ($latestRevision->language()
          ->getId() !== $lang && $latestRevision->hasTranslation($lang)) {
        $latestRevision = $latestRevision->getTranslation($lang);
      }
      return $latestRevision;
    }
    return FALSE;
  }
  /**
   * Helper function to set the Revision data.
   *
   * @return void
   */
  private function setRevisionData($msg = ''): void {
    $this->entity->setRevisionCreationTime($this->dateTime->getRequestTime());
    $this->entity->setRevisionLogMessage($msg);
    $currentUid = $this->currentUser->id();
    $this->entity->setRevisionUserId($currentUid);
  }

  /**
   * Helper function to get settings.
   *
   * @return \Drupal\Core\Config\ImmutableConfig
   *   The settings for the module.
   */
  public function getSettings(): ImmutableConfig {
    return $this->moderatedContentBulkSettings;
  }

}
