<?php

namespace Drupal\moderated_content_bulk_publish\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;

class SettingsForm extends ConfigFormBase {
  public const SETTINGS = 'moderated_content_bulk_publish.settings';
  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a \Drupal\system\ConfigFormBase object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *  The typed configuration manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [static::SETTINGS];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'moderated_content_bulk_publish_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('moderated_content_bulk_publish.settings');

    $form['disable_toolbar_language_switcher'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable toolbar language switcher'),
      '#default_value' => $config->get('disable_toolbar_language_switcher') ?? 0,
      '#description' => $this->t('Hide the language switcher in the toolbar, for sites that have more than one language.'),
    ];
    $form['enable_dialog_node_edit_form'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable dialog on node edit form'),
      '#default_value' => $config->get('enable_dialog_node_edit_form') ?? 1,
      '#description' => $this->t('It shows a confirmation dialog when publishing a node from a node form.'),
    ];
    $form['enable_dialog_admin_content'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable dialog on admin content listing'),
      '#default_value' => $config->get('enable_dialog_admin_content') ?? 1,
      '#description' => $this->t('It shows a confirmation dialog in the admin/content listing to all bulk operations.'),
    ];
    $form['current_translation_only'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Should by default only affect to the selected translation?'),
      '#default_value' => $config->get('current_translation_only') ?? 1,
      '#description' => $this->t('If checked by default it will action upon selected translations and not all translations of the selected content.'),
    ];

    /* @var \Drupal\workflows\WorkflowInterface $workflows */
    $workflows = $this->entityTypeManager->getStorage('workflow')->loadMultiple();
    $options = [];
    foreach ($workflows as $workflow) {
      $options[$workflow->id()] = $workflow->label();
    }
    $form['workflow'] = [
      '#type' => 'select',
      '#title' => t('Workflow'),
      '#description' => t('Choose the workflow used for the bulk publishing options.'),
      '#required' => TRUE,
      '#options' => $options,
      '#default_value' => $config->get('workflow') ?? [],
    ];

    if ($config->get('workflow') && isset($workflows[$config->get('workflow')])) {
      $options = [];
      foreach ($workflows[$config->get('workflow')]->getTypePlugin()->getConfiguration()['states'] as $key => $state) {
        $options[$key] = $state['label'];
      }
      $form['workflow_state_draft'] = [
        '#type' => 'select',
        '#title' => t('Draft'),
        '#description' => t('Choose the corresponding state for this action.'),
        '#required' => TRUE,
        '#options' => $options,
        '#default_value' => $config->get('workflow_state_draft') ?? [],
      ];
      $form['workflow_state_unpublished'] = [
        '#type' => 'select',
        '#title' => t('Unpublished'),
        '#description' => t('Choose the corresponding state for this action.'),
        '#required' => TRUE,
        '#options' => $options,
        '#default_value' => $config->get('workflow_state_unpublished') ?? [],
      ];
      $form['workflow_state_published'] = [
        '#type' => 'select',
        '#title' => t('Published'),
        '#description' => t('Choose the corresponding state for this action.'),
        '#required' => TRUE,
        '#options' => $options,
        '#default_value' => $config->get('workflow_state_published') ?? [],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('moderated_content_bulk_publish.settings');
    $config->setData($form_state->cleanValues()->getValues());
    $config->save();

    parent::submitForm($form, $form_state);
  }

}
