<?php

namespace Drupal\moderated_content_bulk_publish\Plugin\Action;

use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\moderated_content_bulk_publish\AdminModeratedContent;
use Drupal\moderated_content_bulk_publish\AdminPin;
use Drupal\views_bulk_operations\Action\ViewsBulkOperationsActionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\content_moderation\ModerationInformationInterface;

/**
 * RevisionActionBase action plugin base.
 *
 * Provides a base implementation for a configurable
 * and preconfigurable VBO Action plugin.
 */
abstract class RevisionActionBase extends ViewsBulkOperationsActionBase implements PluginFormInterface, ContainerFactoryPluginInterface {

  use MessengerTrait;
  use LoggerChannelTrait;

  /**
   * The config for the module.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * The AdminModeratedContent class.
   *
   * @var \Drupal\moderated_content_bulk_publish\AdminModeratedContent
   */
  protected AdminModeratedContent $adminModeratedContent;

  /**
   * The AdminPin class.
   *
   * @var \Drupal\moderated_content_bulk_publish\AdminPin
   */
  protected AdminPin $adminPin;

  /**
   * The moderation information interface.
   *
   * @var \Drupal\content_moderation\ModerationInformationInterface
   */
  protected ModerationInformationInterface $moderationInformation;

  /**
   * Construct for RevisionActionBase.
   *
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\moderated_content_bulk_publish\AdminModeratedContent $adminModeratedContent
   *   AdminModeratedContent class.
   * @param \Drupal\moderated_content_bulk_publish\AdminPin $adminPin
   *   AdminPin class.
   * @param \Drupal\content_moderation\ModerationInformationInterface $moderationInformation
   *   The moderation information interface.
   */
  public function __construct(
    array $configuration,
          $plugin_id,
          $plugin_definition,
    AccountInterface $currentUser,
    AdminModeratedContent $adminModeratedContent,
    AdminPin $adminPin,
    ModerationInformationInterface $moderationInformation
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->currentUser = $currentUser;
    $this->adminModeratedContent = $adminModeratedContent;
    $this->adminPin = $adminPin;
    $this->moderationInformation = $moderationInformation;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
      $container->get('moderated_content_bulk_publish.admin_moderated_content'),
      $container->get('moderated_content_bulk_publish.admin_pin'),
      $container->get('content_moderation.moderation_information')
    );
  }
  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $formState): array {
    $formState->setRebuild(FALSE);

    $form['current_translation_only'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Should this action be done for only the selected translations?'),
      '#default_value' => $values['current_translation_only'] ?? $this->adminModeratedContent->getSettings()->get('current_translation_only'),
      '#description' => $this->t('If unchecked it will apply to all translations of the selected content.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $formState): void {
    $this->configuration['current_translation_only'] = $formState->getValue('current_translation_only');
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, AccountInterface $account = NULL, $return_as_object = FALSE) {
    if ($object->getEntityType()->id() === 'node') {
      if ($this->moderationInformation->isModeratedEntity($object)) {
        $access = $object->access('update', $account, TRUE)
          ->andIf($object->moderation_state->access('edit', $account, TRUE));
      }
      else {
        $access = $object->access('update', $account, TRUE)
          ->andIf($object->status->access('edit', $account, TRUE));
      }

      $access = $this->checkRequiredPermissionAccess($access);
    }
    else {
      $access = AccessResult::forbidden()->setReason('The chosen Action only acts on entities of type node')->setCacheMaxAge(0);
    }

    // Other entity types may have different
    // access methods and properties.
    return $return_as_object ? $access : $access->isAllowed();
  }

  /**
   * Access check for permission defined in the annotation of the plugin.
   *
   * @param \Drupal\Core\Access\AccessResultInterface $access
   *   Current access result.
   * @return \Drupal\Core\Access\AccessResultInterface
   *   Either unchanged access object or one with checked permission.
   */
  protected function checkRequiredPermissionAccess(AccessResultInterface $access): AccessResultInterface {
    if (!empty($this->pluginDefinition['requirements']['_permission'])) {
      $access = $access->andIf(AccessResult::allowedIfHasPermission($this->currentUser, $this->pluginDefinition['requirements']['_permission']));
    }
    return $access;
  }

}
