<?php

namespace Drupal\moderated_content_bulk_publish;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\pathauto\PathautoGeneratorInterface;

use Drupal\Core\Language\LanguageInterface;

class AdminHelper {

  use MessengerTrait;
  use LoggerChannelTrait;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The pathauto generator.
   *
   * @var \Drupal\pathauto\PathautoGeneratorInterface
   */
  protected PathautoGeneratorInterface $pathautoGenerator;

  /**
   * Entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Construct for AdminHelper.
   *
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager interface.
   * @param \Drupal\pathauto\PathautoGeneratorInterface $pathautoGenerator
   *   The pathauto generator.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    LanguageManagerInterface $languageManager,
    PathautoGeneratorInterface $pathautoGenerator,
    EntityTypeManagerInterface $entityTypeManager
  ) {
    $this->languageManager = $languageManager;
    $this->pathautoGenerator = $pathautoGenerator;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * Helper function to get all enabled languages, excluding current language.
   */
  public function getOtherEnabledLanguages() {
    // Get the list of all languages
    $language = $this->languageManager->getCurrentLanguage();
    $languages = $this->languageManager->getLanguages(LanguageInterface::STATE_ALL);
    $otherLanguages = array();

    // Add each enabled language, aside from the current language to an array.
    foreach ($languages as $fieldLanguageCode => $fieldLanguage) {
      if ($fieldLanguageCode != $language->getId()) {
        $otherLanguages[$fieldLanguageCode] = $fieldLanguage->getName();
      }
    }
    return $otherLanguages;
  }

  /**
   * Helper function get default language.
   */
  public function getDefaultLangcode() {
    $language = $this->languageManager->getDefaultLanguage();
    return $language->getId();
  }

  /**
   * Helper function get current language.
   */
  public function getCurrentLangcode() {
    $language = $this->languageManager->getCurrentLanguage();
    return $language->getId();
  }

  /**
   * Helper function to get all enabled languages, including the current language.
   */
  public function getAllEnabledLanguages() {
    $otherLanguages = $this->getOtherEnabledLanguages();
    $language = $this->languageManager->getCurrentLanguage();
    // We need the current language first in the array.
    $otherLanguages = array_reverse($otherLanguages);
    $allLanguages = $otherLanguages;
    $allLanguages[$language->getId()] = $language->getName();
    $allLanguages = array_reverse($allLanguages);
    return $allLanguages;
  }

  /**
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to update.
   * @param bool $currentTranslationOnly
   *   The settings to update all translations or selected.
   * @return array
   */
  public function getAllLanguagesToUpdate(EntityInterface $entity, bool $currentTranslationOnly): array {
    if ($currentTranslationOnly) {
      return [
        $entity->language()->getId() => $entity->language()->getName(),
      ];
    }
    else {
      return $this->getAllEnabledLanguages();
    }
  }

  /**
   * Helper function for doing stuff after shutdown function to ensure previous db transaction is committed.
   * Make sure the moderation state is processed correctly.
   */
  public function bulkPublishShutdown($entity, $langcode, $moderationState) {
    if (!empty($moderationState)) {
      $vid = 0;
      $latestRevision = $this->bulkLatestRevision($entity->id(), $vid, $langcode, $entity->getEntityTypeId());
      $latestState = $moderationState;
      $latestIsValid = TRUE;
      if ($latestRevision == FALSE) {
        $latestIsValid = FALSE;
      }
      else {
        $latestState = $latestRevision->get('moderation_state')->getString();
      }
      if ($latestIsValid) {
        $latestRevision->setSyncing(TRUE);
        $latestRevision->setRevisionTranslationAffected(TRUE);
        $latestRevision->set('moderation_state', $moderationState);
        $latestRevision->save();
        // Ensure the alias gets updated.
        $this->pathautoGenerator->updateEntityAlias($latestRevision, 'insert', array('language' => $langcode));
        $this->pathautoGenerator->updateEntityAlias($latestRevision, 'update', array('language' => $langcode));
      }
      else {
        // Ensure the alias gets updated.
        $this->pathautoGenerator->updateEntityAlias($entity, 'insert', array('language' => $langcode));
        $this->pathautoGenerator->updateEntityAlias($entity, 'update', array('language' => $langcode));
      }
    }
    else {
      // Ensure the alias gets updated.
      $this->pathautoGenerator->updateEntityAlias($entity, 'insert', array('language' => $langcode));
      $this->pathautoGenerator->updateEntityAlias($entity, 'update', array('language' => $langcode));
    }
  }

/**
 * Retrieve the latest node revision of $lang.
 */
public function bulkLatestRevision($id, &$vid, $lang, $entityType = 'node') {
  $query = $this->entityTypeManager->getStorage($entityType)->getQuery();
  $query->latestRevision();
  if ($entityType === 'node') {
    $query->condition('nid', $id, '=');
  }
  if ($entityType === 'taxonomy_term') {
    $query->condition('tid', $id, '=');
  }
  if ($entityType === 'paragraph') {
    $query->condition('id', $id, '=');
  }
  if ($entityType === 'user') {
    // Likely no moderation states for user entities anyway but just in case.
    $query->condition('uid', $id, '=');
  }

  $query->accessCheck(TRUE);
  $latestRevisionResult = $query->execute();
  if (count($latestRevisionResult)) {
    $nodeRevisionId = key($latestRevisionResult);
    $vid = $nodeRevisionId;
    $latestRevision = $this->entityTypeManager->getStorage($entityType)->loadRevision($nodeRevisionId);
    if ($latestRevision->hasTranslation($lang) && $latestRevision->language()->getId() != $lang) {
      $latestRevision = $latestRevision->getTranslation($lang);
    }
    return $latestRevision;
  }
  return FALSE;
}
}

