<?php

namespace Drupal\moderated_content_bulk_publish;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\RevisionLogInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Session\AccountInterface;

/**
 * A Helper Class to assist with the pin and unpin bulk action.
 *   - Called by Pin and Unpin Content Bulk Operations
 *   - Easy one-stop shop to make modifications to these bulk actions.
 */
class AdminPin {

  use LoggerChannelTrait;

  /**
   * The entity.
   *
   * @var \Drupal\Core\Entity\EntityInterface|null $entity
   */
  private ?EntityInterface $entity;

  /**
   * The id. Default is 0.
   *
   * @var int $nid
   */
  private int $nid;

  /**
   * The status. Default is 0.
   *
   * @var int $status
   */
  private int $status;

  /**
   * The config for the module.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * Entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The time interface.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $dateTime;

  /**
   * The AdminHelper class.
   *
   * @var \Drupal\moderated_content_bulk_publish\AdminHelper
   */
  protected AdminHelper $adminHelper;

  /**
   * Construct for RevisionActionBase.
   *
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Component\Datetime\TimeInterface $dateTime
   *   The time interface.
   * @param \Drupal\moderated_content_bulk_publish\AdminHelper $adminHelper
   *   AdminHelper class.
   */
  public function __construct(
    AccountInterface $currentUser,
    EntityTypeManagerInterface $entityTypeManager,
    TimeInterface $dateTime,
    AdminHelper $adminHelper
  ) {
    $this->currentUser = $currentUser;
    $this->entityTypeManager = $entityTypeManager;
    $this->dateTime = $dateTime;
    $this->adminHelper = $adminHelper;
  }

  /**
   * Initialize values.
   *
   * @param \Drupal\Core\Entity\EntityInterface|null $entity
   *   The entity.
   * @param int $status
   *   The status.
   *
   * @return void
   */
  public function setValues(?EntityInterface $entity, int $status): void {
    $this->entity = !is_null($entity) ? $entity : NULL;
    $this->status = $status ?: 0;
    $this->nid = $this->entity ? $this->entity->id() : 0;
  }

  /**
   * Unpin current revision.
   */
  public function unpin($currentTranslationOnly = FALSE) {

    $this->getLogger('moderated_content_bulk_publish')
      ->notice(mb_convert_encoding('Unpin action in moderated_content_bulk_publish', 'UTF-8'));
    $allLanguages = $this->adminHelper->getAllLanguagesToUpdate($this->entity, $currentTranslationOnly);

    foreach ($allLanguages as $langcode => $languageName) {
      if ($this->entity->hasTranslation($langcode)) {
        $this->entity = $this->entity->getTranslation($langcode);
        $this->entity->setSticky(FALSE);
        if ($this->entity instanceof RevisionLogInterface) {
          $this->setRevisionData('Bulk operation unpin content');
        }
        if ($this->currentUser->hasPermission('moderated content bulk unpin')) {
          $this->entity->save();
        }
        else {
          $this->getLogger('moderated_content_bulk_publish')
            ->notice(mb_convert_encoding("Bulk unpin not permitted, check permissions.", 'UTF-8'));
        }
        if ($this->entity->isSticky()) {
          $this->entity = $this->entityTypeManager->getStorage($this->entity->getEntityTypeId())
            ->load($this->nid);
          $this->entity->setSticky(FALSE);
          if ($this->entity instanceof RevisionLogInterface) {
            $this->setRevisionData('Bulk operation unpin content');
          }
          if ($this->currentUser->hasPermission('moderated content bulk unpin')) {
            $this->entity->save();
          }
          else {
            $this->getLogger('moderated_content_bulk_publish')
              ->notice(mb_convert_encoding("Bulk unpin not permitted, check permissions.", 'UTF-8'));
          }
        }
      }
    }
    return $this->entity;
  }

  /**
   * Pin Content.
   */
  public function pin($currentTranslationOnly = FALSE) {

    $this->getLogger('moderated_content_bulk_publish')
      ->notice(mb_convert_encoding('Pin action in moderated_content_bulk_publish', 'UTF-8'));
    $allLanguages = $this->adminHelper->getAllLanguagesToUpdate($this->entity, $currentTranslationOnly);

    foreach ($allLanguages as $langcode => $languageName) {
      if ($this->entity->hasTranslation($langcode)) {
        $this->entity = $this->entity->getTranslation($langcode);
        $this->entity->setSticky(TRUE);
        if ($this->entity instanceof RevisionLogInterface) {
          $this->setRevisionData('Bulk operation pin content');
        }
        if ($this->currentUser->hasPermission('moderated content bulk pin')) {
          $this->entity->save();
        }
        else {
          $this->getLogger('moderated_content_bulk_publish')
            ->notice(mb_convert_encoding("Bulk pin not permitted, check permissions.", 'UTF-8'));
        }
        if (!$this->entity->isSticky()) {
          $this->entity = $this->entityTypeManager->getStorage($this->entity->getEntityTypeId())
            ->load($this->nid);
          $this->entity->setSticky(TRUE);
          if ($this->entity instanceof RevisionLogInterface) {
            $this->setRevisionData('Bulk operation pin content');
          }
          if ($this->currentUser->hasPermission('moderated content bulk pin')) {
            $this->entity->save();
          }
          else {
            $this->getLogger('moderated_content_bulk_publish')
              ->notice(mb_convert_encoding("Bulk pin not permitted, check permissions.", 'UTF-8'));
          }
          $this->entity = $this->entityTypeManager->getStorage($this->entity->getEntityTypeId())
            ->load($this->nid);
        }
      }
    }
    return $this->entity;
  }

  /**
   * Helper function to set the Revision data.
   *
   * @return void
   */
  private function setRevisionData($msg = ''): void {
    $this->entity->setRevisionCreationTime($this->dateTime->getRequestTime());
    $this->entity->setRevisionLogMessage($msg);
    $current_uid = $this->currentUser->id();
    $this->entity->setRevisionUserId($current_uid);
  }

}
