<?php

namespace Drupal\moderated_content_bulk_publish\Plugin\Action;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\moderated_content_bulk_publish\AdminModeratedContent;
use Drupal\moderated_content_bulk_publish\AdminPin;
use Drupal\node\NodeInterface;
use Drupal\user\Entity\User;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\content_moderation\ModerationInformationInterface;

/**
 * Assigns ownership of a moderated node to a user.
 *
 * @Action(
 *   id = "node_assign_owner_moderated_action",
 *   label = @Translation("Change the author of moderated content"),
 *   type = "node",
 *   requirements = {
 *     "_permission" = "moderated content bulk change owner",
 *   },
 * )
 */
class AssignOwnerModeratedNode extends RevisionActionBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $connection;

  /**
   * Constructs a new AssignOwnerNode action.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\moderated_content_bulk_publish\AdminModeratedContent $adminModeratedContent
   *   AdminModeratedContent class.
   * @param \Drupal\moderated_content_bulk_publish\AdminPin $adminPin
   *   AdminPin class.
   * @param \Drupal\content_moderation\ModerationInformationInterface $moderationInformation
   *   The moderation information interface.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    AccountInterface $currentUser,
    AdminModeratedContent $adminModeratedContent,
    AdminPin $adminPin,
    ModerationInformationInterface $moderationInformation,
    Connection $connection
  ) {

    parent::__construct(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $currentUser,
      $adminModeratedContent,
      $adminPin,
      $moderationInformation
    );

    $this->connection = $connection;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static($configuration, $plugin_id, $plugin_definition,
      $container->get('current_user'),
      $container->get('moderated_content_bulk_publish.admin_moderated_content'),
      $container->get('moderated_content_bulk_publish.admin_pin'),
      $container->get('content_moderation.moderation_information'),
      $container->get('database')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute($entity = NULL) {
    $this->getLogger('moderated_content_bulk_publish')->notice("Executing Change Owner content of " . $entity->label());

    $this->adminModeratedContent->setValues($entity, NodeInterface::NOT_PUBLISHED);
    /** @var \Drupal\node\NodeInterface $entity */
    $entity = $this->adminModeratedContent->changeOwner($this->configuration['current_translation_only'], $this->configuration['owner_uid']);

    // Check if owner has been changed.
    if ($entity->getOwnerId() !== $this->configuration['owner_uid']){
      $msg = "Something went wrong, the entity must have a new owner by this point.";
      $this->messenger()->addError(mb_convert_encoding($msg, 'UTF-8'));
      $this->getLogger('moderated_content_bulk_publish')->warning($msg);
      return $msg;
    }

    $msg = 'A new owner has been set to the node %s (all translations).';
    if ($this->configuration['current_translation_only']) {
      $msg = 'A new owner has been set to the node %s (only selected translations).';
    }
    return sprintf($msg, $entity->id());
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    $defaultConfiguration = parent::defaultConfiguration();
    $defaultConfiguration['owner_uid'] = '';
    return $defaultConfiguration;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $formState): array {
    $form = parent::buildConfigurationForm($form, $formState);
    $description = t('The username of the user to which you would like to assign ownership.');
    $count = $this->connection->query("SELECT COUNT(*) FROM {users}")->fetchField();
    $userUid = $this->configuration['owner_uid'] ?? 0;

    // Use dropdown for fewer than 200 users; textbox for more than that.
    if ((int) $count < 200) {
      $options = [];
      $result = $this->connection->query("SELECT [uid], [name] FROM {users_field_data} WHERE [uid] > 0 AND [default_langcode] = 1 ORDER BY [name]");
      foreach ($result as $data) {
        $options[$data->uid] = $data->name;
      }
      $form['owner_uid'] = [
        '#type' => 'select',
        '#title' => t('Username'),
        '#default_value' => $userUid,
        '#options' => $options,
        '#description' => $description,
      ];
    }
    else {
      $defaultUser = $userUid ? User::load($userUid) : NULL;
      $form['owner_uid'] = [
        '#type' => 'entity_autocomplete',
        '#title' => t('Username'),
        '#target_type' => 'user',
        '#selection_settings' => [
          'include_anonymous' => FALSE,
        ],
        '#default_value' => $defaultUser,
        // Validation is done in static::validateConfigurationForm().
        '#validate_reference' => FALSE,
        '#size' => '50',
        '#maxlength' => '60',
        '#description' => $description,
      ];
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $formState): void {
    parent::validateConfigurationForm($form, $formState);
    $exists = (bool) $this->connection->queryRange('SELECT 1 FROM {users_field_data} WHERE [uid] = :uid AND [default_langcode] = 1', 0, 1, [':uid' => $formState->getValue('owner_uid')])->fetchField();
    if (!$exists) {
      $formState->setErrorByName('owner_uid', t('Enter a valid username.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $formState): void {
    parent::submitConfigurationForm($form, $formState);
    $this->configuration['owner_uid'] = $formState->getValue('owner_uid');
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, AccountInterface $account = NULL, $return_as_object = FALSE) {
    /** @var \Drupal\node\NodeInterface $object */
    $owner = $object->getOwner();
    $result = $object->access('update', $account, TRUE)
      ->andIf($object->access('update', $owner, TRUE));

    $result = $this->checkRequiredPermissionAccess($result);

    return $return_as_object ? $result : $result->isAllowed();
  }

}
