(function ($, Drupal, drupalSettings) {
  "use strict";

  Drupal.behaviors.moduleFileEditor = {
    attach: function (context, settings) {
      if (
        $("#editor", context).length &&
        !$("#editor").hasClass("ace-initialized")
      ) {
        $("#editor").addClass("ace-initialized");

        // Wait for Ace to be available
        function initializeAceEditor() {
          if (typeof ace === "undefined") {
            setTimeout(initializeAceEditor, 100);
            return;
          }

          const editorSettings = drupalSettings.moduleFileEditor || {};

          // Ensure Ace is properly configured
          if (editorSettings.aceBaseUrl) {
            ace.config.set("basePath", editorSettings.aceBaseUrl);
            ace.config.set("modePath", editorSettings.aceBaseUrl);
            ace.config.set("themePath", editorSettings.aceBaseUrl);
            ace.config.set("workerPath", editorSettings.aceBaseUrl);
          }

          const editor = ace.edit("editor", {
            theme: "ace/theme/monokai",
            mode: "ace/mode/php",
            autoScrollEditorIntoView: true,
            useWorker: true,
          });

          initializeUI(editor);

          if (editorSettings.selectedFile) {
            loadFileContent(
              editor,
              editorSettings.selectedModule,
              editorSettings.selectedFile
            );
          }

          function displayMessage(message, type) {
            // Remove any existing messages first
            $(".custom-message").remove();

            // Create simple message element
            const messageElement = $(
              '<div class="custom-message messages--' +
                type +
                '" style="padding: 10px; margin: 10px 0; border-radius: 4px; color: #fff; background: ' +
                (type === "error" ? "#dc3545" : "#28a745") +
                ';">' +
                message +
                "</div>"
            );

            // Insert before the editor wrapper
            $(".module-editor-wrapper").before(messageElement);

            // Auto-remove after 5 seconds
            setTimeout(() => {
              messageElement.fadeOut(300, function () {
                $(this).remove();
              });
            }, 5000);
          }

          function getEditorModeForFile(filename) {
            const extension = filename.split(".").pop().toLowerCase();
            switch (extension) {
              case "css":
                return "ace/mode/css";

              case "js":
                return "ace/mode/javascript";

              case "yml":
              case "yaml":
                return "ace/mode/yaml";

              case "twig":
                return "ace/mode/twig";

              case "html":
                return "ace/mode/html";

              case "md":
                return "ace/mode/markdown";

              case "info":
                return "ace/mode/php";

              default:
                return "ace/mode/php";
            }
          }

function collapseAllFolders() {
  $(".folder-header").each(function () {
    const $folder = $(this).parent();
    const $icon = $(this).find(".folder-icon");
    $icon.removeClass("fa-folder-open").addClass("fa-folder");
    $folder.removeClass("expanded");
    $folder.find("> .folder-content").hide();
  });
}

          function initializeUI(editor) {
            $(document).on("change", "#module-selector", function () {
              const module = $(this).val();
              updateModuleFiles(module);
            });

$(document).on("click", ".folder-header", function (e) {
  e.stopPropagation();
  const $folder = $(this).parent();
  const $folderContent = $folder.find("> .folder-content");
  $folder.toggleClass("expanded");
  $folderContent.stop(true, true).slideToggle(200);

  const $icon = $(this).find(".folder-icon");
  $icon.toggleClass("fa-folder fa-folder-open");
});

            $(document).on("click", ".file-link", function (e) {
              e.preventDefault();
              loadFileContent(
                editor,
                $(this).data("module"),
                $(this).data("file")
              );
            });

            $("#save-button").on("click", function () {
              const activeFile = $(".file-link.active");
              if (activeFile.length) {
                saveFileContent(
                  activeFile.data("module"),
                  activeFile.data("file"),
                  editor.getValue()
                );
              } else {
                displayMessage("Please select a file first.", "error");
              }
            });

            // Initial collapse
            collapseAllFolders();
          }

          function updateModuleFiles(module) {
            const $fileTreeContainer = $(".file-tree-container");
            $fileTreeContainer.html(
              '<div class="loading-message">Loading files...</div>'
            );

            const url = new URL(window.location);
            url.searchParams.set("module", module);
            window.history.pushState({}, "", url);

            $.ajax({
              url: Drupal.url("module-file-editor/file-list"),
              method: "GET",
              data: { module: module },
              success: function (response) {
                if (response.status === "success") {
                  $fileTreeContainer.html(response.html);
                  editor.setValue("");
                  $(".file-link").removeClass("active");

                  // Re-initialize folders after file list reload
                  collapseAllFolders();
                }
              },
              error: function (xhr, status, error) {
                console.error("Error loading module files:", error);
                displayMessage("Failed to load module files.", "error");
              },
            });
          }

          function loadFileContent(editor, module, filePath) {
            if (!module || !filePath) {
              return;
            }

            $(".file-link").removeClass("active");
            $(`.file-link[data-file="${filePath}"]`).addClass("active");

            const mode = getEditorModeForFile(filePath);
            editor.session.setMode(mode);

            $.ajax({
              url: Drupal.url("module-file-editor/load-file-content"),
              method: "GET",
              data: { module: module, file: filePath },
              success: function (response) {
                if (response.status === "success") {
                  editor.setValue(response.content, -1);
                  const url = new URL(window.location);
                  url.searchParams.set("file", filePath);
                  window.history.pushState({}, "", url);
                }
              },
              error: function (xhr, status, error) {
                console.error("Error loading file content:", error);
                displayMessage("Failed to load file content.", "error");
              },
            });
          }

          function saveFileContent(module, filePath, content) {
            const $saveButton = $("#save-button");
            $saveButton.prop("disabled", true).text("Saving...");
            $(".custom-message").remove(); // Clear existing messages

            $.ajax({
              url: Drupal.url("module-file-editor/save-file-content"),
              method: "POST",
              data: {
                module: module,
                file: filePath,
                content: content,
              },
              success: function (response) {
                $saveButton.prop("disabled", false).text("Save File");
                if (response.status === "success") {
                  displayMessage("File saved successfully.", "status");
                } else {
                  displayMessage(
                    "Error: " + (response.message || "Unknown error"),
                    "error"
                  );
                }
              },
              error: function (xhr, status, error) {
                $saveButton.prop("disabled", false).text("Save File");
                console.error("Error saving file:", error);
                displayMessage("Error saving file: " + error, "error");
              },
            });
          }
        }

        // Start initialization
        initializeAceEditor();
      }
    },
  };
})(jQuery, Drupal, drupalSettings);
