<?php

namespace Drupal\module_file_editor\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\Core\Render\Markup;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Extension\ModuleExtensionList;

class ModuleFileEditorController extends ControllerBase
{

  protected $moduleHandler;

  public function __construct(ModuleHandlerInterface $module_handler)
  {
    $this->moduleHandler = $module_handler;
  }

  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('module_handler')
    );
  }

  public function edit(Request $request)
  {
    // Get all installed modules
    $module_handler = \Drupal::service('module_handler');
    $extension_list = \Drupal::service('extension.list.module');
    $module_options = [];
    $module_files = [];

    foreach ($module_handler->getModuleList() as $module_name => $module_data) {
      $module_options[$module_name] = $module_data->info['name'] ?? ucfirst($module_name);
      $module_path = $extension_list->getPath($module_name);
      $module_files[$module_name] = $this->getFilesInDirectory($module_path);
    }

    $selected_module = $request->get('module') ?: array_key_first($module_options);
    $selected_file = $request->get('file');

    // Get paths for Ace Editor - ensure correct path
    $current_module_path = $extension_list->getPath('module_file_editor');
    $ace_editor_url = base_path() . $current_module_path . '/ace-builds/src-min-noconflict/ace.js';
    $ace_base_url = base_path() . $current_module_path . '/ace-builds/src-min-noconflict';

    // Build module dropdown
    $module_dropdown = '<select id="module-selector" class="form-select">';
    foreach ($module_options as $module_name => $module_label) {
      $selected = $module_name === $selected_module ? ' selected' : '';
      $module_dropdown .= '<option value="' . $module_name . '"' . $selected . '>' . $module_label . '</option>';
    }
    $module_dropdown .= '</select>';

    // Build editor markup
    $editor_markup = '
      <div class="module-editor-wrapper">
        <div class="module-selection">
          <label for="module-selector">Select Module:</label>
          ' . $module_dropdown . '
        </div>
        
        <div class="editor-content-wrapper">
          <div class="file-tree-panel">
            <div class="file-tree-header">Module Files</div>
            <div class="file-tree-container">
              ' . $this->renderFileTree($module_files[$selected_module], $selected_module) . '
            </div>
          </div>
          
          <div class="editor-container">
            <div id="editor"></div>
            <button id="save-button" class="button button--primary">Save File</button>
          </div>
        </div>
      </div>';

    $build = [
      '#title' => $this->t('Module File Editor'),
      '#markup' => Markup::create($editor_markup),
      '#attached' => [
        'library' => [
          'module_file_editor/module_file_editor',
          'core/drupal.dialog.ajax',
          'core/jquery.ui.sortable'
        ],
        'drupalSettings' => [
          'moduleFileEditor' => [
            'selectedModule' => $selected_module,
            'selectedFile' => $selected_file,
            'aceEditorUrl' => $ace_editor_url,
            'aceBaseUrl' => $ace_base_url
          ],
        ],
      ],
    ];

    return $build;
  }

  private function renderFileTree(array $files, string $module): string
  {
    $tree = [];
    foreach ($files as $filePath) {
      $parts = explode('/', str_replace('\\', '/', $filePath));
      $current = &$tree;
      foreach ($parts as $part) {
        if (!isset($current[$part])) {
          $current[$part] = [];
        }
        $current = &$current[$part];
      }
    }
    return $this->buildTreeHtml($tree, $module);
  }

  private function buildTreeHtml(array $tree, string $module, string $path = ''): string
  {
    $html = '<ul class="file-tree">';
    foreach ($tree as $name => $children) {
      $currentPath = $path ? $path . '/' . $name : $name;
      if (is_array($children) && !empty($children)) {
        $html .= '<li class="folder">';
        $html .= '<div class="folder-header">';
        $html .= '<i class="folder-icon fas fa-folder"></i>';
        $html .= '<span class="folder-name">' . htmlspecialchars($name) . '</span>';
        $html .= '</div>';
        $html .= '<ul class="folder-content" style="display: none;">';
        $html .= $this->buildTreeHtml($children, $module, $currentPath);
        $html .= '</ul>';
        $html .= '</li>';
      } else {
        $html .= '<li class="file">';
        $html .= '<a href="#" class="file-link" data-module="' . htmlspecialchars($module) . '" data-file="' . htmlspecialchars($currentPath) . '">';
        $html .= '<i class="file-icon fas fa-file"></i>';
        $html .= '<span class="file-name">' . htmlspecialchars($name) . '</span>';
        $html .= '</a>';
        $html .= '</li>';
      }
    }
    $html .= '</ul>';
    return $html;
  }

  public function getFileList(Request $request)
  {
    $module = $request->query->get('module');
    if (!$module) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Module not provided.',
      ], 400);
    }

    $extension_list = \Drupal::service('extension.list.module');
    $module_path = $extension_list->getPath($module);
    $files = $this->getFilesInDirectory($module_path);

    return new JsonResponse([
      'status' => 'success',
      'html' => $this->renderFileTree($files, $module),
    ]);
  }

  public function loadFileContent(Request $request)
  {
    $module = $request->query->get('module');
    $file = $request->query->get('file');

    if (!$module || !$file) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Module or file not provided.',
      ], 400);
    }

    $extension_list = \Drupal::service('extension.list.module');
    $module_path = $extension_list->getPath($module);
    $full_path = $module_path . '/' . $file;

    if (!file_exists($full_path) || !is_readable($full_path)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'File not found or not readable.',
      ], 404);
    }

    try {
      $content = file_get_contents($full_path);
      return new JsonResponse([
        'status' => 'success',
        'content' => $content,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Error reading file: ' . $e->getMessage(),
      ], 500);
    }
  }

  private function getFilesInDirectory($path)
  {
    $files = [];
    if (file_exists($path) && is_dir($path)) {
      $iterator = new \RecursiveIteratorIterator(
        new \RecursiveDirectoryIterator($path, \RecursiveDirectoryIterator::SKIP_DOTS),
        \RecursiveIteratorIterator::SELF_FIRST
      );

      $excluded_extensions = ['png', 'jpg', 'jpeg', 'gif', 'svg', 'webp', 'bmp', 'ico', 'tiff', 'mp3', 'mp4', 'mov', 'avi', 'wmv', 'flv', 'webm', 'ogg'];

      foreach ($iterator as $file) {
        if ($file->isFile()) {
          $extension = strtolower(pathinfo($file->getFilename(), PATHINFO_EXTENSION));
          if (!in_array($extension, $excluded_extensions)) {
            $relativePath = str_replace(
              str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $path) . DIRECTORY_SEPARATOR,
              '',
              str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $file->getPathname())
            );
            $files[] = $relativePath;
          }
        }
      }
    }
    return $files;
  }

  public function saveFileContent(Request $request)
  {
    try {
      $module = $request->request->get('module');
      $file = $request->request->get('file');
      $content = $request->request->get('content');

      if (empty($module) || empty($file) || $content === null) {
        return new JsonResponse(['status' => 'error', 'message' => 'Missing parameters'], 400);
      }

      $extension_list = \Drupal::service('extension.list.module');
      $module_path = $extension_list->getPath($module);
      $full_path = $module_path . '/' . $file;

      if (!file_exists($full_path) || !is_writable($full_path)) {
        return new JsonResponse(['status' => 'error', 'message' => 'File not found or not writable'], 400);
      }

      if (file_put_contents($full_path, $content) === false) {
        return new JsonResponse(['status' => 'error', 'message' => 'Failed to save file'], 500);
      }

      return new JsonResponse(['status' => 'success']);
    } catch (\Exception $e) {
      return new JsonResponse(['status' => 'error', 'message' => $e->getMessage()], 500);
    }
  }
}
