<?php

namespace Drupal\module_manager\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Returns a list of community modules from Drupal.org.
 */
class CommunityModulesController extends ControllerBase {

  /**
   * Drupal.org API service.
   *
   * @var \Drupal\module_manager\Service\DrupalOrgApi
   */
  protected $api;

  /**
   * Crawler service.
   *
   * @var \Drupal\module_manager\Service\Crawler
   */
  protected $crawler;

  /**
   * Extension list service.
   *
   * @var \Drupal\Core\Extension\ExtensionListModule
   */
  protected $extensionList;

  /**
   * Module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->crawler = $container->get('module_manager.crawler');
    $instance->extensionList = $container->get('extension.list.module');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->requestStack = $container->get('request_stack');
    return $instance;
  }

  /**
   * Displays a list of community modules.
   *
   * @return array
   *   Render array for the module list page.
   */
  public function list() {
    $request = $this->requestStack->getCurrentRequest();
    $search = trim($request->query->get('search', ''));
    $category = trim($request->query->get('category', ''));

    try {
      $response = $this->crawler->crawl($search, $category);
      $modules = $response['modules'];
    }
    catch (\Exception $e) {
      return [
        '#markup' => $this->t('Error loading Drupal.org data: @msg', ['@msg' => $e->getMessage()]),
      ];
    }

    $search_form = [
      '#type' => 'form',
      '#method' => 'get',
      '#attributes' => ['class' => ['module-manager-search']],

      'info' => [
        '#type' => 'markup',
        '#markup' => '
        <div class="messages messages--info">
          <strong>Attention:</strong> Only projects that meet the following criteria are supported:
          <ul>
            <li><strong>Full project</strong> listed on Drupal.org</li>
            <li><strong>Has Security Advisory Coverage</strong> (official security coverage)</li>
            <li><strong>Has Stable Release</strong> (stable version)</li>
            <li><strong>Actively maintained</strong> by the community or its maintainers</li>
            <li><strong>Compatible with Drupal ' . explode(".", \Drupal::VERSION)[0] . '</strong></li>
          </ul>
        </div>
      ',
      ],
      'requirements' => [
        '#type' => 'fieldset',
        '#title' => $this->t('Supported Project Requirements'),
        '#collapsible' => TRUE,
        '#collapsed' => TRUE,

        'filters_wrapper' => [
          '#type' => 'container',
          '#attributes' => [
            'class' => [
              'module-filter-flex',
            ],
            'style' => 'display:flex; gap:15px; align-items:flex-end; flex-wrap:wrap;',
          ],

          'search' => [
            '#type' => 'textfield',
            '#title' => $this->t('Module'),
            '#value' => $search,
            '#size' => 30,
            '#attributes' => [
              'name' => 'search',
              'placeholder' => $this->t('Enter module name...'),
            ],
          ],

          'category' => [
            '#type' => 'select',
            '#title' => $this->t('Category'),
            '#options' => [
              '' => ' - Any -',
              'im_vid_3:13434' => 'Access control',
              'im_vid_3:193884' => 'Accessibility',
              'im_vid_3:53' => 'Administration tools',
              'im_vid_3:204588' => 'Artificial Intelligence (AI)',
              'im_vid_3:11480' => 'Automation',
              'im_vid_3:58' => 'Content display',
              'im_vid_3:57' => 'Content editing experience',
              'im_vid_3:186018' => 'Decoupled',
              'im_vid_3:59' => 'Developer tools',
              'im_vid_3:104' => 'E-commerce',
              'im_vid_3:64' => 'Import and export',
              'im_vid_3:52' => 'Integrations',
              'im_vid_3:67' => 'Media',
              'im_vid_3:97' => 'Multilingual',
              'im_vid_3:123' => 'Performance',
              'im_vid_3:36748' => 'Search engine optimization (SEO)',
              'im_vid_3:69' => 'Security',
              'im_vid_3:105' => 'Site search',
              'im_vid_3:20224' => 'Site structure',
              'im_vid_3:60' => 'User engagement',
            ],
            '#value' => $category ?? '',
            '#attributes' => [
              'name' => 'category',
            ],
          ],

          'actions' => [
            '#type' => 'actions',
            'submit' => [
              '#type' => 'submit',
              '#value' => $this->t('Search'),
              '#button_type' => 'primary',
              '#attributes' => [
                'style' => ['margin-block: var(--space-l);'],
              ],
            ],
          ],
        ],
      ],
    ];

    $rows = [];
    foreach ($modules as $module) {
      if (!$module['module']) {
        continue;
      }
      $rows[] = [
        $module['title'],
        $module['module'],
        [
          'data' => [
            '#type' => 'link',
            '#title' => $this->t('View'),
            '#url' => Url::fromRoute(
              'module_manager.community_modules_view',
              [
                'module' => $module['module'],
              ]
            ),
            '#attributes' => [
              'class' => ['button'],
            ],
          ],
        ],
      ];
    }

    $pager_links = $this->buildPagerLinks($response['pager_links']);

    $result = [
      'search_form' => $search_form,

      'table' => [
        '#type' => 'table',
        '#header' => [
          $this->t('Module name'),
          $this->t('Machine name'),
          $this->t('Link'),
        ],
        '#rows' => $rows,
        '#empty' => $this->t('No modules found.'),
      ],
    ];

    $current_page = $this->requestStack->getCurrentRequest()->query->get('page') ?? 0;

    $result['pager'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['custom-pager-wrapper'],
      ],
    ];

    $result['pager']['list'] = [
      '#theme' => 'item_list',
      '#items' => array_map(function ($link) use ($current_page) {
        $query = parse_url($link, PHP_URL_QUERY);
        parse_str($query, $params);

        $page = $params['page'] ?? 0;

        $classes = ['custom-pager-item'];
        if ((int) $page === (int) $current_page) {
          $classes[] = 'is-active';
        }

        return [
          '#type' => 'link',
          '#title' => $page,
          '#url' => Url::fromUri('internal:' . $link),
          '#attributes' => [
            'class' => $classes,
          ],
        ];
      }, $pager_links),
      '#attributes' => [
        'class' => ['custom-pager-list'],
      ],
    ];

    $result['#attached']['library'][] = 'module_manager/list';

    return $result;
  }

  /**
   * View internal project details and releases for a Drupal.org module.
   *
   * @param string $module
   *   The project machine name.
   *
   * @return array
   *   Render array for the module details page.
   */
  public function view($module) {
    try {
      $response = $this->crawler->crawlProject('https://www.drupal.org/project/' . $module);
      $project_data = $response;

      if (!$project_data) {
        throw new \Exception("Project not found on Drupal.org");
      }
    }
    catch (\Exception $e) {
      return [
        '#markup' => $this->t('Error loading project: @msg', ['@msg' => $e->getMessage()]),
      ];
    }

    // Check status of module.
    $is_installed = $this->moduleHandler->moduleExists($module);
    $module_root = DRUPAL_ROOT . '/modules/module_manager_contrib/' . $module;
    $exists = is_dir($module_root);

    $actions = [];
    $install_button = '';

    $rows = [];
    foreach ($project_data['versions'] as $version) {
      if (!$is_installed) {
        $install_button = [
          'data' => [
            '#type' => 'link',
            '#title' => $this->t('Install'),
            '#url' => Url::fromRoute(
              'module_manager.install_release_form',
              [
                'module' => $module,
                'version' => $version,
              ]
            ),
            '#attributes' => ['class' => ['button', 'button--primary']],
          ],
        ];
      }
      else {
        $install_button = ['data' => '—'];
      }

      $rows[] = [
        $version,
        [
          'data' => [
            '#type' => 'container',
            'links' => [
              [
                '#type' => 'link',
                '#title' => $this->t('Download'),
                '#url' => Url::fromUri($this->crawler->getReleaseFtpLink($module, $version)),
                '#attributes' => ['target' => '_blank', 'class' => ['button']],
              ],
              [
                '#type' => 'link',
                '#title' => $this->t('Source Code'),
                '#url' => Url::fromUri('https://git.drupalcode.org/project/' . $module . '/tree/' . $version),
                '#attributes' => ['target' => '_blank', 'class' => ['button']],
              ],
            ],
          ],
        ],
        $install_button,
      ];
    }

    $project_data['content_html'] = preg_replace_callback(
      '/<img([^>]+src=["\'])([^"\']+)(["\'])/i',
      function ($matches) {
        $prefix = 'http://drupal.org';
        if (strpos($matches[2], $prefix) === 0) {
          return $matches[0];
        }
        return '<img' . $matches[1] . $prefix . $matches[2] . $matches[3];
      },
      $project_data['content_html']
    );

    $current_uri = $this->requestStack->getCurrentRequest()->getRequestUri();

    if ($is_installed && $exists) {
      $actions[] = [
        '#type' => 'link',
        '#title' => $this->t('Disable module'),
        '#url' => Url::fromRoute('module_manager.disable_module', [
          'module' => $module,
          'redirect' => $current_uri,
        ]),
        '#attributes' => ['class' => ['button', 'button--danger']],
      ];
    }
    elseif (!$is_installed && $exists) {
      $actions[] = [
        '#type' => 'link',
        '#title' => $this->t('Remove module'),
        '#url' => Url::fromRoute('module_manager.remove_module', [
          'module' => $module,
          'redirect' => $current_uri,
        ]),
        '#attributes' => ['class' => ['button', 'button--danger']],
      ];
    }

    return [
      'header' => [
        '#type' => 'markup',
        '#markup' => '<h2>' . $project_data['title'] . '</h2>',
      ],

      'description' => [
        '#type' => 'markup',
        '#markup' => '<div>' . $project_data['content_html'] . '</div><hr>',
      ],

      'meta' => [
        '#theme' => 'item_list',
        '#items' => [
          $this->t('Machine name: @m', ['@m' => $module]),
        [
          'data' => [
            '#type' => 'link',
            '#title' => $this->t('See on Drupal.org'),
            '#url' => Url::fromUri("https://www.drupal.org/project/$module"),
            '#attributes' => ['target' => '_blank'],
          ],
        ],
        ],
      ],

      'module_actions' => [
        '#type' => 'container',
        'buttons' => $actions,
        '#attributes' => ['class' => ['module-actions-wrapper']],
      ],

      'releases_title' => [
        '#type' => 'markup',
        '#markup' => '<h3>Available releases</h3>',
      ],

      'releases_table' => [
        '#type' => 'table',
        '#header' => [
          $this->t('Version'),
          $this->t('Download'),
          $this->t('Install'),
        ],
        '#rows' => $rows,
        '#empty' => $this->t('No releases found.'),
      ],
    ];
  }

  /**
   * Build pager links for the module list.
   *
   * @param array $links
   *   Array of pager links.
   *
   * @return array
   *   Array of formatted pager links.
   */
  public function buildPagerLinks(array $links): array {
    $result = [];

    foreach ($links as $link) {
      $queryString = parse_url($link, PHP_URL_QUERY);

      if (!$queryString) {
        continue;
      }

      parse_str($queryString, $params);

      $page = $params['page'] ?? 0;
      $text = $params['text'] ?? '';

      $new_link = Url::fromRoute(
        'module_manager.community_modules',
        [],
        [
          'query' => [
            'page' => $page,
            'search' => $text,
          ],
        ]
      )->toString();

      $result[] = $new_link;
    }

    return $result;
  }

  /**
   * Lists installed releases in modules/module_manager_contrib.
   *
   * @return array
   *   Render array for the installed modules table.
   */
  public function listInstalledReleases() {
    $base_path = DRUPAL_ROOT . '/modules/module_manager_contrib';

    if (!is_dir($base_path)) {
      return [
        '#markup' => $this->t('Directory <code>modules/module_manager_contrib</code> does not exist.'),
      ];
    }

    $modules = scandir($base_path);
    $rows = [];

    foreach ($modules as $module) {
      if ($module === '.' || $module === '..') {
        continue;
      }

      $module_path = "$base_path/$module";

      if (!is_dir($module_path)) {
        continue;
      }

      $info_file = "$module_path/$module.info.yml";

      if (!file_exists($info_file)) {
        continue;
      }

      try {
        $info = Yaml::parse(file_get_contents($info_file));
      }
      catch (\Exception $e) {
        continue;
      }

      $name = $info['name'] ?? $module;
      $version = $info['version'] ?? 'unknown';

      $view_link = [
        'data' => [
          '#type' => 'link',
          '#title' => $this->t('View'),
          '#url' => Url::fromRoute(
            'module_manager.community_modules_view',
            [
              'module' => $module,
            ]
          ),
          '#attributes' => [
            'class' => ['button'],
          ],
        ],
      ];

      $rows[] = [
        $name,
        $module,
        $version,
        $view_link,
      ];
    }

    return [
      '#type' => 'table',
      '#header' => [
        $this->t('Module name'),
        $this->t('Machine name'),
        $this->t('Version'),
        $this->t('Actions'),
      ],
      '#rows' => $rows,
      '#empty' => $this->t('No modules found in modules/module_manager_contrib.'),
    ];
  }

}
