<?php

namespace Drupal\module_manager\Service;

use Drupal\Core\Extension\InfoParserInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ModuleInstallerInterface;

/**
 * Service to validate composer.json dependencies for Drupal modules.
 */
class DependencyValidator {

  /**
   * Info parser service.
   *
   * @var \Drupal\Core\Extension\InfoParserInterface
   */
  protected $infoParser;

  /**
   * Module extension list service.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $extensionList;

  /**
   * Module installer service.
   *
   * @var \Drupal\Core\Extension\ModuleInstallerInterface
   */
  protected $moduleInstaller;

  /**
   * Module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * External dependencies (non-drupal/*).
   *
   * @var array
   */
  public array $externalDependencies = [];

  /**
   * Drupal module dependencies (drupal/*).
   *
   * @var array
   */
  public array $drupalDependencies = [];

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Extension\InfoParserInterface $infoParser
   *   Info parser service.
   * @param \Drupal\Core\Extension\ModuleExtensionList $extensionList
   *   Module extension list service.
   * @param \Drupal\Core\Extension\ModuleInstallerInterface $moduleInstaller
   *   Module installer service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   Module handler service.
   */
  public function __construct(
    InfoParserInterface $infoParser,
    ModuleExtensionList $extensionList,
    ModuleInstallerInterface $moduleInstaller,
    ModuleHandlerInterface $moduleHandler,
  ) {
    $this->infoParser = $infoParser;
    $this->extensionList = $extensionList;
    $this->moduleInstaller = $moduleInstaller;
    $this->moduleHandler = $moduleHandler;
  }

  /**
   * Reads composer.json and returns drupal/* and external dependencies.
   *
   * @param string $path
   *   Path to the module.
   *
   * @return array
   *   Array with 'drupal' and 'external' dependencies.
   */
  public function parseComposerDependencies(string $path): array {
    $composer_file = $path . '/composer.json';

    if (!file_exists($composer_file)) {
      return [
        'drupal' => [],
        'external' => [],
      ];
    }

    $data = json_decode(file_get_contents($composer_file), TRUE);
    if (!$data) {
      return [
        'drupal' => [],
        'external' => [],
      ];
    }

    $drupal = [];
    $external = [];

    foreach (['require', 'require-dev'] as $section) {
      if (!isset($data[$section])) {
        continue;
      }

      foreach ($data[$section] as $pkg => $version) {
        if (str_starts_with($pkg, 'drupal/')) {
          $drupal[substr($pkg, 7)] = $version;
        }
        else {
          $external[$pkg] = $version;
        }
      }
    }

    $this->externalDependencies = $external;
    $this->drupalDependencies = $drupal;

    return [
      'drupal' => $drupal,
      'external' => $external,
    ];
  }

  /**
   * Checks if a module exists in the filesystem.
   *
   * @param string $module
   *   The module machine name.
   *
   * @return bool
   *   TRUE if exists.
   */
  public function moduleExists(string $module): bool {
    return $this->extensionList->exists($module);
  }

  /**
   * Checks if a module is installed/activated.
   *
   * @param string $module
   *   The module machine name.
   *
   * @return bool
   *   TRUE if installed.
   */
  public function moduleIsInstalled(string $module): bool {
    return $this->moduleHandler->moduleExists($module);
  }

  /**
   * Returns installed version of a module.
   *
   * @param string $module
   *   The module machine name.
   *
   * @return string|null
   *   Version string or NULL.
   */
  public function getInstalledModuleVersion(string $module): ?string {
    if (!$this->extensionList->exists($module)) {
      return NULL;
    }

    $info = $this->extensionList->getExtensionInfo($module);

    return $info['version'] ?? NULL;
  }

  /**
   * Evaluates if a version satisfies a Composer constraint.
   *
   * Supports: ^ ~ >= <= > < = * ranges separated by comma.
   *
   * @param string $version
   *   Installed version.
   * @param string $constraint
   *   Composer constraint.
   *
   * @return bool
   *   TRUE if satisfies.
   */
  public function versionSatisfies(string $version, string $constraint): bool {
    // Remove "v10.1.0".
    $version = ltrim($version, 'v');

    $groups = preg_split('/\s*,\s*/', $constraint);

    foreach ($groups as $group) {
      if ($this->evaluateConstraintGroup($version, $group)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Evaluates a constraint group.
   *
   * @param string $version
   *   Installed version.
   * @param string $group
   *   Constraint group.
   *
   * @return bool
   *   TRUE if satisfies.
   */
  private function evaluateConstraintGroup(string $version, string $group): bool {
    $patterns = preg_split('/\s+/', trim($group));

    foreach ($patterns as $expr) {
      if (!$this->compareConstraint($version, $expr)) {
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Compares a single constraint expression.
   *
   * @param string $version
   *   Installed version.
   * @param string $expr
   *   Constraint expression.
   *
   * @return bool
   *   TRUE if satisfies.
   */
  private function compareConstraint(string $version, string $expr): bool {

    if ($expr === '*' || $expr === 'x') {
      return TRUE;
    }

    // ^ version
    if (preg_match('/^\^(.+)$/', $expr, $m)) {
      $base = $m[1];
      $parts = explode('.', $base);
      $major = (int) $parts[0];

      return version_compare($version, $base, '>=')
        && version_compare($version, ($major + 1) . '.0.0', '<');
    }

    // ~ version
    if (preg_match('/^~(.+)$/', $expr, $m)) {
      $base = $m[1];
      $parts = explode('.', $base);
      $major = (int) $parts[0];
      $minor = (int) ($parts[1] ?? 0);

      return version_compare($version, $base, '>=')
        && version_compare($version, "$major." . ($minor + 1) . '.0', '<');
    }

    // Operators: >= <= > < =.
    if (preg_match('/^(>=|<=|>|<|=)?\s*(.+)$/', $expr, $m)) {
      $operator = $m[1] ?: '>=';
      $value = $m[2];

      return version_compare($version, $value, $operator);
    }

    return FALSE;
  }

  /**
   * Validates only drupal/* dependencies.
   *
   * @param string $module
   *   The module machine name.
   * @param string $path
   *   Path to the module.
   *
   * @return true|array
   *   TRUE if ok, or array of errors.
   */
  public function validateComposerDependencies(string $module, $path) {
    $deps = $this->parseComposerDependencies($path . '/' . $module);

    $drupalDeps = $deps['drupal'];
    $extDeps = $deps['external'];

    $this->externalDependencies = $extDeps;
    $this->drupalDependencies = $drupalDeps;

    if (empty($drupalDeps)) {
      return TRUE;
    }

    $errors = [];

    foreach ($drupalDeps as $req_module => $constraint) {

      // Skip core dependencies.
      if (in_array($req_module, ['core', 'core-composer-scaffold', 'core-project-message', 'core-recommended'])) {
        continue;
      }

      if (!$this->moduleExists($req_module)) {
        $errors[] = "Required module '$req_module' is missing.";
        continue;
      }

      $installed_version = $this->getInstalledModuleVersion($req_module);
      if (!$installed_version) {
        $errors[] = "Module '$req_module' has no version info.";
        continue;
      }

      if (!$this->versionSatisfies($installed_version, $constraint)) {
        $errors[] = "Module '$req_module' version '$installed_version' does NOT satisfy '$constraint'.";
      }
    }

    return empty($errors) ? TRUE : $errors;
  }

  /**
   * Validates module dependencies based on {module}.info.yml.
   *
   * @param string $module
   *   The module machine name.
   * @param string $extractDir
   *   Directory where module was extracted.
   *
   * @return string|true|array
   *   TRUE if ok, string with error message, or array of required dependencies.
   */
  public function validateInfoDependencies(string $module, string $extractDir) {

    $infoFile = $extractDir . '/' . $module . '/' . $module . '.info.yml';
    if (!file_exists($infoFile)) {
      return TRUE;
    }

    $info = $this->infoParser->parse($infoFile);

    if (empty($info['dependencies'])) {
      return TRUE;
    }

    $installed = $this->moduleHandler->getModuleList();
    $allModules = $this->extensionList->getList();
    $required_deps = [];
    foreach ($info['dependencies'] as $dependency) {

      // Supports "module" or "module:version".
      $dep = explode(':', $dependency);
      if (isset($dep[1])) {

        // Skip core dependencies.
        if (in_array($dep[1], ['core', 'core-composer-scaffold', 'core-project-message', 'core-recommended'])) {
          continue;
        }

        if ($dep[1] == 'drupal') {
          // Validate if the module exists in the system.
          if (!isset($allModules[$dep[1]])) {
            $required_deps[] = $dep[0] . ':' . $dep[1];
            $this->drupalDependencies[] = $dep[0] . ':' . $dep[1];
          }
        }
        else {
          if (!isset($installed[$dep[1]])) {
            $required_deps[] = $dep[0] . ':' . $dep[1];
            $this->externalDependencies[] = $dep[0] . ':' . $dep[1];
          }
        }
      }
    }

    if ($required_deps) {
      return $required_deps;
    }

    return TRUE;
  }

}
