<?php

namespace Drupal\module_manager\Batch;

use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Batch operations used to install a release.
 */
class ReleaseInstallerBatch {

  /**
   * Downloads the release ZIP file.
   *
   * @param string $zip_url
   *   The URL of the ZIP file.
   * @param string $module
   *   The module machine name.
   * @param string $version
   *   The module version.
   * @param array $context
   *   The batch context array.
   */
  public static function downloadRelease($zip_url, $module, $version, &$context) {
    $installer = \Drupal::service('module_manager.release_installer');
    $file_uri = $installer->downloadRelease($zip_url);

    if (!$file_uri) {
      $context['results']['error'] = t('Failed to download ZIP: @url', ['@url' => $zip_url]);
      $context['finished'] = 1;
      return;
    }

    $context['results']['file_uri'] = $file_uri;
    $context['results']['module'] = $module;
    $context['results']['version'] = $version;
  }

  /**
   * Extracts the downloaded release ZIP file.
   *
   * @param string $module
   *   The module machine name.
   * @param array $context
   *   The batch context array.
   */
  public static function extractRelease($module, &$context) {
    if (!empty($context['results']['error'])) {
      return;
    }

    $installer = \Drupal::service('module_manager.release_installer');
    $file_uri = $context['results']['file_uri'];
    $dir = $installer->extractRelease($file_uri, $module);

    if (!$dir) {
      $context['results']['error'] = t('Error extracting module @module.', ['@module' => $module]);
      return;
    }

    $context['results']['extract_dir'] = $dir;

  }

  /**
   * Activates the module.
   *
   * @param string $module
   *   The module machine name.
   * @param string $version
   *   The module version.
   * @param array $context
   *   The batch context array.
   */
  public static function activateModule($module, $version, &$context) {
    if (!empty($context['results']['error'])) {
      return;
    }
    $context['results']['module'] = $module;
    $context['results']['version'] = $version;

    if (!empty($context['results']['redirect_to_confirm'])) {
      $url = Url::fromRoute('module_manager.composer_confirm');
      return new RedirectResponse($url->toString());
    }

    try {
      \Drupal::service('module_installer')->install([$module]);
    }
    catch (\Throwable $th) {
      $message = t('Error activating module @module: @msg', [
        '@module' => $module,
        '@msg' => $th->getMessage(),
      ]);
      $context['finished'] = 1;
      $context['results']['error'] = $message;
      \Drupal::messenger()->addError($message);
    }
  }

  /**
   * Validates dependencies from info.yml file.
   *
   * @param string $module
   *   The module machine name.
   * @param array $context
   *   The batch context array.
   */
  public static function validateInfoYml($module, &$context) {
    $validator = \Drupal::service('module_manager.dependency_validator');
    $dir = $context['results']['extract_dir'];
    $result = $validator->validateInfoDependencies($module, $dir);

    if ($result !== TRUE) {
      $context['results']['error'] = $result;
      $context['results']['needs_confirmation'] = TRUE;
      $context['finished'] = 1;

      if (!empty($validator->externalDependencies)) {
        $context['results']['needs_confirmation'] = TRUE;
        $context['results']['external_dependencies'] = $validator->externalDependencies;
        return;
      }

      if (!empty($validator->drupalDependencies)) {
        $context['results']['needs_confirmation'] = TRUE;
        $context['results']['drupal_dependencies'] = $validator->drupalDependencies;
        return;
      }

      return;
    }
  }

  /**
   * Validates composer dependencies.
   *
   * @param string $module
   *   The module machine name.
   * @param array $context
   *   The batch context array.
   */
  public static function validateComposer($module, &$context) {
    if (!empty($context['results']['error'])) {
      return;
    }

    $validator = \Drupal::service('module_manager.dependency_validator');
    $dir = $context['results']['extract_dir'];
    $result = $validator->validateComposerDependencies($module, $dir);

    if ($result !== TRUE) {
      $context['results']['error'] = $result;
    }

    if (!empty($validator->externalDependencies)) {
      $context['results']['needs_confirmation'] = TRUE;
      $context['results']['external_dependencies'] = $validator->externalDependencies;
      return;
    }

    if (!empty($validator->drupalDependencies)) {
      $context['results']['needs_confirmation'] = TRUE;
      $context['results']['drupal_dependencies'] = $validator->drupalDependencies;
      return;
    }
  }

  /**
   * Batch finished callback.
   *
   * @param bool $success
   *   Whether the batch was successful.
   * @param array $results
   *   The batch results.
   * @param array $operations
   *   The batch operations.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse|null
   *   Redirect response on success, otherwise NULL.
   */
  public static function finished($success, $results, $operations) {
    $messenger = \Drupal::messenger();

    // Se o batch foi interrompido para pedir confirmação.
    if (!empty($results['redirect_to_confirm'])) {
      $url = Url::fromRoute('module_manager.composer_confirm');
      return new RedirectResponse($url->toString());
    }

    if (!$success || !empty($results['error'])) {
      $url = Url::fromRoute('module_manager.community_modules_view', [
        'module' => $results['module'],
      ]);
      return new RedirectResponse($url->toString());
    }

    if ($success && !empty($results['module']) && !empty($results['version'])) {
      $url = Url::fromRoute('module_manager.community_modules_view', [
        'module' => $results['module'],
      ]);
      $messenger->addStatus(t('Module @M installed and activated successfully.', ['@M' => $results['module']]));
      return new RedirectResponse($url->toString());
    }
  }

  /**
   * Redirects to confirmation form if needed.
   *
   * @param string $module
   *   The module machine name.
   * @param array $context
   *   The batch context array.
   */
  public static function postValidateRedirect($module, &$context) {
    if (!empty($context['results']['needs_confirmation'])) {

      $_SESSION['module_manager_composer_confirm'] = [
        'module' => $module,
        'external_dependencies' => $context['results']['external_dependencies'] ?? [],
        'drupal_dependencies' => $context['results']['drupal_dependencies'] ?? [],
        'info_dependencies' => $context['results']['info_dependencies'] ?? [],
        'extract_dir' => $context['results']['extract_dir'] ?? '',
        'version' => $context['results']['version'] ?? '',
      ];

      $context['finished'] = 1;
      $context['results']['redirect_to_confirm'] = TRUE;
    }
  }

}
