<?php

namespace Drupal\module_manager\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\module_manager\Batch\ReleaseInstallerBatch;

/**
 * Form to confirm composer dependencies before module installation.
 */
class ComposerDependenciesConfirmForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'module_manager_composer_dependencies_confirm';
  }

  /**
   * Builds the confirmation form.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $data = $_SESSION['module_manager_composer_confirm'] ?? NULL;

    if (!$data) {
      return ['#markup' => '<p>Data not found.</p>'];
    }

    $module = $data['module'];
    $version = $data['version'];
    $deps = $data['external_dependencies'];
    $drupal_deps = $data['drupal_dependencies'];

    $form['fieldset_warnings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Module Dependencies Warning'),
    ];
    if ($deps) {
      $items = [];
      foreach ($deps as $i => $pkg) {
        $items[] = "<strong>$i:$pkg</strong>";
      }

      $form['fieldset_warnings']['warning_external_deps'] = [
        '#type' => 'markup',
        '#markup' => '
        <p>The module <strong>' . $module . '</strong> depends on the following external packages:</p>
        <ul><li>' . implode('</li><li>', $items) . '</li></ul>
        ',
      ];
    }

    $drupal_deps = array_merge($drupal_deps);
    if ($drupal_deps) {
      $items = [];
      foreach ($drupal_deps as $i => $pkg) {
        $items[] = "<strong>$i:$pkg</strong>";
      }
      $form['fieldset_warnings']['warning_drupal_deps'] = [
        '#type' => 'markup',
        '#markup' => '
          <p>The module <strong>' . $module . '</strong> depends on the following drupal dependencies:</p>
          <ul><li>' . implode('</li><li>', $items) . '</li></ul>
        ',
      ];
    }

    $form['check_source_code'] =
      [
        '#type' => 'link',
        '#title' => $this->t('Check out the Source Code'),
        '#url' => Url::fromUri('https://git.drupalcode.org/project/' . $module . '/tree/' . $version),
        '#attributes' => ['target' => '_blank', 'class' => ['button', 'button--primary']],
      ];
    $form['warning_continue'] = [
      '#type' => 'markup',
      '#markup' => '
          <p><strong>Continuing may affect the stability or break your website.</strong></p>
        ',
    ];
    $form['warning_continue'] = [
      '#type' => 'markup',
      '#markup' => '
          <p><strong>Continuing may affect the stability or break your website.</strong></p>
        ',
    ];

    $form['confirm_risk'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('I confirm that I understand the risks and that this operation may break my site.'),
      '#required' => TRUE,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => Url::fromRoute('module_manager.community_modules_view', [
        'module' => $module,
      ]),
      '#attributes' => [
        'class' => ['button'],
      ],
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Continue'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * Validation: checkbox must be checked.
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if (!$form_state->getValue('confirm_risk')) {
      $form_state->setErrorByName('confirm_risk', $this->t('You must confirm that you understand the risks before continuing.'));
    }
  }

  /**
   * Submit handler.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $data = $_SESSION['module_manager_composer_confirm'];

    $batch = [
      'title' => $this->t('Installing module @module...', ['@module' => $data['module']]),
      'operations' => [
        [[ReleaseInstallerBatch::class, 'activateModule'], [$data['module'], $data['version']]],
      ],
      'finished' => [ReleaseInstallerBatch::class, 'finished'],
    ];

    unset($_SESSION['module_manager_composer_confirm']);

    batch_set($batch);
  }

}
