<?php

namespace Drupal\module_manager\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Controller for module actions (disable, remove).
 */
class ModuleActionController extends ControllerBase {

  /**
   * Module installer service.
   *
   * @var \Drupal\Core\Extension\ModuleInstallerInterface
   */
  protected $moduleInstaller;

  /**
   * Extension list service.
   *
   * @var \Drupal\Core\Extension\ExtensionListModule
   */
  protected $extensionList;

  /**
   * File system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messengerService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->moduleInstaller = $container->get('module_installer');
    $instance->extensionList = $container->get('extension.list.module');
    $instance->fileSystem = $container->get('file_system');
    $instance->requestStack = $container->get('request_stack');
    $instance->messengerService = $container->get('messenger');
    return $instance;
  }

  /**
   * Disables a module and redirects back.
   *
   * @param string $module
   *   The module machine name.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response.
   */
  public function disable($module) {
    try {
      $this->moduleInstaller->uninstall([$module]);
      $this->messengerService->addStatus($this->t('Módulo @m desativado com sucesso.', ['@m' => $module]));
    }
    catch (\Exception $e) {
      $this->messengerService->addError($e->getMessage());
    }

    $redirect = $this->requestStack->getCurrentRequest()->query->get('redirect') ?? '/admin/modules';
    return new RedirectResponse($redirect);
  }

  /**
   * Enable a module and redirects back.
   *
   * @param string $module
   *   The module machine name.
   * @param string $version
   *   The module version.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response.
   */
  public function enable($module, $version) {
    try {
      $this->moduleInstaller->install([$module], TRUE);
      $this->messengerService->addStatus($this->t('Módulo @m ativado com sucesso.', ['@m' => $module]));
    }
    catch (\Exception $e) {
      $this->messengerService->addError($e->getMessage());
    }

    $url = Url::fromRoute('module_manager.community_modules_view', [
      'module' => $module,
    ]);
    return new RedirectResponse($url->toString());
  }

  /**
   * Removes a module folder from filesystem.
   *
   * @param string $module
   *   The module machine name.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response.
   */
  public function remove($module) {
    try {
      $module_path = $this->extensionList->getPath($module);

      if (!$module_path) {
        $this->messengerService->addError($this->t('Module @m not found.', ['@m' => $module]));
      }

      $full_path = DRUPAL_ROOT . '/' . $module_path;
      $this->fileSystem->deleteRecursive($full_path);

      $this->messengerService->addStatus(
        $this->t('Module @m removed from the filesystem.', ['@m' => $module])
      );
    }
    catch (\Exception $e) {
      // $this->messengerService->addError($e->getMessage());
    }

    $redirect = $this->requestStack->getCurrentRequest()->query->get('redirect') ?? '/admin/modules';
    return new RedirectResponse($redirect);
  }

}
