<?php

namespace Drupal\module_manager\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\module_manager\Batch\ReleaseInstallerBatch;
use Drupal\module_manager\Service\ReleaseInstaller;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to confirm installation of a module release.
 */
class InstallReleaseConfirmForm extends FormBase {

  /**
   * Release installer service.
   *
   * @var \Drupal\module_manager\Service\ReleaseInstaller
   */
  protected ReleaseInstaller $installer;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->installer = $container->get('module_manager.release_installer');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'module_manager_install_release_confirm';
  }

  /**
   * Builds the confirmation form for installing a release.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   * @param string|null $module
   *   The module machine name.
   * @param string|null $version
   *   The module version.
   *
   * @return array
   *   The render array for the form.
   */
  public function buildForm(array $form, FormStateInterface $form_state, $module = NULL, $version = NULL) {

    $form['module'] = [
      '#type' => 'value',
      '#value' => $module,
    ];

    $form['version'] = [
      '#type' => 'value',
      '#value' => $version,
    ];

    $form['message'] = [
      '#markup' => "<p> Do you confirm installing the module <strong>{$module}</strong> version <strong>{$version}</strong>?</p>",
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => 'Install module',
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * Submit handler for the install release confirmation form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $module = $form_state->getValue('module');
    $version = $form_state->getValue('version');

    $zip_url = "https://ftp.drupal.org/files/projects/{$module}-{$version}.zip";

    $batch = [
      'title' => $this->t('Installing module @module...', ['@module' => $module]),
      'operations' => [
        [[ReleaseInstallerBatch::class, 'downloadRelease'], [$zip_url, $module, $version]],
        [[ReleaseInstallerBatch::class, 'extractRelease'], [$module]],
        [[ReleaseInstallerBatch::class, 'validateInfoYml'], [$module]],
        [[ReleaseInstallerBatch::class, 'validateComposer'], [$module]],
        [[ReleaseInstallerBatch::class, 'postValidateRedirect'], [$module]],
        // Activate the module.
        [[ReleaseInstallerBatch::class, 'activateModule'], [$module, $version]],
      ],
      'finished' => [ReleaseInstallerBatch::class, 'finished'],
    ];

    batch_set($batch);
  }

}
