<?php

namespace Drupal\module_manager\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ModuleInstallerInterface;
use Drupal\Core\File\FileSystemInterface;
use GuzzleHttp\ClientInterface;

/**
 * Service for installing module releases.
 */
class ReleaseInstaller {

  /**
   * HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * File system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * Module installer service.
   *
   * @var \Drupal\Core\Extension\ModuleInstallerInterface
   */
  protected ModuleInstallerInterface $moduleInstaller;

  /**
   * Module extension list service.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $extensionListModule;

  /**
   * Module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructor.
   *
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   HTTP client.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   File system service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   Module handler service.
   * @param \Drupal\Core\Extension\ModuleExtensionList $extensionListModule
   *   Module extension list service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory service.
   */
  public function __construct(
    ClientInterface $httpClient,
    FileSystemInterface $fileSystem,
    ModuleHandlerInterface $moduleHandler,
    ModuleExtensionList $extensionListModule,
    ConfigFactoryInterface $configFactory,
  ) {
    $this->httpClient = $httpClient;
    $this->fileSystem = $fileSystem;
    $this->moduleHandler = $moduleHandler;
    $this->extensionListModule = $extensionListModule;
    $this->configFactory = $configFactory;
  }

  /**
   * Downloads the release ZIP to temporary://$module_name.
   *
   * @param string $url
   *   ZIP file URL.
   *
   * @return string|null
   *   File URI or NULL.
   */
  public function downloadRelease(string $url): ?string {
    try {
      $response = $this->httpClient->get($url, ['timeout' => 60]);
      $data = $response->getBody()->getContents();

      $filename = basename(parse_url($url, PHP_URL_PATH));

      // Salva o arquivo em /tmp ao invés de temporary://.
      $file_path = '/tmp/' . $filename;

      // Usa file_put_contents para salvar diretamente em /tmp.
      if (file_put_contents($file_path, $data) === FALSE) {
        return NULL;
      }

      return $file_path;
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

  /**
   * Extracts the ZIP file.
   *
   * @param string $zip_uri
   *   ZIP file URI.
   * @param string $module
   *   Module machine name.
   *
   * @return string|null
   *   Extracted directory path or NULL.
   */
  public function extractRelease(string $zip_uri, $module): ?string {
    $module_installer_dir = DRUPAL_ROOT . '/modules/module_manager_contrib/';

    $this->fileSystem->prepareDirectory(
      $module_installer_dir,
      FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
    );

    $zip_path = $this->fileSystem->realpath($zip_uri);

    $zip = new \ZipArchive();
    if ($zip->open($zip_path) !== TRUE) {
      return NULL;
    }

    $zip->extractTo($module_installer_dir);
    $zip->close();

    // Delete the ZIP file after extraction.
    if (file_exists($zip_path)) {
      @unlink($zip_path);
    }

    return $module_installer_dir;
  }

  /**
   * Moves the extracted module to modules/custom/{module_name}.
   *
   * @param string $extracted_path
   *   Path to extracted module.
   *
   * @return string|null
   *   Clean module name or NULL.
   */
  public function moveExtractedModule(string $extracted_path): ?string {
    $real = $this->fileSystem->realpath($extracted_path);

    if (!is_dir($real)) {
      return NULL;
    }

    $folder_name = basename($real);

    $clean_name = preg_replace('/-[0-9][0-9a-zA-Z\.\-_]*/', '', $folder_name);

    $target_dir = DRUPAL_ROOT . '/modules/custom/' . $clean_name;

    if (is_dir($target_dir)) {
      $this->fileSystem->deleteRecursive($target_dir);
    }

    $moved = $this->fileSystem->move(
      $real,
      $target_dir,
      FileSystemInterface::EXISTS_REPLACE
    );

    if (!$moved) {
      return NULL;
    }

    return $clean_name;
  }

  /**
   * Validates dependencies and installs the module.
   *
   * @param string $module_name
   *   Module machine name.
   *
   * @return array
   *   Array with 'success' and 'errors'.
   */
  public function installModule(string $module_name): array {

    $info = $this->extensionListModule->getExtensionInfo($module_name);

    if (!$info) {
      return [
        'success' => FALSE,
        'errors' => ["Module '$module_name' not found (info.yml missing)."],
      ];
    }

    $dependencies = $info['requires'] ?? [];
    $errors = [];

    foreach ($dependencies as $dependency => $version) {

      if (!$this->extensionListModule->getExtensionInfo($dependency)) {
        $errors[] = "Dependency '$dependency' does not exist in the system.";
        continue;
      }

      if (!$this->moduleHandler->moduleExists($dependency)) {
        $errors[] = "Dependency '$dependency' exists but is not enabled.";
      }
    }

    if (!empty($errors)) {
      return [
        'success' => FALSE,
        'errors' => $errors,
      ];
    }

    try {
      $this->moduleInstaller->install([$module_name]);
      return [
        'success' => TRUE,
        'errors' => [],
      ];
    }
    catch (\Exception $e) {
      return [
        'success' => FALSE,
        'errors' => [$e->getMessage()],
      ];
    }
  }

}
