<?php

namespace Drupal\modules_list\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Render\RendererInterface;
use Drupal\Component\Datetime\TimeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Controller for the modules list page.
 */
class ModulesListController extends ControllerBase {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The module extension list.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs a ModulesListController object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\Extension\ModuleExtensionList $module_extension_list
   *   The module extension list.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(
    ModuleHandlerInterface $module_handler,
    ModuleExtensionList $module_extension_list,
    RendererInterface $renderer,
    DateFormatterInterface $date_formatter,
    TimeInterface $time,
  ) {
    $this->moduleHandler = $module_handler;
    $this->moduleExtensionList = $module_extension_list;
    $this->renderer = $renderer;
    $this->dateFormatter = $date_formatter;
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_handler'),
      $container->get('extension.list.module'),
      $container->get('renderer'),
      $container->get('date.formatter'),
     $container->get('datetime.time')
    );
  }

  /**
   * Lists all enabled modules.
   *
   * @return array
   *   Render array for the page.
   */
  public function listModules() {
    // Get all installed modules.
    $modules = $this->moduleHandler->getModuleList();

    // Build a table of module information.
    $header = [
      'name' => $this->t('Module Name'),
      'version' => $this->t('Version'),
      'description' => $this->t('Description'),
      'machine_name' => $this->t('Machine Name'),
      'path' => $this->t('Path'),
      'datestamp' => $this->t('Release Date'),
    ];

    $rows = [];
    // Get module info using the extension list service.
    $module_info = $this->moduleExtensionList->getAllInstalledInfo();

    foreach ($modules as $machine_name => $module) {
      $info = $module_info[$machine_name] ?? [];
      $version = $info['version'] ?? $this->t('N/A');
      $description = $info['description'] ?? $this->t('No description available');

      // Get release date if available, format it nicely.
      $datestamp = $info['datestamp'] ?? NULL;
      $formatted_date = $datestamp ? $this->dateFormatter->format($datestamp, 'custom', 'd M Y') : $this->t('N/A');

      $rows[] = [
        'name' => $info['name'] ?? $machine_name,
        'version' => $version,
        'description' => $description,
        'machine_name' => $machine_name,
        'path' => $module->getPath(),
        'datestamp' => $formatted_date,
      ];
    }

    // Sort modules alphabetically by name.
    usort($rows, function ($a, $b) {
      return strcasecmp($a['name'], $b['name']);
    });

    // Add the CSS and JS for the filter functionality.
    $build['#attached']['library'][] = 'localgov_telemetry/module-filter';

    // Add filter textfield.
    $build['filters'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['table-filter', 'js-show']],
      '#weight' => -10,
    ];

    $build['filters']['text'] = [
      '#type' => 'search',
      '#title' => $this->t('Filter modules'),
      '#title_display' => 'invisible',
      '#size' => 30,
      '#placeholder' => $this->t('Filter by name or description'),
      '#attributes' => [
        'class' => ['module-filter-text'],
        'data-table' => '.module-list-table',
        'autocomplete' => 'off',
      ],
    ];

    $build['description'] = [
      '#markup' => '<p>' . $this->t('This page displays all modules currently enabled on this site.') . '</p>',
    ];

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#attributes' => ['class' => ['module-list-table']],
      '#sticky' => TRUE,
    ];

    return $build;
  }

  /**
   * Returns list of enabled modules as JSON.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with modules information.
   */
  public function listModulesJson() {
    // Get all installed modules.
    $modules = $this->moduleHandler->getModuleList();
    $module_info = $this->moduleExtensionList->getAllInstalledInfo();

    $modules_data = [];
    foreach ($modules as $machine_name => $module) {
      $info = $module_info[$machine_name] ?? [];
      $version = $info['version'] ?? 'N/A';
      $description = $info['description'] ?? 'No description available';

      // Get release date if available.
      $datestamp = $info['datestamp'] ?? NULL;
      // For JSON, include both timestamp and formatted date.
      $date_info = [
        'timestamp' => $datestamp,
        'formatted' => $datestamp ? $this->dateFormatter->format($datestamp, 'custom', 'd M Y') : 'N/A',
      ];

      $modules_data[] = [
        'name' => $info['name'] ?? $machine_name,
        'machine_name' => $machine_name,
        'version' => $version,
        'description' => $description,
        'path' => $module->getPath(),
        'release_date' => $date_info,
      ];
    }

    // Sort modules alphabetically by name.
    usort($modules_data, function ($a, $b) {
      return strcasecmp($a['name'], $b['name']);
    });

    return new JsonResponse([
      'modules' => $modules_data,
      'count' => count($modules_data),
      // Keep a machine-friendly timestamp for consumers that need it.
      'generated_timestamp' => $this->time->getCurrentTime(),
      // Human-readable generated time.
      'generated' => $this->dateFormatter->format($this->time->getCurrentTime(), 'custom', 'd M Y H:i:s'),
    ]);
  }

}
