<?php

namespace Drupal\modules_manager_enhanced\Service;

use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

/**
 * Service to alter the system modules form.
 */
class FormAlterService {

  use StringTranslationTrait;

  /**
   * The module extension list service.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * Constructs a FormAlterService object.
   *
   * @param \Drupal\Core\Extension\ModuleExtensionList $module_extension_list
   *   The module extension list service.
   */
  public function __construct(ModuleExtensionList $module_extension_list) {
    $this->moduleExtensionList = $module_extension_list;
  }

  /**
   * Alters the system modules form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function alterForm(array &$form, FormStateInterface $form_state) {
    // Only users who can administer modules should see enhanced features
    if (!\Drupal::currentUser()->hasPermission('administer modules')) {
      return;
    }
    
    // Add filter controls at the top
    $form['filters']['package_filter_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Filter by Package'),
      '#open' => TRUE,
      '#weight' => -10,
    ];

    // Get current filter from URL - clean format (?packages=pkg1,pkg2,pkg3)
    $request = \Drupal::request();
    $packages_param = $request->query->get('packages');
    $selected_packages = [];
    
    if ($packages_param) {
      $selected_packages = explode('&', $packages_param);
      $selected_packages = array_map('trim', $selected_packages);
    }
    
    // Fallback: support old format for backward compatibility
    if (empty($selected_packages)) {
      $selected_packages = $request->query->all('filter_packages') ?? [];
      if (empty($selected_packages)) {
        $single_package = $request->query->get('filter_packages');
        if ($single_package) {
          $selected_packages = is_array($single_package) ? $single_package : [$single_package];
        }
      }
    }

    $form['filters']['package_filter_section']['packages'] = [
      '#type' => 'select',
      '#title' => $this->t('Select packages to display'),
      '#multiple' => TRUE,
      '#size' => 8,
      '#options' => $this->getPackageOptions(),
      '#default_value' => $selected_packages,
      '#attributes' => [
        'class' => ['packages-filter'],
        'id' => 'packages-filter',
      ],
    ];

    $form['filters']['package_filter_section']['actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['filter-actions']],
    ];

    $form['filters']['package_filter_section']['actions']['apply'] = [
      '#type' => 'button',
      '#value' => $this->t('Apply Filter'),
      '#attributes' => [
        'class' => ['apply-filter', 'button', 'button--primary'],
        'onclick' => 'return false;', // Prevent form submission
      ],
    ];

    $form['filters']['package_filter_section']['actions']['reset'] = [
      '#type' => 'button',
      '#value' => $this->t('Reset Filter'),
      '#attributes' => [
        'class' => ['reset-filter', 'button'],
        'onclick' => 'return false;', // Prevent form submission
      ],
    ];

    // Add package data for JavaScript
    $form['#attached']['drupalSettings']['modulesManagerEnhanced'] = [
      'currentPackages' => $selected_packages,
      'modulesPageUrl' => Url::fromRoute('system.modules_list')->toString(),
      'defaultPackages' => $this->getDefaultPackages(),
    ];

    // Apply default packages if no filter is specified
    if (empty($selected_packages)) {
      $selected_packages = $this->getDefaultPackages();
      $form['filters']['package_filter_section']['packages']['#default_value'] = $selected_packages;
      $form['#attached']['drupalSettings']['modulesManagerEnhanced']['currentPackages'] = $selected_packages;
    }

    // Apply server-side filtering if active
    if (!empty($selected_packages)) {
      $this->applyServerSideFiltering($form, $selected_packages);
    }


  }

  /**
   * Apply server-side filtering to the form.
   *
   * @param array $form
   *   The form array.
   * @param array $selected_packages
   *   The selected packages to filter by.
   */
  protected function applyServerSideFiltering(array &$form, array $selected_packages) {
    // Get filtered data from EventSubscriber
    $filtered_data = \Drupal\modules_manager_enhanced\EventSubscriber\ModulesFilterSubscriber::getFilteredData();
    
    if (empty($filtered_data['modules'])) {
      return;
    }

    $filtered_modules = array_keys($filtered_data['modules']);
    
    // Add status message
    $form['filters']['filter_status'] = [
      '#type' => 'markup',
      '#markup' => '<div class="filter-status"><strong>' . $this->t('Active Filter:') . '</strong> ' . 
        $this->t('Showing @count modules from packages: @packages', [
          '@count' => count($filtered_modules),
          '@packages' => implode(', ', $selected_packages),
        ]) . '</div>',
      '#weight' => -5,
    ];

    // Remove modules that are not in the filtered list
    $this->filterFormModules($form, $filtered_modules);
    

  }

  /**
   * Filter form modules to show only selected ones.
   *
   * @param array $form
   *   The form array.
   * @param array $allowed_modules
   *   Array of module names to show.
   */
  protected function filterFormModules(array &$form, array $allowed_modules) {


    // Find all package sections - they are usually in $form['modules']
    if (isset($form['modules'])) {
      foreach ($form['modules'] as $key => &$element) {
        // Skip non-array elements and form API keys starting with #
        if (!is_array($element) || strpos($key, '#') === 0) {
          continue;
        }

        if (isset($element['#type']) && $element['#type'] === 'details') {
          // Check if this section contains any allowed modules
          $has_allowed_modules = $this->sectionHasAllowedModules($element, $allowed_modules);
          
          if (!$has_allowed_modules) {
            // Hide the entire section if no modules match
            unset($form['modules'][$key]);
            
          } else {
            // Filter individual modules within the section
            $this->filterSectionModules($element, $allowed_modules);
          }
        }
      }
    }
  }

  /**
   * Check if a section contains any allowed modules.
   *
   * @param array $section
   *   The section element.
   * @param array $allowed_modules
   *   Array of allowed module names.
   *
   * @return bool
   *   TRUE if section has allowed modules.
   */
  protected function sectionHasAllowedModules(array $section, array $allowed_modules) {
    foreach ($section as $key => $element) {
      if (is_array($element) && strpos($key, '#') !== 0) {
        // Check multiple patterns for module detection
        $module_name = null;
        
        // Pattern 1: modules[module_name][enable]
        if (isset($element['enable']['#name']) && 
            preg_match('/modules\[([^\]]+)\]\[enable\]/', $element['enable']['#name'], $matches)) {
          $module_name = $matches[1];
        }
        // Pattern 2: direct key might be the module name
        elseif (is_string($key) && isset($element['name'])) {
          $module_name = $key;
        }
        // Pattern 3: check in element structure
        elseif (isset($element['#module_name'])) {
          $module_name = $element['#module_name'];
        }
        
        if ($module_name && in_array($module_name, $allowed_modules)) {
          return TRUE;
        }
      }
    }
    
    return FALSE;
  }

  /**
   * Filter modules within a section.
   *
   * @param array $section
   *   The section element.
   * @param array $allowed_modules
   *   Array of allowed module names.
   */
  protected function filterSectionModules(array &$section, array $allowed_modules) {
    foreach ($section as $key => &$element) {
      if (is_array($element) && strpos($key, '#') !== 0) {
        // Check if this is a module element
        if (isset($element['enable']['#name']) && 
            preg_match('/modules\[([^\]]+)\]/', $element['enable']['#name'], $matches)) {
          $module_name = $matches[1];
          if (!in_array($module_name, $allowed_modules)) {
            unset($section[$key]);
          }
        }
      }
    }
  }

  /**
   * Get available package options.
   *
   * @return array
   *   Array of package options.
   */
  protected function getPackageOptions() {
    $options = [];
    $modules = $this->moduleExtensionList->getList();

    foreach ($modules as $module) {
      $package = $module->info['package'] ?? 'Other';
      if (!isset($options[$package])) {
        $options[$package] = $package;
      }
    }

    ksort($options);
    return $options;
  }

  /**
   * Get default packages to display for optimization.
   *
   * @return array
   *   Array of default package names.
   */
  protected function getDefaultPackages() {
    // Load most commonly used packages by default for better performance
    return [
      'Core',
      'Administration',
    ];
  }



} 