<?php

namespace Drupal\modules_manager_enhanced\EventSubscriber;

use Drupal\Core\Extension\ModuleExtensionList;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Event subscriber for filtering modules on the modules page.
 */
class ModulesFilterSubscriber implements EventSubscriberInterface {

  /**
   * The module extension list service.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Static cache for filtered data.
   *
   * @var array
   */
  protected static $filteredData = [];

  /**
   * Constructs a ModulesFilterSubscriber object.
   *
   * @param \Drupal\Core\Extension\ModuleExtensionList $module_extension_list
   *   The module extension list service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(ModuleExtensionList $module_extension_list, RequestStack $request_stack) {
    $this->moduleExtensionList = $module_extension_list;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      KernelEvents::REQUEST => ['onRequest', 10],
    ];
  }

  /**
   * Responds to the kernel request event.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The event object.
   */
  public function onRequest(RequestEvent $event) {
    $request = $event->getRequest();
    
    // Only process on modules page
    if ($request->attributes->get('_route') !== 'system.modules_list') {
      return;
    }

    // Get filter packages from clean URL format (?packages=pkg1,pkg2,pkg3)
    $packages_param = $request->query->get('packages');
    $filter_packages = [];
    
    if ($packages_param) {
      $filter_packages = explode('&', $packages_param);
      $filter_packages = array_map('trim', $filter_packages);
    }
    
    // Fallback: support old format for backward compatibility
    if (empty($filter_packages)) {
      $filter_packages = $request->query->all('filter_packages') ?? [];
      if (empty($filter_packages)) {
        $single_package = $request->query->get('filter_packages');
        if ($single_package) {
          $filter_packages = is_array($single_package) ? $single_package : [$single_package];
        }
      }
    }

    // Clean empty values
    $filter_packages = array_filter($filter_packages);

    // Apply default packages if no filter is specified
    if (empty($filter_packages)) {
      $filter_packages = $this->getDefaultPackages();
    }

    if (!empty($filter_packages)) {
      $this->preFilterModules($filter_packages);
    }
  }

  /**
   * Pre-filter modules based on selected packages.
   *
   * @param array $filter_packages
   *   Array of package names to filter by.
   */
  protected function preFilterModules(array $filter_packages) {
    $all_modules = $this->moduleExtensionList->getList();
    $filtered_modules = [];

    foreach ($all_modules as $name => $module) {
      $package = $module->info['package'] ?? 'Other';
      if (in_array($package, $filter_packages)) {
        $filtered_modules[$name] = [
          'name' => $name,
          'package' => $package,
          'status' => $module->status,
          'info' => $module->info,
        ];
      }
    }

    self::$filteredData = [
      'packages' => $filter_packages,
      'modules' => $filtered_modules,
    ];
  }

  /**
   * Get filtered data.
   *
   * @return array
   *   The filtered data or empty array if no filtering.
   */
  public static function getFilteredData() {
    return self::$filteredData;
  }

  /**
   * Check if filtering is active.
   *
   * @return bool
   *   TRUE if filtering is active.
   */
  public static function isFilteringActive() {
    return !empty(self::$filteredData);
  }

  /**
   * Get default packages for optimization.
   *
   * @return array
   *   Array of default package names.
   */
  protected function getDefaultPackages() {
    // Load most commonly used packages by default for better performance
    return [
      'Core',
      'Administration',
    ];
  }

} 