<?php

namespace Drupal\mom\Controller;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Url;
use Drupal\matomo_reporting_api\MatomoQueryFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Matomo onpage metrics controller: builds the content for the stats modal.
 */
final class MomController extends ControllerBase {

  /**
   * The onpage metrics config factory.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $onpageMetricsConfig;

  /**
   * The onpage metrics config factory.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $matomoReportingApiConfig;

  /**
   * The module extension list service.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $extensionList;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * Constructs a new CustomModalController object.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The current route match object.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The Drupal config factory.
   * @param \Drupal\matomo_reporting_api\MatomoQueryFactoryInterface $matomoQueryFactory
   *   The Matomo query factory.
   * @param \Drupal\Core\Extension\ModuleExtensionList $extension_list
   *   The extension list.
   */
  public function __construct(protected RequestStack $request_stack, protected RouteMatchInterface $routeMatch, ConfigFactoryInterface $configFactory, protected MatomoQueryFactoryInterface $matomoQueryFactory, ModuleExtensionList $extension_list) {
    $this->onpageMetricsConfig = $configFactory->get('mom.settings');
    $this->matomoReportingApiConfig = $configFactory->get('matomo_reporting_api.settings');
    $this->extensionList = $extension_list;
    $this->requestStack = $this->request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new self(
      $container->get('request_stack'),
      $container->get('current_route_match'),
      $container->get('config.factory'),
      $container->get('matomo.query_factory'),
      $container->get('extension.list.module'),
    );
  }

  /**
   * Constructs the modal content data.
   */
  public function modal() {
    $stats_data = [];

    $config_data = $this->onpageMetricsConfig->get();

    if (!empty($config_data)) {
      $onpage_metrics_config = $this->onpageMetricsConfig;

      $mom_inherit = $onpage_metrics_config->get('inherit_matomo_settings');
      $mom_title = $onpage_metrics_config->get('title');
      $mom_live_url = $onpage_metrics_config->get('live_url');
      $mom_display_graph_type = $onpage_metrics_config->get('graph_type');
      $stats_data['url'] = $onpage_metrics_config->get('url');
      $stats_data['display_date'] = $onpage_metrics_config->get('date');
      $stats_data['display_total_visitors'] = $onpage_metrics_config->get('total_visitors');
      $stats_data['display_unique_visitors'] = $onpage_metrics_config->get('unique_visitors');
      $stats_data['display_visit_time'] = $onpage_metrics_config->get('visit_time');
      $stats_data['display_graph'] = $onpage_metrics_config->get('graph');
      $stats_data['display_link'] = $onpage_metrics_config->get('display_link');
      $stats_data['link_text'] = $onpage_metrics_config->get('link_text');

      $matomo_reporting_api_config = $this->matomoReportingApiConfig;
      $token = $matomo_reporting_api_config->get('token_auth');
      $site_id = $matomo_reporting_api_config->get('site_id');
      $stats_data['token'] = $token;
      $stats_data['site_id'] = $site_id;

      if ($mom_inherit || $stats_data['url'] == NULL) {
        $matomo_url = $matomo_reporting_api_config->get('url_https');
      }
      else {
        $matomo_url = $stats_data['url'];
      }
      $stats_data['matomo_url'] = $matomo_url;

      $base_url = (string) $mom_live_url;

      if ($token != NULL && $site_id != NULL && $mom_live_url != NULL) {
        $path = '/';
        $mom_path_cookie_value = $this->requestStack->getCurrentRequest()->cookies->get('MoMpath');
        if ($mom_path_cookie_value) {
          $path = $mom_path_cookie_value;
        }
        if ($path == '/homepage') {
          $path = '/';
        }

        $path = (string) $path;

        $segment = 'entryPageUrl==' . $base_url . $path;
        $stats_data['segment'] = $segment;

        $two_months_ago = date('Y-m-d', strtotime('-2 months'));
        $one_month_ago = date('Y-m-d', strtotime('-1 month'));

        $date_range_previous = $two_months_ago . ',' . $one_month_ago;

        $date_range = 'previous30';

        $one_month_ago_timestamp = strtotime('-1 month');
        $two_months_ago_timestamp = strtotime('-2 month');

        if (date('Y', $one_month_ago_timestamp) === date('Y')) {
          $stats_data['date_range'] = date('F j', $one_month_ago_timestamp) . " - " . date('F j, Y');
        }
        else {
          $stats_data['date_range'] = date('F j, Y', $one_month_ago_timestamp) . " - " . date('F j, Y');
        }

        if (date('Y', $one_month_ago_timestamp) === date('Y', $two_months_ago_timestamp)) {
          $stats_data['date_range_previous'] = date('F j', $two_months_ago_timestamp) . " - " . date('F j, Y', $one_month_ago_timestamp);
        }
        else {
          $stats_data['date_range_previous'] = date('F j, Y', $two_months_ago_timestamp) . " - " . date('F j, Y', $one_month_ago_timestamp);
        }

        $stats_data['graph_type'] = $mom_display_graph_type;

        $stats_data_previous = array_merge($stats_data, $this->getVisitsSummary($site_id, $token, $date_range_previous, $segment));
        $stats_data = array_merge($stats_data, $this->getVisitsSummary($site_id, $token, $date_range, $segment));

        // Construct the visitors data.
        $visitors_change = $stats_data['visitors'] - $stats_data_previous['visitors'];
        $stats_data['visitors_change_positive'] = TRUE;

        if ($visitors_change < 0) {
          $stats_data['visitors_change_positive'] = FALSE;
        }

        $stats_data['visitors_change'] = $this->shortenLargeNumbers(abs($visitors_change));

        // Construct the unique visitors data.
        $unique_visitors_change = $stats_data['unique_visitors'] - $stats_data_previous['unique_visitors'];
        $stats_data['unique_visitors_change_positive'] = TRUE;

        if ($unique_visitors_change < 0) {
          $stats_data['unique_visitors_change_positive'] = FALSE;
        }

        $stats_data['unique_visitors_change'] = $this->shortenLargeNumbers(abs($unique_visitors_change));

        // Construct the visit time data.
        $average_visit_time_change = $stats_data['average_visit_time'] - $stats_data_previous['average_visit_time'];
        $stats_data['average_visit_time_positive'] = TRUE;

        if ($average_visit_time_change < 0) {
          $stats_data['average_visit_time_positive'] = FALSE;
        }

        $stats_data['average_visit_time_change'] = $this->shortenLargeNumbers(abs($average_visit_time_change));

        if (isset($stats_data['link_text']) && $stats_data['link_text'] != NULL) {
          $stats_data['link_text_display'] = preg_replace('(^https?://)', '', $stats_data['link_text']);
        }

        $stats_data['config_url'] = Url::fromRoute('mom.form')->toString();

        $content = [
          '#theme' => 'mom_modal',
          '#data' => $stats_data,
          '#data_previous' => $stats_data_previous,
        ];

      }
      else {
        if ($token == NULL) {
          $data['feedback_url'] = Url::fromRoute('matomo_reporting_api.settings_form')->toString();

          $data['feedback'] = 'The Matomo authentication token not configured correctly.
          It is advised to add the token in the local settings file as described in the readme file.
          Alternatively it can be set in the Matomo Reporting API settings but be aware that this could expose the API key:';
        }
        else {
          $data['feedback_url'] = Url::fromRoute('mom.form')->toString();

          $data['feedback'] = 'Matomo settings not configured correctly, please check the settings:';
        }
        $content = [
          '#theme' => 'mom_modal',
          '#data' => $data,
        ];
      }
    }
    else {
      $data['feedback_url'] = Url::fromRoute('mom.form')->toString();

      $data['feedback'] = 'The Matomo on-page metrics settings need to be set first:';

      $content = [
        '#theme' => 'mom_modal',
        '#data' => $data,
      ];
    }

    $options = [
      'dialogClass' => 'matomo-onpage-metrics-modal',
      'width' => '1100',
      'height' => '720',
    ];

    $response = new AjaxResponse();

    if (!isset($mom_title) || !$mom_title) {
      $mom_title = 'Matomo on-page metrics';
    }

    $module_path = $this->extensionList->getPath('mom');
    $content['#attached']['drupalSettings']['bootstrap_css_path'] = "/{$module_path}/static/css/bootstrap-grid.min.css";
    $content['#attached']['drupalSettings']['mom_css_path'] = "/{$module_path}/static/css/mom.css";

    $response->addCommand(new OpenModalDialogCommand($mom_title, $content, $options));
    return $response;
  }

  /**
   * Gets the visits summary data.
   */
  private function getVisitsSummary($site_id, $token, $date_range, $segment) {
    $current_stats_data = [];

    $visits_query = $this->matomoQueryFactory->getQuery('VisitsSummary.get');
    $visits_query->setParameters([
      'idSite' => $site_id,
      'token_auth' => $token,
      'period' => 'day',
      'date' => $date_range,
      'segment' => $segment,
    ]);

    $visit_results = $visits_query->execute()->getResponse();

    $visitors = 0;
    $unique_visitors = 0;
    $average_visit_time = 0;
    $total_visit_time = 0;

    foreach ($visit_results as $visit_day) {
      if (isset($visit_day->nb_visits)) {
        $visitors += $visit_day->nb_visits;
      }
      if (isset($visit_day->nb_uniq_visitors)) {
        $unique_visitors += $visit_day->nb_uniq_visitors;
      }
      if (isset($visit_day->nb_uniq_visitors)) {
        /**
         * @var \Drupal\mom\Controller\MomController $visit_day
         * @phpstan-ignore property.notFound
         */
        $total_visit_time += $visit_day->sum_visit_length;
      }
    }

    if ($total_visit_time > 0 && $visitors > 0) {
      $average_visit_time = number_format((float) $total_visit_time / $visitors, 2, '.', ' ');
    }

    $current_stats_data['visitors'] = $visitors;
    $current_stats_data['unique_visitors'] = $unique_visitors;
    $current_stats_data['average_visit_time'] = $average_visit_time;
    $current_stats_data['total_visit_time'] = $total_visit_time;

    return $current_stats_data;
  }

  /**
   * Shortens large numbers for display.
   */
  private function shortenLargeNumbers($number, $precision = 2) {
    if ($number >= 1000 && $number < 1000000) {
      $formatted_number = number_format($number / 1000, $precision) . 'K';
    }
    elseif ($number >= 1000000 && $number < 1000000000) {
      $formatted_number = number_format($number / 1000000, $precision) . 'M';
    }
    elseif ($number >= 1000000000) {
      $formatted_number = number_format($number / 1000000000, $precision) . 'B';
    }
    else {
      $formatted_number = number_format($number, 0);
    }
    return $formatted_number;
  }

}
