<?php

namespace Drupal\mom\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Matomo onpage metrics form: allows users to configure the modal.
 */
class MomForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'mom.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'mom_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('mom.settings');

    $config_check = $config->getRawData();

    if (count($config_check) !== 0) {
      $inherit_matomo_settings = $config->get('inherit_matomo_settings');
      $date = $config->get('date');
      $total_visitors = $config->get('total_visitors');
      $unique_visitors = $config->get('unique_visitors');
      $visit_time = $config->get('visit_time');
      $graph = $config->get('graph');
      $display_link = $config->get('display_link');
    }
    else {
      $inherit_matomo_settings = TRUE;
      $date = TRUE;
      $total_visitors = TRUE;
      $unique_visitors = TRUE;
      $visit_time = TRUE;
      $graph = TRUE;
      $display_link = FALSE;
    }

    $form['live_url'] = [
      '#default_value' => $config->get('live_url'),
      '#placeholder' => 'https://yoursite.com/',
      '#description' => $this->t('The URL of the live website.'),
      '#maxlength' => 255,
      '#size' => 80,
      '#title' => $this->t('Live website URL'),
      '#type' => 'url',
      '#pattern' => 'https://.*',
      '#required' => TRUE,
    ];

    // Get the settings required for the Matomo Reporting API connection.
    $form['inherit_matomo_settings'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use server connection settings from the <a href=":url">Matomo Reporting API configuration form</a>.', [
        ':url' => Url::fromRoute('matomo_reporting_api.settings_form')->toString(),
      ]),
      '#default_value' => $inherit_matomo_settings,
    ];

    $form['connection_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Server connection settings'),
      '#open' => TRUE,
      '#states' => [
        'invisible' => [
          ':input[name="inherit_matomo_settings"]' => ['checked' => TRUE],
        ],
        'disabled' => [
          ':input[name="inherit_matomo_settings"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $url = $config->get('url');
    $form['connection_settings']['url'] = [
      '#default_value' => $url,
      '#placeholder' => 'https://analytics.example.com/',
      '#description' => $this->t('The URL of your Matomo application.'),
      '#maxlength' => 255,
      '#size' => 80,
      '#title' => $this->t('Matomo URL'),
      '#type' => 'url',
      '#pattern' => 'https://.*',
    ];

    // Display settings to allow the user to select and/or enter the details
    // that should be displayed in the modal.
    $form['display_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Display settings'),
      '#open' => TRUE,
    ];

    $form['display_settings']['date'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display the date range label and data.'),
      '#default_value' => $date,
    ];

    $form['display_settings']['total_visitors'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display the total visitors label and data.'),
      '#default_value' => $total_visitors,
    ];

    $form['display_settings']['unique_visitors'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display the unique visitors label and data.'),
      '#default_value' => $unique_visitors,
    ];

    $form['display_settings']['visit_time'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display the average visit time label and data.'),
      '#default_value' => $visit_time,
    ];

    // Get the graph settings.
    $form['display_settings']['graph'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display the graph.'),
      '#default_value' => $graph,
    ];

    $form['display_settings']['graph_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Graph settings'),
      '#open' => TRUE,
      '#states' => [
        'invisible' => [
          ':input[name="graph"]' => ['checked' => FALSE],
        ],
        'disabled' => [
          ':input[name="graph"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['display_settings']['graph_settings']['graph_type'] = [
      '#title' => $this->t('Default graph type'),
      '#type' => 'select',
      '#default_value' => $config->get('graph_type'),
      '#options' => [
        'pageviews' => $this->t('Pageviews'),
        'visits' => $this->t('Visits'),
        'browsers' => $this->t('Browsers'),
        'countries' => $this->t('Countries'),
        'resolutions' => $this->t('Resolutions'),
        'os' => $this->t('OS'),
      ],
    ];

    // Get the link settings.
    $form['display_settings']['display_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display the custom stats link.'),
      '#default_value' => $display_link,
    ];

    $form['display_settings']['link_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Link settings'),
      '#open' => TRUE,
      '#states' => [
        'invisible' => [
          ':input[name="display_link"]' => ['checked' => FALSE],
        ],
        'disabled' => [
          ':input[name="display_link"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['display_settings']['link_settings']['link_text'] = [
      '#placeholder' => 'https://example.com/',
      '#title' => $this->t('Custom stats link'),
      '#description' => $this->t('This field can be used to add a custom link to more detailed stats'),
      '#maxlength' => 255,
      '#size' => 80,
      '#default_value' => $config->get('link_text'),
      '#type' => 'url',
      '#pattern' => 'https://.*',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $this->config('mom.settings')
      ->set('title', $form_state->getValue('title'))
      ->set('live_url', $form_state->getValue('live_url'))
      ->set('inherit_matomo_settings', $form_state->getValue('inherit_matomo_settings'))
      ->set('url', $form_state->getValue('url'))
      ->set('date', $form_state->getValue('date'))
      ->set('total_visitors', $form_state->getValue('total_visitors'))
      ->set('unique_visitors', $form_state->getValue('unique_visitors'))
      ->set('visit_time', $form_state->getValue('visit_time'))
      ->set('graph', $form_state->getValue('graph'))
      ->set('graph_type', $form_state->getValue('graph_type'))
      ->set('display_link', $form_state->getValue('display_link'))
      ->set('link_text', $form_state->getValue('link_text'))
      ->save();
  }

}
