<?php

declare(strict_types=1);

namespace Drupal\monitoring_endpoint\Plugin\monitoring\SensorPlugin;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\RfcLogLevel;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\monitoring\Attribute\SensorPlugin;
use Drupal\monitoring\Entity\SensorConfig;
use Drupal\monitoring\Entity\SensorResultDataInterface;
use Drupal\monitoring\Result\SensorResultInterface;
use Drupal\monitoring\SensorPlugin\SensorPluginBase;
use Drupal\ultimate_cron\Entity\CronJob;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Checks the status of monitored cron jobs.
 *
 * @Sensor(
 *   id = "monitored_cron_jobs_status",
 *   label = @Translation("Monitored Cron Jobs Status"),
 *   description = @Translation("Checks if all the monitored cron jobs succeeded or failed."),
 * )
 */
#[SensorPlugin(
  id: 'monitored_cron_jobs_status',
  label: new TranslatableMarkup('Monitored Cron Jobs Status'),
  addable: TRUE,
  metric_type: 'gauge',
)]
class MonitoredCronJobSensorPlugin extends SensorPluginBase {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a MonitoredCronJobSensorPlugin instance.
   *
   * @param \Drupal\monitoring\Entity\SensorConfig $sensor_config
   *   The sensor config.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   */
  public function __construct(
    SensorConfig $sensor_config,
    $plugin_id,
    $plugin_definition,
    ConfigFactoryInterface $config_factory,
    ModuleHandlerInterface $module_handler,
  ) {
    parent::__construct($sensor_config, $plugin_id, $plugin_definition);
    $this->configFactory = $config_factory;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, SensorConfig $sensor_config, $plugin_id, $plugin_definition) {
    return new static(
      $sensor_config,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function runSensor(SensorResultInterface $sensor_result) {
    // Skip if Ultimate Cron is not available.
    if (!$this->moduleHandler->moduleExists('ultimate_cron')) {
      $sensor_result->setStatus(SensorResultDataInterface::STATUS_WARNING);
      $sensor_result->setMessage($this->t('Ultimate Cron module is not enabled.'));
      return;
    }

    // Get the list of monitored cron jobs from configuration.
    $config = $this->configFactory->get('monitoring_endpoint.settings');
    $monitored_jobs = $config->get('monitored_cron_jobs') ?: [];

    if (empty($monitored_jobs)) {
      $sensor_result->setStatus(SensorResultDataInterface::STATUS_INFO);
      $sensor_result->setMessage($this->t('No cron jobs are configured to be monitored.'));
      return;
    }

    $failedJobs = [];
    $warningJobs = [];
    $jobDetails = [];

    // Check the status of each monitored job.
    foreach ($monitored_jobs as $job_id => $enabled) {
      if (!$enabled) {
        continue;
      }

      /** @var \Drupal\ultimate_cron\CronJobInterface $job */
      $job = CronJob::load($job_id);

      if (!$job) {
        $failedJobs[] = $job_id;
        $jobDetails[$job_id] = $this->t('Job not found');
        continue;
      }

      // Skip if the cron job is disabled.
      if (!$job->status()) {
        $warningJobs[] = $job_id;
        $jobDetails[$job_id] = $this->t('Job disabled');
        continue;
      }

      /** @var \Drupal\ultimate_cron\Logger\LogEntry $log_entry */
      $log_entry = $job->loadLatestLogEntry();

      if (!$log_entry) {
        $warningJobs[] = $job_id;
        $jobDetails[$job_id] = $this->t('Never executed');
        continue;
      }

      $logData = $log_entry->getData();
      $severity = $logData['severity'] ?? RfcLogLevel::EMERGENCY;

      // Check if the job has errors.
      // If severity is -1, it means no info was logged (success case).
      // Only consider as failed if there's an actual error
      // with severity <= ERROR.
      if ($severity != -1 && $severity <= RfcLogLevel::ERROR) {
        $failedJobs[] = $job_id;
        $jobDetails[$job_id] = $this->t('Failed with severity: @severity', ['@severity' => $severity]);
      }
      else {
        $jobDetails[$job_id] = $this->t('OK');
      }
    }

    // Create a separate message for each job.
    $messages = [];
    $status = SensorResultDataInterface::STATUS_OK;

    // Process failed jobs (critical status).
    foreach ($failedJobs as $job_id) {
      $messages[] = $this->t('Cron job @job_id: @status - @detail', [
        '@job_id' => $job_id,
        '@status' => SensorResultDataInterface::STATUS_CRITICAL,
        '@detail' => $jobDetails[$job_id],
      ]);
      $status = SensorResultDataInterface::STATUS_CRITICAL;
    }

    // Process warning jobs.
    foreach ($warningJobs as $job_id) {
      $messages[] = $this->t('Cron job @job_id: @status - @detail', [
        '@job_id' => $job_id,
        '@status' => SensorResultDataInterface::STATUS_WARNING,
        '@detail' => $jobDetails[$job_id],
      ]);
      if ($status != SensorResultDataInterface::STATUS_CRITICAL) {
        $status = SensorResultDataInterface::STATUS_WARNING;
      }
    }

    // Process successful jobs.
    foreach ($jobDetails as $job_id => $detail) {
      if (!in_array($job_id, array_merge($failedJobs, $warningJobs))) {
        $messages[] = $this->t('Cron job @job_id: @status', [
          '@job_id' => $job_id,
          '@status' => SensorResultDataInterface::STATUS_OK,
        ]);
      }
    }

    // Set the overall status.
    $sensor_result->setStatus($status);

    // Join all messages with newlines to create separate lines for each job.
    if (!empty($messages)) {
      $sensor_result->setMessage(implode("\n", $messages));
    }
    else {
      $sensor_result->setStatus(SensorResultDataInterface::STATUS_INFO);
      $sensor_result->setMessage($this->t('No cron jobs are being monitored.'));
    }

    // Add detailed job status to the verbose output.
    $verboseOutput = [];
    foreach ($jobDetails as $job_id => $status) {
      $verboseOutput[] = "$job_id: $status";
    }

    $sensor_result->setVerboseOutput((array) implode("\n", $verboseOutput));
  }

}
