<?php

declare(strict_types=1);

namespace Drupal\monitoring_endpoint\Controller;

use Drupal\Core\Cache\CacheableJsonResponse;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\RfcLogLevel;
use Drupal\monitoring\Entity\SensorResultDataInterface;
use Drupal\monitoring\Result\SensorResult;
use Drupal\monitoring\SensorRunner;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Monitoring controller.
 */
class MonitoringEndpointController extends ControllerBase {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a MonitoringEndpointController.
   *
   * @param \Drupal\monitoring\SensorRunner $sensorRunner
   *   The sensor runner service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   */
  public function __construct(
    protected SensorRunner $sensorRunner,
    EntityTypeManagerInterface $entityTypeManager,
  ) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('monitoring.sensor_runner'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * API endpoint to get status for the site.
   *
   * This endpoint provides a JSON response containing the status of all
   * enabled monitoring sensors and any monitored cron jobs.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing:
   *   - count_failures: Number of sensors in CRITICAL state
   *   - sensor_id: Status for each enabled sensor
   *   - cron_job_[id]: Status for each monitored cron job (if applicable)
   */
  public function statusEndpoint(): JsonResponse {
    // Results may be cached from sensor runner.
    $results = $this->sensorRunner->runSensors();

    $resultFailures = \array_filter($results, static fn(SensorResult $result): bool => $result->getStatus() === SensorResultDataInterface::STATUS_CRITICAL,
    );

    $details = [];
    $details['count_failures'] = \count($resultFailures);

    $config = $this->config('monitoring_endpoint.settings');
    $monitoredJobs = $config->get('monitored_cron_jobs') ?: [];

    foreach ($results as $sensor_id => $sensor_result) {
      // Special handling for monitored cron jobs sensor.
      if ($sensor_id === 'cron_jobs_status') {
        // Process each monitored cron job.
        foreach ($monitoredJobs as $job_id => $enabled) {
          if ($enabled) {
            // Load the cron job.
            $job = $this->entityTypeManager
              ->getStorage('ultimate_cron_job')
              ->load($job_id);

            if (!$job) {
              $details['cron_job_' . $job_id] = SensorResultDataInterface::STATUS_CRITICAL;
              continue;
            }

            // Check if job is disabled.
            if (!$job->status()) {
              $details['cron_job_' . $job_id] = SensorResultDataInterface::STATUS_WARNING;
              continue;
            }

            // Get the log entry.
            $logEntry = $job->loadLatestLogEntry();
            if (!$logEntry) {
              $details['cron_job_' . $job_id] = SensorResultDataInterface::STATUS_WARNING;
              continue;
            }

            // Check the status.
            $logData = $logEntry->getData();
            $severity = $logData['severity'] ?? RfcLogLevel::EMERGENCY;

            // If severity is -1, it means no info was logged (success case).
            // Only consider as CRITICAL if there's an actual error
            // with severity <= ERROR.
            if ($severity != -1 && $severity <= RfcLogLevel::ERROR) {
              $details['cron_job_' . $job_id] = SensorResultDataInterface::STATUS_CRITICAL;
            }
            else {
              $details['cron_job_' . $job_id] = SensorResultDataInterface::STATUS_OK;
            }
          }
        }
      }
      else {
        // Standard handling for other sensors.
        $details[$sensor_id] = $sensor_result->getStatus();
      }
    }

    $response = (new CacheableJsonResponse())
      ->setData($details)
      ->setEncodingOptions(JSON_PRETTY_PRINT);

    return $response;
  }

}
