<?php

namespace Drupal\mm_webform;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\NodeInterface;
use Drupal\webform\Access\WebformHandlerAccess;
use Drupal\webform\Entity\Webform;
use Drupal\webform\WebformInterface;
use Drupal\webform_ui\Access\WebformUiAccess;

class MMWebformAccess {

  static private $submissionAccessCache = [];

  public static function getWebformForNode(NodeInterface $node) {
    if ($field_def = mm_webform_get_webform_field($node)) {
      $field_name = $field_def->getName();
      if (isset($node->get($field_name)->getValue()[0]['target_id'])) {
        $target_id = $node->get($field_name)->getValue()[0]['target_id'];
        /** @var WebformInterface|null $webform */
        $webform = Webform::load($target_id);
        if ($webform) {
          return $webform;
        }
      }
    }
  }

  public static function access(AccountInterface $account, NodeInterface $node) {
    if ($field_def = mm_webform_get_webform_field($node)) {
      $field_name = $field_def->getName();
      return AccessResult::allowedIf(isset($node->get($field_name)->getValue()[0]['target_id']) && $node->access('update', $account));
    }
    return AccessResult::forbidden();
  }

  public static function accessViewSubmission(AccountInterface $account, NodeInterface $node) {
    return static::accessSubmission($account, $node, 'view');
  }

  public static function accessSubmission(AccountInterface $account, NodeInterface $node, $op) {
    $cid = implode(':', [$account->id(), $node->id(), $op]);
    if (!isset(self::$submissionAccessCache[$cid])) {
      self::$submissionAccessCache[$cid] = AccessResult::forbidden();
      if ($field_def = mm_webform_get_webform_field($node)) {
        $field_name = $field_def->getName();
        if (isset($node->get($field_name)->getValue()[0]['target_id']) && $node->access('view', $account)) {
          /** @var WebformInterface|null $webform */
          $webform = Webform::load($node->get($field_name)->getValue()[0]['target_id']);
          if ($webform) {
            self::$submissionAccessCache[$cid] = mm_webform_webform_submission_access(NULL, $op, $account, $webform);
          }
        }
      }
    }
    return self::$submissionAccessCache[$cid];
  }

  public static function accessWebformElement($type, NodeInterface $node, AccountInterface $account) {
    if (class_exists(WebformUiAccess::class) && ($webform = static::getWebformForNode($node))) {
      return WebformUiAccess::checkWebformElementAccess($webform, $type, $account);
    }
    return AccessResult::forbidden();
  }

  public static function accessWebformEdit(NodeInterface $node, AccountInterface $account) {
    if (class_exists(WebformUiAccess::class) && ($webform = static::getWebformForNode($node))) {
      return WebformUiAccess::checkWebformEditAccess($webform, $account);
    }
    return AccessResult::forbidden();
  }

  public static function accessHandler(NodeInterface $node, $webform_handler = NULL) {
    if (static::getWebformForNode($node)) {
      return WebformHandlerAccess::checkHandlerAccess($webform_handler);
    }
    return AccessResult::forbidden();
  }

}
