<?php

namespace Drupal\mm_webform\RouteProcessor;

use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\RouteProcessor\OutboundRouteProcessorInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\webform\Entity\Webform;
use Symfony\Component\Routing\Route;

/**
 * Processes outbound routes, switching them to node-specific routes when needed.
 */
class OutboundRouteProcessor implements OutboundRouteProcessorInterface {

  /**
   * The route provider.
   *
   * @var \Drupal\Core\Routing\RouteProviderInterface
   */
  protected $routeProvider;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $currentRouteMatch;

  /**
   * Constructs a OutboundRouteProcessor object.
   *
   * @param \Drupal\Core\Routing\RouteProviderInterface $route_provider
   *   The route provider.
   */
  public function __construct(RouteProviderInterface $route_provider, RouteMatchInterface $current_route_match) {
    $this->routeProvider = $route_provider;
    $this->currentRouteMatch = $current_route_match;
  }

  /**
   * {@inheritdoc}
   */
  public function processOutbound($route_name, Route $route, array &$parameters, BubbleableMetadata $bubbleable_metadata = NULL) {
    $page_route_params = $this->currentRouteMatch->getParameters();
    $mm_tree = NULL;
    $applies = ($node = $page_route_params->get('node')) && mm_webform_has_webform_field($node);
    if (!$applies) {
      // In some cases the webform module uses the links contained in the
      // Webform entity, which aren't associated with a node. So attempt to
      // figure out which node to use.
      //
      // There is a webform parameter...
      if (isset($parameters['webform']) && ($webform = Webform::load($parameters['webform']))) {
        // and the webform is associated with one or more nodes...
        if ($nodes = mm_webform_get_nodes_from_webform($webform)) {
          // and one of those nodes is on the current page.
          if ($mm_tree = $page_route_params->get('mm_tree')) {
            if ($nids = array_intersect(mm_content_get_nids_by_mmtid($mm_tree->id()), array_keys($nodes))) {
              $node = reset($nids);
            }
            else {
              return;
            }
          }
        }
      }
    }

    $routes = [
      'entity.webform.edit_form' =>                 'mm_webform.edit_form',
      'entity.webform.settings' =>                  'mm_webform.settings',
      'entity.webform.settings_form' =>             'mm_webform.settings_form',
      'entity.webform.settings_submissions' =>      'mm_webform.settings_submissions',
      'entity.webform.settings_confirmation' =>     'mm_webform.settings_confirmation',
      'entity.webform.handler.edit_form' =>         'mm_webform.handler.edit_form',
      'entity.webform.handler.duplicate_form' =>    'mm_webform.handler.duplicate_form',
      'entity.webform.handler.delete_form' =>       'mm_webform.handler.delete_form',
      'entity.webform.handler.disable' =>           'mm_webform.handler.disable',
      'entity.webform.handler.enable' =>            'mm_webform.handler.enable',
      'entity.webform_ui.element' =>                'mm_webform_ui.element',
      'entity.webform_ui.element.add_form' =>       'mm_webform_ui.element.add_form',
      'entity.webform_ui.element.add_page' =>       'mm_webform_ui.element.add_page',
      'entity.webform_ui.element.add_layout' =>     'mm_webform_ui.element.add_layout',
      'entity.webform_ui.element.edit_form' =>      'mm_webform_ui.element.edit_form',
      'entity.webform_ui.element.duplicate_form' => 'mm_webform_ui.element.duplicate_form',
      'entity.webform_ui.element.delete_form' =>    'mm_webform_ui.element.delete_form',
    ];
    if (isset($routes[$route_name])) {
      $source_route = $this->routeProvider->getRouteByName($routes[$route_name]);
      $requirements = $source_route->getRequirements();
      foreach (['mm_tree' => $mm_tree, 'node' => $node] as $parameter_name => $value) {
        if (!empty($requirements[$parameter_name])) {
          if (!$value) {
            $value = $page_route_params->get($parameter_name);
          }
          if ($value) {
            $parameters[$parameter_name] = is_numeric($value) ? $value : $value->id();
          }
          else {
            // Don't have required parameter, probably because this is not an
            // MM page.
            return;
          }
        }
      }
      $route->setPath($source_route->getPath());
      $route->setDefaults($source_route->getDefaults());
      $route->setRequirements($requirements);
      $route->setOptions($source_route->getOptions());
      $route->setHost($source_route->getHost());
      $route->setSchemes($source_route->getSchemes());
      $route->setMethods($source_route->getMethods());
    }
  }

}
