<?php
/**
 * @file
 * Contains \Drupal\rss_page\Element\RSSFeedList.
 */

namespace Drupal\rss_page\Element;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Attribute\FormElement;
use Drupal\Core\Url;
use Drupal\monster_menus\Element\MMCatlist;
use Drupal\monster_menus\Element\MMRepeatlist;

/**
 * Provides a form element which allows the user to manipulate a list of RSS
 * feeds.
 */
#[FormElement('rss_feed_list')]
class RSSFeedList extends MMRepeatlist {

  public function getInfo() {
    $class = static::class;
    return [
      '#input'                      => TRUE,
      '#default_value'              => [],
      '#process'                    => [[$class, 'processGroup']],
      '#pre_render'                 => [[$class, 'preRenderGroup'],
        [$class, 'preRender']],
      '#attached'                   => ['library' => ['rss_page/rss_page_js']],
      // min number of rows
      '#mm_list_min'                => 0,
      // max number of rows
      '#mm_list_max'                => 0,
      '#mm_list_reorder'            => TRUE,
      '#rss_list_types'           => [],
      '#rss_list_is_portal_page'    => FALSE,
      '#theme'                      => 'mm_catlist',
      '#theme_wrappers'             => ['form_element'],
      '#description_display'        => 'before',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function processGroup(&$element, FormStateInterface $form_state, &$complete_form) {
    $element = parent::processGroup($element, $form_state, $complete_form);
    $desc = t('<strong>:url</strong> displays the content of a standard URL-based feed. <strong>:page</strong> displays the content of a CMS page. <strong>:tag</strong> displays any CMS content labeled with the indicated tag.', [
      ':url' => t('RSS feed'),
      ':page' => t('Page feed'),
      ':tag' => t('Tag feed'),
    ]);
    $element += [
      '#title' => t('RSS feeds'),
      '#description' => $desc,
      '#mm_list_reorder' => TRUE,
    ];
    $element['#rss_list_types'] = [
      'url' => t('RSS feed'),
      'cat' => t('Page feed'),
    ];

    $element['url'] = [
      '#type' => 'container',
      '#attributes' => ['style' => 'display: none', 'id' => 'feed-url', 'class' => ['mm-list-hidden']],
      'info' => [
        '#markup' => '<div class="mm-list-info"></div>',
      ],
      'url' => [
        '#type' => 'textfield',
        '#title' => t('URL of RSS feed'),
        '#placeholder' => 'https://example.com/feed',
        '#description' => t("Enter the full URL, starting with <code>https://</code>, <code>feed://</code>, etc. The feed's title will be set automatically the first time you view the resulting page."),
        '#size' => 50,
        '#maxlength' => 255,
      ],
    ];

    $element['cat'] = [
      '#type' => 'container',
      '#attributes' => ['style' => 'display: none', 'id' => 'feed-cat', 'class' => ['mm-list-hidden']],
      'info' => [
        '#markup' => '<div class="info"><div class="description"></div></div>',
      ],
      'choose' => [
        '#type' => 'button',
        '#attributes' => ['class' => ['mm-list-button-edit']],
        '#value' => t('Choose a Page'),
      ],
    ];

    $have_tax = mm_module_exists('taxonomy') && \Drupal::config('taxonomy.settings')->get('maintain_index_table');
    if ($have_tax) {
      $element['#rss_list_types']['taxon'] = t('Tag feed');
      $element['taxon'] = [
        '#type' => 'container',
        '#title' => t('Choose a term or free tag'),
        '#attributes' => ['style' => 'display: none', 'id' => 'feed-taxon', 'class' => ['mm-list-hidden']],
        'tax-text' => [
          '#type' => 'textfield',
          '#autocomplete_route_name' => 'rss_page.taxonomy_autocomplete',
          '#placeholder' => t('Tag'),
          '#size' => 35,
          '#maxlength' => 100,
          '#description' => t('Type part of the tag, then choose a tag from the list of matches.'),
        ],
      ];
      $element['#mm_list_autocomplete_name'] = 'tax-text';
    }

    return $element;
  }

  /**
   * Add Javascript code to a page, allowing the user to manipulate a list of
   * RSS feeds.
   *
   * @param array $element
   *   The form element to display
   * @return mixed[]
   *   The modified form element
   */
  public static function preRender($element) {
    if (isset($element['#mm_list_instance'])) {
      return $element;
    }
    $mmlist_instance = MMCatlist::getInstance($element);
    $max = intval($element['#mm_list_max']);
    $min = intval($element['#mm_list_min']);

    $flags = ['isRssList' => TRUE];
    if (!$element['#rss_list_is_portal_page']) {
      $is_portal = FALSE;
    }
    else {
      $is_portal = TRUE;
      $flags['isPortal'] = TRUE;
    }
    if (!empty($element['#mm_list_reorder'])) {
      $flags['reorder'] = TRUE;
      $element['#attached']['library'][] = 'core/sortable';
    }

    $delConfirm = t("Are you sure you want to delete this feed?\n\n(You can skip this alert in the future by holding the Shift key while clicking the Delete button.)");

    $popup_base = Url::fromRoute('monster_menus.browser_load', [], ['query' => ['_path' => "1-rss-$mmlist_instance-r-r/"]])->toString();
    $popup_URL = mm_home_mmtid();
    $popup_label = t('Select a page');

    $adds = [];
    if (is_string($element['#value']) && str_starts_with($element['#value'], 'a:')) {
      foreach (unserialize($element['#value']) as $obj) {
        $url = '';
        self::getAdd($obj->type, $obj->data, $obj->name, $url, $info, $popup_URL, $is_portal);
        $adds[] = [$obj->type, $obj->name, $url, $info];
      }
    }
    else {
      foreach (_rss_page_split_feed_list($element['#value']) as $m) {
        $name = $m->name;
        $url = !$is_portal && $m->type == 'cat' ? '' : $m->data;
        self::getAdd($m->type, $m->data, $name, $url, $info, $popup_URL, $is_portal);
        $adds[] = [$m->type, $name, $url, $info];
      }
    }

    $children = [];
    $select = [
      '#type' => 'select',
      '#attributes' => ['class' => ['mm-list-rss-select']],
      '#options' => ['' => t('(choose)')],
    ];
    foreach ($element['#rss_list_types'] as $type => $name) {
      $children[$type] = $element[$type];
      unset($element[$type]);
      $select['#options'][$type] = $name;
    }

    $name = $element['#parents'][0];
    if (count($element['#parents']) > 1) {
      $name .= '[' . join('][', array_slice($element['#parents'], 1)) . ']';
    }

    $settings = [
      'isSearch'           => mm_ui_is_search(),
      'hiddenId'           => $element['#id'],
      'hiddenName'         => $name,
      'add'                => $adds,
      'autoName'           => $element['#mm_list_autocomplete_name'] ?? NULL,
      'parms'              => [
        'popupBase'         => $popup_base,
        'popupURL'          => $popup_URL,
        'popupLabel'        => $popup_label,
        'outerDivSelector'  => "div[name=mm_list_obj$mmlist_instance]",
        'rowSelector'       => "div[name=mm_list_obj$mmlist_instance] details",
        'minRows'           => $min,
        'maxRows'           => $max,
        'flags'             => $flags,
        'addCallback'       => 'rssAddCallback',
        'replaceCallback'   => 'rssReplCallback',
        'infoCallback'      => 'rssInfoCallback',
        'dataCallback'      => 'rssDataCallback',
        'updateOnChangeCallback' => 'rssUpdateOnChangeCallback',
        'autocompleteCallback' => 'rssAutocompleteCallback',
        'delConfirmMsg'     => $delConfirm,
      ],
    ];
    mm_ui_modal_dialog('init', $element);
    $element['#attached']['drupalSettings']['MM']['mmListInit'][$mmlist_instance] = $settings;
    $renderer = \Drupal::service('renderer');
    $element += [
      '#mm_list_instance' => $mmlist_instance++,
      '#mm_list_summary_tag' => $renderer->render($select) . '<span class="summary"></span>',
      '#mm_list_details_tag' => $renderer->render($children),
      '#mm_list_class' => MMCatlist::addClass($element, 'rss-page'),
    ];
    return $element;
  }

  private static function getAdd($type, $data, &$name, &$url, &$info, &$popup_URL, $is_portal) {
    if ($type == 'cat') {
      $parents = mm_content_get_parents($data);
      array_shift($parents);  // skip root
      $url = implode('/', $parents) . "/$data";
      if (!isset($popup_URL)) {
        $popup_URL = $url;
      }

      $path = [];
      foreach ($parents as $par) {
        if (!($tree = mm_content_get($par))) {
          break;
        }
        $path[] = mm_content_get_name($tree);
      }

      $path[] = $name;
      $info = implode('&nbsp;&raquo; ', $path);
    }
    elseif ($is_portal && $type == 'query') {
      $info = $name;
      $url = $data;
    }
    elseif ($type == 'taxon') {
      $info = '';
      $url = $data;
      $name = preg_replace('/ \(\d+\)$/i', '', $name);
    }
    else {  // URL
      $u = preg_replace('/\/(?!\/)/', '/&#8203;', $data);
      $info = strpos($url, str_replace('...', '', $name)) ? $u : $name . '<br />' . $u;
      $url = $data;
    }
  }

}
