<?php

namespace Drupal\rss_page\FeedDataFetcher;

use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Database\Connection;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use SimplePie\Parse\Date;
use SimplePie\SimplePie;

class FeedDataFetchUrl implements FeedDataFetcherInterface {

  /**
   * @inheritdoc
   */
  public static function fetch($feed, Connection $db, NodeInterface $node, $index, $mark, $atime, &$newest, &$new_rss_link) {
    $sp_feed = new SimplePie();
    $sp_feed->set_feed_url($feed->data);
    $sp_feed->set_cache_location(\Drupal::config('rss_page.settings')->get('cache_path'));
    $sp_feed->set_cache_duration($max_age = \Drupal::config('rss_page.settings')->get('cache_duration'));
    $sp_feed->init();
    if (!$sp_feed->error) {
      $sp_feed->handle_content_type();
    }

    if (!$sp_feed->error && $sp_feed->data) {
      $title = Html::decodeEntities($sp_feed->get_title());
      if (empty($feed->name_isset) && $title) {
        $feed->name_isset = 1;
        $db->update('rss_page_feed')
          ->fields(['name_isset' => 1, 'name' => $title])
          ->condition('name_isset', 1, '<>')
          ->condition('type', 'url')
          ->condition('data', $feed->data)
          ->execute();
      }
      if (!$title) {
        $title = $feed->name;
      }

      $feed_defaults = [
        'items' => [],
        'feed_title' => $title,
        'show_title' => TRUE,
        'feed_link' => Url::fromUri($sp_feed->get_link()),
        'fid' => $feed->fid ?? 0,
        'img' => '',
        'img_title' => '',
        'img_width' => 0,
        'img_height' => 0,
      ];
      if (_rss_page_get_node_value($node, 'show_feed_img')) {
        $feed_defaults = [
          'img' => $sp_feed->get_image_url() ?? '',
          'img_title' => Html::escape($sp_feed->get_image_title() ?? ''),
          'img_width' => (int) $sp_feed->get_image_width() ?? 0,
          'img_height' => (int) $sp_feed->get_image_height() ?? 0,
        ] + $feed_defaults;
      }
      $out = new FeedData($feed_defaults);
      $cache = ['contexts' => [], 'tags' => [], 'max-age' => $max_age];

      for ($i = 0; ; $i++) {
        $nitems = _rss_page_get_node_value($node, 'items');
        if ($nitems >= 0 && count($out->items) >= $nitems) {
          break;
        }

        if (!($item = $sp_feed->get_item($i))) {
          break;
        }

        // Take the max of the updated and published fields
        if ($updated = $item->get_item_tags(SimplePie::NAMESPACE_ATOM_10, 'updated')) {
          $parser = Date::get();
          $updated = $parser->parse($updated[0]['data']);
        }
        $date = max($item->get_date('U'), $updated);

        if ($mark == 'read' && $node->getType() == 'rss_page') {
          if ($date > $newest) {
            $newest = $date;
          }
        }
        elseif (is_null($atime) || $date > $atime) {
          $perma = html_entity_decode($item->get_permalink(), ENT_QUOTES);
          $desc = _rss_page_fix_descr($item->get_description(), $node);
          if (_rss_page_get_node_value($node, 'show_image')) {
            $enclosures = $item->get_enclosures();
            if (is_array($enclosures)) {
              $enc_html = '';
              foreach ($enclosures as $enclosure) {
                if ($enclosure->get_type()) {
                  $enc_title = Html::escape(Html::decodeEntities($item->get_title()));
                  if (stripos($enclosure->get_type(), 'image') !== FALSE) { //image file
                    $enc_html .= '<img src="' . UrlHelper::stripDangerousProtocols($enclosure->get_link()) . '" alt="' . $enc_title . '" title="' . $enc_title . '" class="image icon" />';
                  }
                  elseif (function_exists('_media_get_icon')) {
                    $icon = _media_get_icon($enclosure->get_type());
                    if (empty($icon)) {
                      $icon = 'genericdoc.gif';
                    }
                    $url = base_path() . \Drupal::service('extension.list.module')->getPath('media') . '/icons/' . $icon;
                    $enc_html .= '<a href="' . UrlHelper::stripDangerousProtocols($enclosure->get_link()) . '"><img src="' . UrlHelper::stripDangerousProtocols($url) . '" alt="' . $enc_title . '" title="' . $enc_title . '" /></a>';
                  }
                }
                if ($enc_html) {
                  $desc = "<p>$desc</p>$enc_html";
                }
              }
            }
          }
          $desc = '<div class="content">' . $desc . '</div>';

          $out->items[] = new FeedItem([
            'feed_index' => $index,
            'description' => $desc,
            'date' => $date,
            'title' => Html::decodeEntities($item->get_title()),
            'link' => $perma,
            'categories' => $item->get_category(),
            'cache' => $cache,
          ]);
        }
      }
      $new_rss_link = Url::fromUri($feed->data);
    }
    else {    // !$feed->data
      $out = new FeedData(['error' => _rss_page_rss_error($feed->data, $sp_feed), 'feed_title' => $feed->name]);
    }

    if (empty($feed->name_isset)) {
      $out->feed_title = '';
    }
    return $out;
  }

}
