<?php

/**
 * @file
 * Contains \Drupal\monster_menus\Controller\DefaultController.
 */

namespace Drupal\monster_menus\Controller;

use Drupal\Core\Controller\ControllerResolverInterface;
use Drupal\diff\Form\RevisionOverviewForm;
use Drupal\node\Form\NodeRevisionRevertForm;
use Drupal\node\Form\NodeRevisionDeleteForm;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Database;
use Drupal\Core\Session\AccountInterface;
use Drupal\diff\Controller\NodeRevisionController;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller for node revisions operations.
 */
class NodeRevisionsController extends ControllerBase {

  private ContainerInterface $container;

  private Request $currentRequest;

  private ControllerResolverInterface $controllerResolver;

  /**
   * {@inheritdoc}
   */
  public function __construct(ContainerInterface $container, Request $current_request, $controller_resolver) {
    $this->container = $container;
    $this->currentRequest = $current_request;
    $this->controllerResolver = $controller_resolver;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container,
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('controller_resolver')
    );
  }

  static public function menuAccessNodeRevisions(NodeInterface $node, AccountInterface $account) {
    if (!$node->access('update', $account) || !$node->access('view', $account) || (!$account->hasPermission('view ' . $node->getType() . ' revisions') && !$account->hasPermission('bypass node access'))) {
      return AccessResult::forbidden();
    }
    $num_versions = Database::getConnection()
      ->select('node_revision', 'r')
      ->fields('r', ['vid'])
      ->condition('r.nid', $node->id())
      ->countQuery()->execute()->fetchField();
    return AccessResult::allowedIf($num_versions > 1);
  }

  /**
   * Display an overview table of node revisions.
   *
   * @param NodeInterface $node
   *   The node to display revisions for
   * @return mixed[]
   *   The render array
   */
  public function revisionOverview(NodeInterface $node) {
    $this->currentRequest->query->remove('destination');
    if ($this->moduleHandler()->moduleExists('diff')) {
      return $this->formBuilder()->getForm(RevisionOverviewForm::class, $node);
    }
    $callable = $this->controllerResolver->getControllerFromDefinition('\Drupal\node\Controller\NodeController::revisionOverview');
    return $callable($node);
  }

  /**
   * Display a single node revision.
   *
   * @param NodeInterface $node
   *   The node to display revisions for
   * @param NodeInterface $node_revision
   *   ID of the revision to display
   * @return mixed[]
   *   The render array
   */
  public function revisionShow(NodeInterface $node, NodeInterface $node_revision) {
    return mm_node_show($node_revision);
  }

  /**
   * Display a node revision revert confirmation form.
   *
   * @param NodeInterface $node
   *   The node to revert a revision for
   * @param NodeInterface $node_revision
   *   The revision to revert to
   * @return mixed[]
   *   The render array
   */
  public function revisionRevertConfirm(NodeInterface $node, NodeInterface $node_revision) {
    return $this->formBuilder()->getForm(NodeRevisionRevertForm::class, $node_revision);
  }

  /**
   * Display a node revision deletion confirmation form.
   *
   * @param NodeInterface $node
   *   The node to delete a revision for
   * @param NodeInterface $node_revision
   *   The revision to delete
   * @return mixed[]
   *   The render array
   */
  public function revisionDeleteConfirm(NodeInterface $node, NodeInterface $node_revision) {
    return $this->formBuilder()->getForm(NodeRevisionDeleteForm::class, $node_revision);
  }

  /**
   * Returns a table which shows the differences between two node revisions.
   *
   * @param NodeInterface $node
   *   The node whose revisions are compared.
   * @param $left_revision
   *   Vid of the node revision from the left.
   * @param $right_revision
   *   Vid of the node revision from the right.
   * @param $filter
   *   If $filter == 'raw' raw text is compared (including html tags)
   *   If filter == 'raw-plain' markdown function is applied to the text before
   *   comparison.
   *
   * @return mixed[]
   *   Table showing the diff between the two node revisions.
   */
  public function compareRevisions(NodeInterface $node, $left_revision, $right_revision, $filter) {
    if ($this->moduleHandler()->moduleExists('diff')) {
      $this->testNodeRevision($node, $left_revision);
      $this->testNodeRevision($node, $right_revision);
      $controller = NodeRevisionController::create($this->container);
      return $controller->compareNodeRevisions($node, $left_revision, $right_revision, $filter);
    }

    throw new NotFoundHttpException();
  }

  /**
   * {@inheritdoc}
   */
  private function testNodeRevision(NodeInterface $node, $revision_id) {
    /** @var NodeInterface|null $revision */
    $revision = $revision_id ? static::entityTypeManager()->getStorage('node')->loadRevision($revision_id) : NULL;
    if (!$revision || $revision->id() != $node->id()) {
      throw new NotFoundHttpException();
    }
  }

}
