<?php

namespace Drupal\monster_menus\Element;

use Drupal\Component\Uuid\Uuid;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Attribute\FormElement;
use Drupal\Core\Render\Element\FormElementBase;
use Drupal\Core\Url;
use Drupal\monster_menus\Constants;
use Drupal\monster_menus\Plugin\MMTreeBrowserDisplay\Fallback;
use Drupal\monster_menus\Plugin\MMTreeBrowserDisplay\Groups;
use Drupal\monster_menus\Plugin\MMTreeBrowserDisplay\Nodes;
use Drupal\monster_menus\Plugin\MMTreeBrowserDisplay\Users;

/**
 * Provides a graphical chooser for MM Tree entries.
 */
#[FormElement('mm_catlist')]
class MMCatlist extends FormElementBase {

  static $instance;

  /**
   * {@inheritdoc}
   * @return mixed[]
   */
  public function getInfo() {
    $class = static::class;
    return [
        '#input'                      => TRUE,
        '#default_value'              => [],
        '#process'                    => [[$class, 'processGroup'],
                                          [$class, 'process'],
                                          [$class, 'processAjaxForm']],
        '#pre_render'                 => [[$class, 'preRenderGroup'],
                                          [$class, 'preRender']],
        '#attached'                   => ['library' => ['monster_menus/mm_list']],
        '#title'                      => '',
        // min number of rows
        '#mm_list_min'                => 0,
        // max number of rows
        '#mm_list_max'                => 0,
        '#mm_list_popup_start'        => '',
        // if 0, show the "[top]" entry
        '#mm_list_popup_root'         => 1,
        // in category browser, attribute of cats user can open
        '#mm_list_enabled'            => '',
        // in category browser, attribute of cats user can choose
        '#mm_list_selectable'         => Constants::MM_PERMS_APPLY,
        // let the user select rows, but not edit them
        '#mm_list_readonly'           => FALSE,
        // don't show an item's info when clicked
        '#mm_list_no_info'            => FALSE,
        // route to the popup tree browser
        '#mm_list_route'              => 'monster_menus.browser_load',
        // allow drag to reorder rows
        '#mm_list_reorder'            => FALSE,
        // auto-submit the outer form upon first choice
        '#mm_list_submit_on_add'      => FALSE,
        // automatically click the "Choose" button upon load
        '#mm_list_auto_choose'        => FALSE,
        // form element to get focus by default
        '#mm_list_initial_focus'      => '',
        '#mm_list_mode'               => Fallback::BROWSER_MODE_PAGE,
        '#mm_list_other_name'         => '',
        '#mm_list_other_callback'     => 'null',
        // set field_name and bundle_name when widget for a field
        '#mm_list_field_name'         => '',
        '#mm_list_bundle_name'        => '',
        '#mm_list_bundle_type'        => 'node',
        '#mm_list_info_func'          => [$class, 'makeEntry'],
        // a string in the format [width]x[height]
        '#mm_list_min_wh'             => '',
        // comma-separated list of types (image, video, etc.)
        '#mm_list_file_types'         => '',
        '#theme'                      => 'mm_catlist',
        '#theme_wrappers'             => ['form_element'],
        '#description_display'        => 'before',
    ];
  }

  public static function valueCallback(&$element, $input, FormStateInterface $form_state) {
  }

  public static function preRender($element) {
    static::preRenderMMList($element['#mm_list_mode'], $element,
      $element['#mm_list_popup_root'] ?? '', 'mm-catlist',
      '', '', '<span class="summary"></span>', static::defaultDetails(t('Choose a Page')));
    return $element;
  }

  /**
   * Generate the form code to allow the user to add pages/groups to a node
   *
   * @param $mode
   *   The MM_BROWSER_X constant mode
   * @param array &$elt
   *   An associative array containing the properties of the element. Properties
   *   used: #default_value, #description, #mm_list_autocomplete_name,
   *   #mm_list_route, #mm_list_enabled,
   *   #mm_list_max, #mm_list_min, #mm_list_no_info, #mm_list_other_callback,
   *   #mm_list_other_name, #mm_list_popup_root, #mm_list_readonly,
   *   #mm_list_selectable, #mm_list_info_func, #name, #parents, #required,
   *   #title, #type, #value.
   * @param $start_mmtid
   *   The location in the tree at which to default the tree view
   * @param $css_class
   *   The class to be added to the outer tag
   * @param $info_label
   *   The label appearing above the item information line
   * @param $info_const
   *   Information line common to all entries
   * @param $summary_tag
   *   HTML tag of the element within the <summary>
   * @param $details_tag
   *   HTML tag of the element within the <details>
   */
  static function preRenderMMList($mode, array &$elt, $start_mmtid, $css_class, $info_label, $info_const = '', $summary_tag = '', $details_tag = '') {
    if (isset($elt['#mm_list_instance'])) {
      return;
    }
    $mmlist_instance = static::getInstance($elt);

    // Set a default AJAX event type if none is present.
    if (isset($elt['#ajax']) && empty($elt['#ajax']['event'])) {
      $elt['#ajax']['event'] = 'change';
    }

    if (!\Drupal::currentUser()->hasPermission('use tree browser')) {
      $elt['#mm_list_readonly'] = TRUE;
    }

    $max = intval($elt['#mm_list_max']);
    $min = intval($elt['#mm_list_min']);
    $exactly1 = $max == 1 && $min == 1;

    $flags = [];
    $js_flags = [
      '#mm_list_submit_on_add' => 'submitOnAdd',
      '#mm_list_initial_focus' => 'initialFocus',
    ];
    foreach ($js_flags as $api_flag => $js_flag) {
      if (!empty($elt[$api_flag])) {
        $flags[$js_flag] = $elt[$api_flag];
      }
    }
    if (!empty($elt['#mm_list_reorder'])) {
      $flags['reorder'] = TRUE;
      $elt['#attached']['library'][] = 'core/sortable';
    }
    $flags['useEditButton'] = $flags['addOpensTreeBrowser'] = $mode != Users::BROWSER_MODE_USER;

    $info_callback = !empty($elt['#mm_list_readonly']) && $elt['#mm_list_no_info'] ? 'null' : 'catInfoCallback';
    $label_above_info = !empty($elt['#mm_list_no_info']) ? '' : $info_label;

    if ($mode == Users::BROWSER_MODE_USER) {
      $popup_base = '';
      $add_callback = 'userAddCallback';
      $replace_callback = 'userReplCallback';
    }
    else {
      $add_callback = 'catAddCallback';
      $replace_callback = 'catReplCallback';
      $start = "$start_mmtid-$mode-$mmlist_instance-";
      if ($elt['#type'] == 'mm_nodelist') {
        $field_id = !empty($elt['#mm_list_field_name']) && !empty($elt['#mm_list_bundle_name']) ? $elt['#mm_list_field_name'] . ',' . $elt['#mm_list_bundle_name'] . ',' . $elt['#mm_list_bundle_type'] : '';
        $start .= $elt['#mm_list_enabled'] . '-' . $elt['#mm_list_selectable'] . '--' . $field_id;
      }
      else {
        $start .= ($elt['#mm_list_enabled'] ?? '') . '-' . $elt['#mm_list_selectable'] . '--';
      }
      $start .= '-' . $elt['#mm_list_file_types'] . '-' . $elt['#mm_list_min_wh'] . '/';
      $popup_base = Url::fromRoute($elt['#mm_list_route'], [], ['query' => ['_path' => $start]])->toString();
      mm_ui_modal_dialog('init', $elt);
    }
    $popup_URL = $elt['#mm_list_popup_start'];

    if ($max == 1 && !$elt['#value']) {
      if ($elt['#default_value']) {
        $elt['#value'] = $elt['#default_value'];
      }
      else {
        $msgs = [
          Users::BROWSER_MODE_USER => t('(choose a user)'),
          Groups::BROWSER_MODE_GROUP => t('(choose a group)'),
          Nodes::BROWSER_MODE_NODE => t('(choose content)'),
          Fallback::BROWSER_MODE_PAGE => t('(choose a location)'),
        ];
        $elt['#value'] = [0 => $msgs[$mode] ?? $msgs[Fallback::BROWSER_MODE_PAGE]];
      }
    }

    $adds = [];
    if (!mm_ui_is_search()) {
      $url = $info = '';
      if (!empty($elt['#value']) && is_scalar($elt['#value'])) {
        // The value is sometimes corrupted, so replace it with the default.
        $elt['#value'] = $elt['#default_value'];
      }
      if (is_array($elt['#value'])) {
        foreach ($elt['#value'] as $mmtid => $name) {
          if (!$mmtid || empty($elt['#mm_list_info_func'])) {
            $url = $exactly1 ? $popup_URL : '';
            $info = '';
          }
          else {
            $elt['#mm_list_info_func']($mmtid, $name, $url, $info, $popup_URL, $elt);
          }

          $adds[] = [$name, $mmtid, $url, $info];
        }
      }
      else if (!empty($elt['#value'])) {
        foreach (self::splitMMlist($elt['#value']) as $m) {
          $name = $m[2];
          if (!empty($elt['#mm_list_info_func'])) {
            $elt['#mm_list_info_func']($m[1], $name, $url, $info, $popup_URL, $elt);
          }
          [$mmtid, $nid] = explode('/', $m[1]);

          if ($mode != Users::BROWSER_MODE_USER && !$nid && !mm_content_user_can((int) $mmtid, $elt['#mm_list_selectable'])) {
            if ($exactly1) {
              $name = t('(choose)');
              $info = '';
            }
            else {
              if ($url == $popup_URL) {
                $popup_URL = '';
              }
              continue;
            }
          }

          $adds[] = [$name, $mmtid, $url, $info];
        }
      }
    }
    else if ($exactly1) {
      $adds[] = ['', '', '', ''];
    }

    if (empty($popup_URL)) {
      $popup_URL = $start_mmtid;
    }
    $popup_label = t('Tree Browser');

    if (isset($elt['#name'])) {
      $name = $elt['#name'];
    }
    else {
      $name = $elt['#parents'][0];
      if (count($elt['#parents']) > 1) {
        $name .= '[' . join('][', array_slice($elt['#parents'], 1)) . ']';
      }
    }

    $other_name = $elt['#mm_list_other_name'] ?? '';
    $other_callback = $elt['#mm_list_other_callback'] ?? NULL;

    $auto = '';
    if (!empty($elt['#mm_list_autocomplete_name'])) {
      $auto = $elt['#mm_list_autocomplete_name'] === TRUE ? "$name-choose" : $elt['#mm_list_autocomplete_name'];
      $auto = $elt[$auto]['#name'];
    }

    $settings = [
      'hiddenId' => $elt['#id'],
      'hiddenName' => $name,
      'add' => $adds,
      'autoName' => $auto,
      'parms' => [
        'outerDivSelector' => "div[name=mm_list_obj$mmlist_instance]",
        'rowSelector' => "div[name=mm_list_obj$mmlist_instance] details",
        'isSearch' => mm_ui_is_search(),
        'minRows' => $min,
        'maxRows' => $max,
        'popupBase' => $popup_base,
        'popupURL' => $popup_URL,
        'popupLabel' => $popup_label,
        'flags' => $flags,
        'addCallback' => $add_callback,
        'replaceCallback' => $replace_callback,
        'infoCallback' => $info_callback,
        'dataCallback' => 'catDataCallback',
        'labelAboveInfo' => $label_above_info,
        'infoConst' => $info_const,
        'updateOnChangeName' => $other_name,
        'updateOnChangeCallback' => $other_callback,
        'autocompleteCallback' => 'catAutocompleteCallback',
        'autoChoose' => !empty($elt['#mm_list_auto_choose']),
      ]
    ];
    $elt['#attached']['drupalSettings']['MM']['mmListInit'][$mmlist_instance] = $settings;
    $elt += [
      '#mm_list_instance' => $mmlist_instance++,
      '#mm_list_summary_tag' => $summary_tag,
      '#mm_list_details_tag' => $details_tag,
      '#mm_list_class' => self::addClass($elt, $css_class),
    ];
  }

  // Helper function to pre-generate an entry in the list.
  public static function makeEntry($mmtid, $name, &$url, &$info, &$popup_URL, &$elt) {
    $parts = explode('/', $mmtid);
    $mmtid = $parts[0] ?: NULL;
    $node = $parts[1] ?? NULL;

    $parents = mm_content_get_parents((int) $mmtid);
    array_shift($parents);  // skip root
    if ($node || isset($parts[2])) {
      $parents[] = $mmtid;
    }

    $url = implode('/', $parents);
    if ($mmtid) {
      $url .= "/$mmtid";
    }

    $path = [];
    foreach ($parents as $par) {
      if (!($tree = mm_content_get($par))) {
        break;
      }
      $path[] = mm_content_get_name($tree);
    }

    if (!$node) $path[] = $name;
    $info = implode('&nbsp;&raquo; ', $path);

    if (isset($popup_URL)) {
      $top = explode('/', $popup_URL, 2);
      if (($found = strstr($url, "/$top[0]/")) !== FALSE) {
        $url = substr($found, 1);
      }
    }
    else {
      $popup_URL = $url;
    }
  }

  /**
   * Split the result generated by setHiddenElt in mm_list.js.
   *
   * @param array $element
   *   The form element to process.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @return mixed[]
   *   The form element.
   */
  public static function process(&$element, FormStateInterface $form_state) {
    if (is_string($element['#value'])) {
      $temp = $element['#value'];
      $element['#value'] = [];
      if (preg_match_all('#(\d+(?:/\d+)*)\{([^}]*)\}#', $temp, $matches, PREG_SET_ORDER)) {
        foreach ($matches as $match) {
          $element['#value'][$match[1]] = $match[2];
        }
      }
      $form_state->setValueForElement($element, $element['#value']);
    }
    return $element;
  }

  public static function addClass($elt, $add = '') {
    $list = $elt['#attributes']['class'] ?? [];
    $list[] = 'form-item';
    if ($add) {
      $list[] = $add;
    }
    return $list;
  }

  private static function splitMMlist($str) {
    if (is_array($str)) {
      $matches = [];
      foreach ($str as $key => $val) {
        $matches[] = [NULL, $key, $val];
      }
      return $matches;
    }
  }

  static function getInstance($element = NULL) {
    return static::$instance = 'x' . random_int(0, 1_000_000_000);
  }

  static function getCurrentInstance($element = NULL) {
    return static::$instance ?? 0;
  }

  static function getAutocompleteName(&$element) {
    assert(!empty($element['#mm_list_autocomplete_name']), '#mm_list_autocomplete_name must be set.');
    if ($element['#mm_list_autocomplete_name'] === TRUE) {
      $element['#mm_list_autocomplete_name'] = end($element['#parents']) . '-choose';
    }
    assert(!empty($element[$element['#mm_list_autocomplete_name']]), t('Based on #mm_list_autocomplete_name, child element :name must be present.', [':name' => $element['#mm_list_autocomplete_name']])->render());
    return $element['#mm_list_autocomplete_name'];
  }

  static function defaultDetails($button = NULL) {
    $details = [
      'info' => [
        '#markup' => '<div class="info"><div class="description"></div></div>',
      ],
      'choose' => [
        '#type' => 'button',
        '#attributes' => ['class' => ['mm-list-button-edit']],
        '#value' => $button ?? t('Choose'),
      ],
    ];
    return \Drupal::service('renderer')->render($details);
  }

}
