<?php
/**
 * @file
 * Contains \Drupal\monster_menus\Element\MMRepeatlist.
 */

namespace Drupal\monster_menus\Element;

use Drupal\Core\Render\Attribute\FormElement;
use Drupal\Core\Render\Element\FormElementBase;

/**
 * Provides a form element which allows the user to manipulate groups of other
 * form elements.
 */
#[FormElement('mm_repeatlist')]
class MMRepeatlist extends FormElementBase {

  /**
   * {@inheritdoc}
   * @return mixed[]
   */
  public function getInfo() {
    $class = static::class;
    return [
        '#input'                      => TRUE,
        '#default_value'              => [],
        '#process'                    => [[$class, 'processGroup']],
        '#pre_render'                 => [[$class, 'preRenderGroup'],
                                          [$class, 'preRender']],
        '#attached'                   => ['library' => ['monster_menus/mm_repeatlist']],
        // REQUIRED: name of child element containing elements to repeat
        '#mm_list_row_name'           => '',
        // REQUIRED: number of input values per row, corresponding to the number of HTML inputs
        '#mm_list_inputs_per_row'     => '',
        // min number of rows
        '#mm_list_min'                => 1,
        // max number of rows
        '#mm_list_max'                => 0,
        // text label for the 'add' button
        '#mm_list_add_button'         => $this->t('Add a Row'),
        // display the data rows as text instead of form inputs
        '#mm_list_readonly'           => FALSE,
        // allow drag to reorder rows
        '#mm_list_reorder'            => FALSE,
        '#theme'                      => 'mm_catlist',
        '#theme_wrappers'             => ['form_element'],
        '#description_display'        => 'before',
    ];
  }

  /**
   * Add Javascript code to a page, allowing the user to repeat a set of form
   * fields any number of times without having to submit the form multiple
   * times.
   *
   * To specify which data is repeatable, use '#theme_wrappers' =>
   * array('container') to construct a new container DIV and pass its ID in the
   * mm_repeatlist form element using the #mm_list_row_name parameter. In order
   * for the code to work properly, the container DIV MUST come before the
   * mm_repeatlist form element in the form array.
   *
   * Default values for the data fields can be passed using the #default_value
   * parameter. It should contain an array of arrays, one per row of data to
   * pre-populate in the form. To parse the data after it has been submitted by
   * the user, call MMRepeatlist::parseRepeatlist().
   *
   * Required fields in the form element:
   *   #mm_list_row_name           Name of child element containing repeatable
   *                               children
   *   #mm_list_inputs_per_row     Number of values per row in repeatable DIV
   *                               and #default_value
   *
   * Optional fields in the form element:
   *   #mm_list_min                Min number of rows (default: 1)
   *   #mm_list_max                Max number of rows (default: 0)
   *   #mm_list_reorder            Allow user to change the order (default: FALSE)
   *
   * Caveats:
   *   - In order for the code to work properly, the DIV container MUST come
   *     before the mm_repeatlist form element in the form array
   *   - The '#multiple' option is not supported for the 'select' type.
   *   - The 'file' and 'password' types are not supported.
   *   - If using the 'date' type, you must be sure to allocate three elements
   *     per row in the '#default_value' field, and increase the
   *     '#mm_list_inputs_per_row' accordingly.
   *
   * Example: Create a form element that allows the user to enter up to 10 rows
   * of data in two textfields per row. The form is pre-populated with some
   * default values:
   *
   *     $form['name_age'] = array(
   *        '#theme_wrappers' => array('container'),
   *        '#id' => 'name_age',
   *        '#attributes' => array('class' => array('hidden')));
   *     $form['name_age']['name'] = array(
   *       '#type' => 'textfield',
   *       '#title' => t('Your first name'),
   *       '#description' => t('What is your first name?'));
   *     $form['name_age']['age'] = array(
   *       '#type' => 'textfield',
   *       '#title' => t('Your age'),
   *       '#description' => t('What is your age?'));
   *
   *     $form['grouping'] = array(
   *       '#type' => 'details',
   *       '#open' => TRUE);
   *     $form['grouping']['data'] = array(
   *       '#type' => 'mm_repeatlist',
   *       '#title' => t('Tell us about yourself'),
   *       '#mm_list_row_name' => 'name_age',
   *       '#mm_list_inputs_per_row' => 2,
   *       '#default_value' => array(
   *           array('name1', '18'),
   *           array('name2', '26')));
   *
   * @param array $element
   *   The form element to display
   * @return mixed[]
   *   The modified form element
   */
  public static function preRender($element) {
    if (isset($element['#mm_list_instance'])) {
      return $element;
    }

    assert(!empty($element['#mm_list_row_name']) && !empty($element[$element['#mm_list_row_name']]), '#mm_list_row_name must be set and it must point to a child element.');
    assert(!empty($element['#mm_list_inputs_per_row']), '#mm_list_inputs_per_row must be set and non-zero.');
    $tag = \Drupal::service('renderer')->render($element[$element['#mm_list_row_name']]);

    $mmlist_instance = MMCatlist::getInstance($element);

    $max = intval($element['#mm_list_max']);
    $min = intval($element['#mm_list_min']);

    $flags = ['showEditButton' => TRUE];
    if ($element['#mm_list_readonly']) {
      $flags['readonly'] = TRUE;
    }
    if (!empty($element['#mm_list_reorder'])) {
      $flags['reorder'] = TRUE;
      $element['#attached']['library'][] = 'core/sortable';
    }

    if (isset($element['#name'])) {
      $name = $element['#name'];
    }
    else {
      $name = $element['#parents'][0];
      if (count($element['#parents']) > 1) {
        $name .= '[' . join('][', array_slice($element['#parents'], 1)) . ']';
      }
    }

    $del_confirm = t("Are you sure you want to delete this row?\n\n(You can skip this alert in the future by holding the Shift key while clicking the Delete button.)");

    $value = $element['#value'];
    if (!is_array($value)) {
      $value = $value[0] == '{' ? static::parseRepeatlist($value, $element['#mm_list_inputs_per_row']) : [];
    }
    if ($min) {
      $value = array_pad($value, $min, []);
    }
    $value = array_map(function ($arr) use ($element) {
      return $arr + array_fill(count($arr), $element['#mm_list_inputs_per_row'] - count($arr), '');
    }, $value);

    $settings = [
      'isSearch'         => mm_ui_is_search(),
      'hiddenName'       => $name,
      'add'              => $value,
      'autoName'         => NULL,
      'parms'            => [
        'outerDivSelector' => "div[name=mm_list_obj$mmlist_instance]",
        'rowSelector'      => "div[name=mm_list_obj$mmlist_instance] details",
        'minRows'          => $min,
        'maxRows'          => $max,
        'flags'            => $flags,
        'addCallback'      => 'listAddCallback',
        'dataCallback'     => 'listDataCallback',
        'updateOnChangeCallback' => 'listUpdateOnChangeCallback',
        'delConfirmMsg'    => $del_confirm,
      ],
    ];
    $element['#attached']['drupalSettings']['MM']['mmListInit'][$mmlist_instance] = $settings;
    $element += [
      '#mm_list_instance' => $mmlist_instance++,
      '#mm_list_summary_tag' => '<span class="summary"></span>',
      '#mm_list_details_tag' => $tag,
      '#mm_list_class' => MMCatlist::addClass($element, 'mm-repeatlist'),
    ];
    return $element;
  }

  /**
   * Parse the hidden data field generated by an mm_repeatlist form element into
   * a set of arrays.
   *
   * @param string $str
   *   Form data to parse
   * @param int $per_row
   *   The number of data elements per row (set)
   * @return mixed[]
   *   An array of arrays. Each inner array represents one set of data entered
   *   by the user.
   */
  public static function parseRepeatlist($str, $per_row) {
    if ($per_row <= 0 || !preg_match_all('/\{:(.*?):\}/', $str, $matches, PREG_PATTERN_ORDER)) {
      return [];
    }
    return array_chunk($matches[1], $per_row);
  }

}
